/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Aug 15 00:23:24 2007 by Jeff Dalton
 * Copyright: (c) 2001 - 2007, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.ip2.event.*;
import ix.icore.*;
import ix.icore.process.ProcessModelManager;

import ix.util.*;

/**
 * Controller for process and other, similar panels.
 */
public class PanelController {

    private Ip2 ip2;
    private Ip2ModelManager modelManager;

    private List listeners = new LinkedList();

    private ActivityAgenda activities;
    private Agenda issues;

    public PanelController(Ip2 ip2) {
	this.ip2 = ip2;
	this.activities = makeActivityAgenda();
	this.issues = makeIssueAgenda();
    }

    protected ActivityAgenda makeActivityAgenda() {
	return new ActivityAgenda(this);
    }

    protected IssueAgenda makeIssueAgenda() {
	return new IssueAgenda(this);
    }

    public void connectTo(ProcessModelManager modelManager) {
	this.modelManager = (Ip2ModelManager)modelManager;
	modelManager.addProcessStatusListener(activities);
	modelManager.addProcessStatusListener(issues);
    }

    public void reset() {
	activities.reset();
	issues.reset();
    }

    public void clearModel() {
	activities.clear();
	issues.clear();
    }

    public Ip2 getAgent() {
	return ip2;
    }

    public Ip2ModelManager getModelManager() {
        // /\/: Used to return only ProcessModelManager, so it might
        // make sense to remove casts from some of the calls.
	return modelManager;
    }

    public Agenda getActivityAgenda() {
	return activities;
    }

    public Agenda getIssueAgenda() {
	return issues;
    }

    public Agenda getAgenda(TaskItem issueOrActivity) {
	if (issueOrActivity instanceof Issue)
	    return issues;
	else if (issueOrActivity instanceof Activity)
	    return activities;
	else
	    throw new ConsistencyException
		("Not an issue or activity", issueOrActivity);
    }

    public void addAboutInfo(List about) {
	addAboutHandlersInfo(about, "Issue", issues.getItemHandlers());
	addAboutHandlersInfo(about, "Activity", activities.getItemHandlers());
    }

    private void addAboutHandlersInfo(List about, String type,
				      List handlers) {
	about.add("");
	if (handlers.isEmpty())
	    about.add(type + " handlers: none");
	else {
	    about.add(type + " handlers:");
	    for (Iterator i = handlers.iterator(); i.hasNext();) {
		ItemHandler h = (ItemHandler)i.next();
		about.add("   " + h + (h.isAutomatic() ? " - Automatic" : ""));
	    }
	}
    }

    /*
     * Incoming events.
     */

    public ActivityItem addActivity(Activity activity) {
	ActivityItem act;
	// /\/: Should call activities.makeItem(activity)?
	if (activity instanceof WaitForReportsActivity)
	    // /\/: irritating special case
	    activities.addItem(act = new WaitForReportsItem(activity));
	else
	    activities.addItem(act = new ActivityItem(activity));
	return act;
    }

    public IssueItem addIssue(Issue issue) {
	IssueItem iss;
	// /\/: Should call issues.makeItem(issue)?
	issues.addItem(iss = new IssueItem(issue));
	return iss;
    }

    public boolean canHandleAutomatically(TaskItem issueOrActivity) {
	// /\/: Is this method needed only because of options
	// and they way they can delay input?
	Debug.noteln("Can handle automatically?", issueOrActivity);
	return getAgenda(issueOrActivity)
	           .canHandleAutomatically(issueOrActivity);
    }

    public void handleAutomatically(TaskItem issueOrActivity) {
	Debug.noteln("Automatically handling", issueOrActivity);
	getAgenda(issueOrActivity).handleAutomatically(issueOrActivity);
    }

    public void addHandledItem(AgendaItem item) {
	getAgenda(item.getAbout()).addItem(item);
    }

    public void newReport(Report report) {
	// N.B. Must try issues first because of "note other reports"
	// activity.
	if (!(issues.acceptReport(report)
	      || activities.acceptReport(report)))
	    throw new ConsistencyException("Failed to accept " + report);
    }

    public boolean wantsReport(Report report) {
	return issues.wantsReport(report) || activities.wantsReport(report);
    }

    /*
     * Handler-related methods.
     */

    /**
     * Install the handler as a handler for issues, activities,
     * or both, as appropriate.  The decision is based on the
     * handler's class.  {@link IssueHandler}s are asked to handle
     * only issues, {@link ActivityHandler}s only activities,
     * and other handlers both.
     */
    public void addHandler(ItemHandler handler) {
	if (handler instanceof IssueHandler)
	    addIssueHandler(handler);
	else if (handler instanceof ActivityHandler)
	    addActivityHandler(handler);
	else 
	    addItemHandler(handler);
    }

    public void addActivityHandler(ItemHandler handler) {
	activities.addItemHandler(handler);
    }

    public void addIssueHandler(ItemHandler handler) {
	issues.addItemHandler(handler);
    }

    public void addItemHandler(ItemHandler handler) {
	activities.addItemHandler(handler);
	issues.addItemHandler(handler);
    }

    public ItemHandler findHandler(Predicate1 p) {
	Object h = Collect.findIf(issues.getItemHandlers(), p);
	if (h == null)
	    h = Collect.findIf(activities.getItemHandlers(), p);
	return (ItemHandler)h;
    }

    /**
     * Called by a handler when it discovers that some actions may
     * need to be deleted.  This method avoids repeating checks about
     * the most recently seen previous reason.
     */
    public void checkActionValidity(ItemHandler handler, Object reason) {
	// Since we always check all items and more than one handler
	// can ask us to check for the same reason, we ought to
	// skip repeated checks.
	if (reason != null && reason == validityCheckReason) {
	    Debug.noteln("Skipping check for", handler);
	    return;
	}
	else {
	    validityCheckReason = reason;
	}
	Debug.noteln("Controller asked to check action validity by", handler);
	Debug.noteln("For reason", reason);
	if (issues.handlers.contains(handler))
	    issues.checkActionValidity(handler, reason);
	if (activities.handlers.contains(handler))
	    activities.checkActionValidity(handler, reason);
    }

    protected Object validityCheckReason = null;

    /**
     * Called by a handler when it discovers that it may need to add
     * actions.
     */
    public void reconsiderHandler(ItemHandler handler, Object reason) {
	// Called by a handler when it thinks it may need to
	// revise some AgendaItems' lists of handler actions.
	// /\/: For now at least, this is only for additions.
	// Deletions are handled separately by checkActionValidity.
	// /\/: Somewhat awkward that it has to go via the controller,
	// but a handler may be attached to more than one agenda
	// and doesn't have a direct way to reach any of them.

	Debug.noteln("Controller asked to reconsider", handler);
	Debug.noteln("For reason", reason);

	if (issues.handlers.contains(handler))
	    issues.reconsiderHandler(handler, reason);
	if (activities.handlers.contains(handler))
	    activities.reconsiderHandler(handler, reason);
    }

    /*
     * Adding listeners to the agendas.
     */

    // /\/: Should these "add" methods exist?
    // /\/: Should the setAgendaManager method exist?

    public void addActivityListener(AgendaListener listener) {
	activities.addAgendaListener(listener);
	listener.setAgendaManager(activities);
    }

    public void addIssueListener(AgendaListener listener) {
	issues.addAgendaListener(listener);
	listener.setAgendaManager(issues);
    }

    /*
     * Methods for dealing with ControllerListeners.
     */

    public void addControllerListener(ControllerListener listener) {
	listeners.add(listener);
    }

}
