/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu May 11 02:27:54 2006 by Jeff Dalton
 * Copyright: (c) 2005, 2006, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.icore.domain.ObjectProperty;
import ix.icore.domain.ListOfObjectProperty;

// import ix.icore.plan.Plan;

import ix.util.*;
import ix.util.lisp.*;

public class ObjectView {

    protected static final Symbol TYPE = Symbol.intern("type");

    protected static final Object NO_VALUE = new UniqueObject("NO_VALUE");

    protected String name;
    protected String objectHeader;
    protected List objects;
    protected List types;
    protected ListOfObjectProperty properties;

    protected ValueParser parser = makeValueParser();

    public ObjectView() {
        super();
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getObjectHeader() {
        return objectHeader;
    }

    public String getObjectHeader(String defaultHeader) {
	return objectHeader != null ? objectHeader : defaultHeader;
    }

    public void setObjectHeader(String objectHeader) {
        this.objectHeader = objectHeader;
    }

    public List getObjects() {
        return objects;
    }

    public void setObjects(List objects) {
        this.objects = objects;
    }

    public void addObject(Object obj) {
	if (objects == null)
	    objects = new LinkedList();
	else if (objects instanceof LList)
	    objects = new LinkedList(objects); // make it add-able
	objects.add(obj);
    }

    public List getTypes() {
        return types;
    }

    public void setTypes(List types) {
        this.types = types;
    }

    public ListOfObjectProperty getProperties() {
        return properties;
    }

    public void setProperties(ListOfObjectProperty properties) {
        this.properties = properties;
    }

    public SortedSet getInitialObjects(Map worldState, Function1 lookup) {
	SortedSet result = new TreeSet(new ObjectComparator());
	if (objects != null)
	    result.addAll(objects);
	result.addAll(getNewObjects(worldState, result, lookup));
	return result;
    }

    public SortedSet getNewObjects(Map newState,
				   Set knownObjects,
				   Function1 lookup) {
	SortedSet result = new TreeSet(new ObjectComparator());
	Set rejected = new HashSet();
	for (Iterator i = newState.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    LList pattern = (LList)e.getKey();
	    if (pattern.cdr() != Lisp.NIL) {
		Object maybe = pattern.get(1);
		if ((maybe instanceof Symbol || maybe instanceof String)
		       && !knownObjects.contains(maybe)
		       && !rejected.contains(maybe)) {
		    if (isRelevantObject(maybe, lookup))
			result.add(maybe);
		    else
			rejected.add(maybe);
		}
	    }
	}
	return result;
    }

    public boolean isRelevantObject(Object obj, Function1 lookup) {
	LList pattern = Lisp.list(TYPE, obj);
	Object type = lookup.funcall(pattern);
	return types != null && types.contains(type);
    }

    // /\/: Tried addind a Plan asPlan(Map worldState) method,
    // but it's more work than it ought to be. ...


    /*
     * Value-parser
     */

    protected ValueParser makeValueParser() {
	return new ValueParser();
    }

    public Object read(ObjectProperty property, String text, Object oldValue) {
	return parser.read(property, text, oldValue);
    }

    public String write(ObjectProperty property, Object value) {
	return parser.write(property, value);
    }

    public static class ValueParser {

	public ValueParser() { }

	public Object read(ObjectProperty property,
			   String text,
			   Object oldValue) {
	    Debug.noteln("read " + property, Strings.quote(text));
	    if (text.equals(""))
		return readEmptyValue(property, oldValue);
	    ObjectProperty.Syntax syntax = property.getDefaultedSyntax();
	    if (syntax == syntax.NUMBER)
		return Util.mustBe(Number.class, Lisp.readFromString(text));
	    else if (syntax == syntax.SYMBOL)
		return Symbol.intern(text);
	    else if (syntax == syntax.STRING)
		return text;
	    else if (syntax == syntax.LIST)
		return Lisp.elementsFromString(text);
	    else if (syntax == syntax.OBJECT)
		return Lisp.readFromString(text);
	    else // e.g. syntax == DEFAULT
		return Lisp.readFromString(text);
	}

	public Object readEmptyValue(ObjectProperty property,
				     Object oldValue) {
	    ObjectProperty.Syntax syntax = property.getDefaultedSyntax();
	    if (syntax == syntax.NUMBER)
		return NO_VALUE;
	    else if (syntax == syntax.SYMBOL)
		return NO_VALUE;
	    else if (syntax == syntax.STRING)
		return oldValue == NO_VALUE ? NO_VALUE : "";
	    else if (syntax == syntax.LIST)
		return NO_VALUE;
	    else if (syntax == syntax.OBJECT)
		return NO_VALUE;
	    else // e.g. syntax == DEFAULT
		return NO_VALUE;
	}

	public String write(ObjectProperty property, Object value) {
	    Debug.noteln("write " + property, Lisp.printToString(value));
	    Debug.expect(value != null, "Null object property value");
	    if (value == NO_VALUE)
		return "";
	    ObjectProperty.Syntax syntax = property.getDefaultedSyntax();
	    if (syntax == syntax.NUMBER)
		return Lisp.printToString(value);
	    else if (syntax == syntax.SYMBOL)
		return value.toString();
	    else if (syntax == syntax.STRING)
		return value.toString();
	    else if (syntax == syntax.LIST)
		return value instanceof LList
		    ? Lisp.elementsToString((LList)value)
		    : Lisp.printToString(value);
	    else if (syntax == syntax.OBJECT)
		return Lisp.printToString(value);
	    else // e.g. syntax == DEFAULT
		return Lisp.printToString(value);
	}

    }

}
