/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Oct 21 20:08:11 2004 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.ip2;

import javax.swing.*;

import java.awt.Color;
import java.awt.Container;
import java.awt.Component;
import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.event.*;

import java.util.*;

import ix.iface.util.*;

import ix.icore.Activity;
import ix.icore.plan.Plan;

import ix.util.*;
import ix.util.xml.*;

class LoadPlanFrame extends JFrame implements ActionListener {

    Ip2 ip2;
    Plan plan;

    Container contentPane = getContentPane();

    LoadPlanFrame(Ip2 ip2, Activity act, Plan plan) {

	super(ip2.getAgentDisplayName() + " Load Plan");
	this.ip2 = ip2;
	this.plan = plan;

	setDefaultCloseOperation(DISPOSE_ON_CLOSE);

	setJMenuBar(makeMenuBar());

	Name senderId = act.getSenderId();
	String text =
	    "Received a load-plan request from " +
	    (senderId == null ? "an unnamed agent" : senderId.toString());

	JLabel label = new JLabel(text, JLabel.CENTER);
	contentPane.add(label, BorderLayout.NORTH);
	contentPane.add(makeButtonPanel(), BorderLayout.SOUTH);

	pack();
	setVisible(true);
    }

    JMenuBar makeMenuBar() {
	JMenuBar bar = new JMenuBar();
	JMenu fileMenu = new JMenu("File");
	bar.add(fileMenu);
	fileMenu.add(IFUtil.makeMenuItem("Close", this));
	JMenu viewMenu = new JMenu("View");
	bar.add(viewMenu);
	viewMenu.add(IFUtil.makeMenuItem("As XML", this));
	viewMenu.add(IFUtil.makeMenuItem("As Tree", this));
	return bar;
    }

    JPanel makeButtonPanel() {
	JPanel panel = new JPanel(); 	// defaults to FlowLayout
	// /\/: With a BoxLayout, the buttons are all different
	// widths, but with a grid, the text is all centered,
	// hence the setHorizontalAlignment in makeButton.
	panel.setLayout(new GridLayout(0, 1));
	// panel.setLayout(new BoxLayout(panel, BoxLayout.Y_AXIS));
	panel.add(makeButton("Load Plan in Current Option"));
	panel.add(makeButton("Replace Plan in Current Option"));
	panel.add(makeButton("Replace Plan in Current Option, Keeping State"));
	panel.add(makeButton("Finished"));
	return panel;
    }

    JButton makeButton(String command) {
	JButton b = new JButton(command);
	b.addActionListener(CatchingActionListener.listener(this));
	b.setHorizontalAlignment(SwingConstants.LEFT);
	return b;
    }

    public void actionPerformed(ActionEvent e) {
	String command = e.getActionCommand();
	Debug.noteln("LoadPlanFrame action:", command);
	if (command.equals("Close")) {
	    finished();
	}
	else if (command.equals("As XML")) {
	    new TextViewFrame(plan);
	}
	else if (command.equals("As Tree")) {
	    // /\/: This should be easier.  Visible by default?
	    // Constructor with object as argument?
	    XMLTreeViewFrame view = 
		new XMLTreeViewFrame(ip2, getTitle() + " Tree View");
	    view.editObject(plan);
	    view.setVisible(true);
	}
	// N.B. For now, the load / replace commands leave this
	// frame around so that the same plan can be used more
	// than once.
	else if (command.equals("Load Plan in Current Option")) {
	    loadPlan();
	}
	else if (command.equals("Replace Plan in Current Option")) {
	    replacePlan();
	}
	else if (command.equals
		 ("Replace Plan in Current Option, Keeping State")) {
	    replacePlanKeepingState();
	}
	else if (command.equals("Finished")) {
	    finished();
	}
	else
	    throw new ConsistencyException
		("Nothing to do for " + command);
    }

    void loadPlan() {
	ip2.loadPlan(plan);
    }

    void replacePlan() {
	ip2.resetViewers();	// /\/
	ip2.clearModel();
	ip2.loadPlan(plan);
    }

    void replacePlanKeepingState() {
	// /\/: What if the plan's state overrides some of the
	// state we're supposedly keeping?
	ip2.resetViewers();	// /\/
	ip2.clearAllButState();
	ip2.loadPlan(plan);
    }

    void finished() {
	dispose();
    }

    // /\/: Merge with the one in ItemEditor or in XMLTreeEditFrame.
    // Or make TextAreaFrame more convenient.

    class TextViewFrame extends TextAreaFrame {
	TextViewFrame(Object obj) {
	    super(LoadPlanFrame.this.getTitle() + " XML View",
		  new String[] {"Cancel"});
	    setText(XML.objectToXMLString(obj));
	    setVisible(true);
	}
	public void setText(String text) {
	    super.setText(text);
	    setCaretPosition(0); // scroll to beginning
	}
	public void whenClosed() {
	    frame.setVisible(false);
	    frame.dispose();
	}
    }

}

