/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Sep 11 16:17:14 2006 by Jeff Dalton
 * Copyright: (c) 2001 - 2002, 2004, 2006, AIAI, University of Edinburgh
 */

package ix.ip2;

import javax.swing.*;

import javax.swing.plaf.metal.MetalLookAndFeel;	// for colours

import java.awt.Container;
import java.awt.Component;
import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.*;

import java.util.*;

import ix.ip2.event.*;

import ix.iview.SimpleDomainEditor;
import ix.iface.util.ToolFrame;
import ix.iface.util.IconImage;
import ix.iface.util.CatchingActionListener;

import ix.icore.domain.Refinement;

import ix.icore.process.PNode;
import ix.icore.process.PNodeEnd;

import ix.icore.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.xml.*;

/**
 * A simple editor for entering and viewing individual items.
 */
public abstract class ItemEditor implements ActionListener {

    protected String title;
    protected JFrame frame;
    protected Container contentPane;

    protected JMenu fileMenu = new JMenu("File");
    protected JMenu viewMenu = new JMenu("View");

    protected AgendaViewer agendaViewer; // who we work for

    protected String itemType = "Unknown"; // will be "Issue" or "Activity"

    protected EditPanel editPanel;

    protected TextViewFrame textView;
    protected TreeViewFrame treeView;
    protected TextViewFrame statusView;

    protected XMLTranslator viewXMLTranslator = makeViewXMLTranslator();

    // /\/: The agent is an Ip2 only because of undo.
    protected Ip2 agent = (Ip2)IXAgent.getAgent();

    ItemEditor(AgendaViewer agendaViewer, String title) {
	this.agendaViewer = agendaViewer;
	this.title = title;
	setUpFrame();
    }

    protected void setUpFrame() {
	frame = new ToolFrame(title);
	frame.setIconImage(IconImage.getIconImage(this));
	frame.setSize(500, 300);
	frame.setJMenuBar(makeMenuBar());

	contentPane = frame.getContentPane();

	editPanel = makeEditPanel();
	contentPane.add(editPanel);
	frame.pack();
    }

    protected abstract EditPanel makeEditPanel();

    public void setVisible(boolean v) {
	frame.setVisible(v);
    }

    public void showItem(AgendaItem i) {
	editPanel.showItem(i);
	setVisible(true);
    }

    public void showNewItem() {
	editPanel.showNewItem();
	setVisible(true);
    }

    /**
     * Constructs the menu bar.  This method add standard items
     * to the "File" and "View" menus; subclasses may wish to add
     * other items before or after calling this method as
     * super.makeMenuBar().
     */
    protected JMenuBar makeMenuBar() {
	JMenuBar bar = new JMenuBar();

	fileMenu.add(makeMenuItem("Close"));
	bar.add(fileMenu);

	viewMenu.add(makeMenuItem("As XML"));
	viewMenu.add(makeMenuItem("As Tree"));
	viewMenu.add(makeMenuItem("Status Information"));
	bar.add(viewMenu);

	return bar;
    }

    protected JMenuItem makeMenuItem(String text) {
	JMenuItem item = new JMenuItem(text);
	item.addActionListener(CatchingActionListener.listener(this));
	return item;
    }

    /**
     * Action interpreter.  Note that the ItemEditor will be wrapped
     * in a CatchingActionListener when used as an ActionListener.
     *
     * @see ix.iface.util.CatchingActionListener
     */
    public abstract void actionPerformed(ActionEvent e);

    /**
     * Interpreter for "View" menu commands.
     */
    public boolean handleViewCommand(ActionEvent e) {
	String command = e.getActionCommand();
	if (command.equals("As XML")) {
	    AgendaItem item = editPanel.getItem();
	    if (item != null) {
		ensureTextView();
		textView.setVisible(true);
		showTextIfVisible(item);
	    }
	}
	else if (command.equals("As Tree")) {
	    AgendaItem item = editPanel.getItem();
	    if (item != null) {
		ensureTreeView();
		treeView.setVisible(true);
		showTreeIfVisible(item);
	    }
	}
	else if (command.equals("Status Information")) {
	    AgendaItem item = editPanel.getEditingItem();
	    if (item == null) {
		Util.displayAndWait(frame, new String[] {
		    "The item has not been given to the panel,",
		    "so no status information is available."});
	    }
	    else {
		ensureStatusView();
		statusView.setVisible(true);
		showStatusIfVisible(item);
	    }
	}
	else return false;	// command didn't match
	return true;		// command did match
    }

    void showViewsIfVisible(AgendaItem item) {
	showTextIfVisible(item);
	showTreeIfVisible(item);
	showStatusIfVisible(item);
    }

    void clearViews() {
	if (textView != null) textView.setText("");
	if (treeView != null) treeView.clear();
	if (statusView != null) statusView.setText("");
    }

    /*
     * XML Text View
     */

    protected void ensureTextView() {
	if (textView == null) {
	    textView = new TextViewFrame();
	}
    }

    protected void showTextIfVisible(AgendaItem item) {
	if (textView != null && textView.isVisible()) {
	    textView.setText
		(viewXMLTranslator.objectToXMLString(item.getAbout()));
	}
    }

    class TextViewFrame extends TextAreaFrame {
	TextViewFrame() {
	    super(agent.getAgentSymbolName() + " " + itemType,
		  new String[] {"Cancel"});
	}
	public void setText(String text) {
	    super.setText(text);
	    setCaretPosition(0); // scroll to beginning
	}
    }

    /*
     * XML Tree View
     */

    protected void ensureTreeView() {
	if (treeView == null) {
	    treeView = new TreeViewFrame();
	}
    }

    protected void showTreeIfVisible(AgendaItem item) {
	if (treeView != null && treeView.isVisible()) {
	    treeView.editObject(item.getAbout());
	}
    }

    class TreeViewFrame extends XMLTreeViewFrame {
	TreeViewFrame() {
	    super(ItemEditor.this.agent,
		  ItemEditor.this.agent
		      .getAgentSymbolName() + " " + itemType);
	}
	public void editObject(Object obj) {
	    // Use our own XMLTranslator
	    TreeViewFrame.this.editPanel
		.editDocument(viewXMLTranslator.objectToDocument(obj));
	}
	public void clear() {
	    reset();
	}
	protected String makeHelpText() {
	    List help = new LListCollector();
	    help.add("This view shows a tree that exactly matches the XML.");
	    help.add(""); help.add("");
	    return Strings.joinLines(help) + super.makeHelpText();
	}
    }

    /**
     * Returns an XMLTranslator for use in the XML text and tree views.
     */
    XMLTranslator makeViewXMLTranslator() {
	// Make a new XML translator like the default one.
	XMLTranslator xmlt = XML.config().makeXMLTranslator();
	// Set translator properties
	xmlt.setOmitImplAttributes(true);
	xmlt.setPrefilter(new Function1() {
	    // Maps a Variable to its value, if it has one,
	    // else to the ItemVar that is its name.
	    public Object funcall(Object arg) {
		return arg instanceof Variable
		    ? filterVariable((Variable)arg)
		    : arg;
	    }
	    Object filterVariable(Variable var) {
		Object val = var.getValue();
		return val != null ? val : var.getName();
	    }
	});
	return xmlt;
    }

    /*
     * Status View
     */

    protected void ensureStatusView() {
	if (statusView == null) {
	    statusView = new TextViewFrame();
	}
    }

    protected void showStatusIfVisible(AgendaItem item) {
	if (statusView != null && statusView.isVisible()) {
	    statusView.setText(makeStatusText(item));
	}
    }

    protected String makeStatusText(AgendaItem item) {
	List lines = new LinkedList();
	// Status of the node and its ends.
	addStatus(lines, "Status: ", item);
	if (item.getBegin() != null) addEndStatus(lines, item.getBegin());
	if (item.getEnd() != null) addEndStatus(lines, item.getEnd());
	// HandlerActions
	addHandlerActionStatus(lines, item);
	// Conditions
	addItemConditons(lines, item);
	// Join it all together.
	return Strings.joinLines(lines);
    }

    protected void addStatus(List lines, String prefix, HasStatus h) {
	lines.add(prefix + upStatus(h) + " " + h);
    }

    private String upStatus(HasStatus h) {
	return h.getStatus().toString().toUpperCase();
    }

    protected void addEndStatus(List lines, PNodeEnd ne) {
	lines.add("");
	lines.add("Status of " + ne.getEnd() + " end: " + upStatus(ne));
	List pre = ne.getPredecessors();
	if (pre.isEmpty())
	    lines.add("Has no predecessors.");
	else {
	    lines.add("Has predecessors:");
	    for (Iterator i = pre.iterator(); i.hasNext();) {
		PNodeEnd p = (PNodeEnd)i.next();
		if (p.getNode() == ne.getNode())
		    lines.add("   " + upStatus(p) + " " +
			      p.getEnd() + "_of same item");
		else
		    addStatus(lines, "   ", p);
	    }
	}
    }

    protected void addHandlerActionStatus(List lines, AgendaItem item) {
	lines.add("");
	lines.add("Handler actions:");
	for (Iterator ai = item.getActions().iterator(); ai.hasNext();) {
	    HandlerAction act = (HandlerAction)ai.next();
	    if (act.isReady())
		lines.add(act.toString() + " ready");
	    else {
		lines.add(act.toString() + " not ready");
		ActionUnreadyReason reason = act.getUnreadyReason();
		if (reason == null)
		    lines.add("   Reason unknown");
		else {
		    String[] explanation = reason.getExplanation();
		    for (int i = 0; i < explanation.length; i++) {
			lines.add("   " + explanation[i]);
		    }
		}
	    }
	}
    }

    protected void addItemConditons(List lines, AgendaItem item) {
	lines.add("");
	List conds = item.getModelManager().getNodeConditions(item);
	if (conds == null || conds.isEmpty())
	    lines.add("No conditions");
	else {
	    lines.add("Conditions:");
	    for (Iterator i = conds.iterator(); i.hasNext();) {
		lines.add(i.next().toString());
	    }
	}
    }

    /**
     * Item-editing panel
     */
    protected abstract class EditPanel extends JPanel
	implements ActionListener, AgendaItemListener {

	int textCols = 50;
	int commentRows = 6;

	JTextArea itemText = new ItemTextArea(3, textCols);
	JTextArea commentText = new JTextArea(commentRows, textCols);

	// Buttons that can be disabled or have their text changed.
	JButton modifyButton;

	Box newItemButtons = Box.createHorizontalBox();
	Box editItemButtons = Box.createHorizontalBox();

	AgendaItem editingItem = null;

	EditPanel() {
	    super();
	    setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
	    // setUp(); // done in leaf subclass constructors
	}

	protected abstract void setUp();

	protected Box makeLeftLabel(String text) {
	    return makeLeftLabel(new JLabel(text));
	}

	protected Box makeLeftLabel(JLabel label) {
	    Box b = Box.createHorizontalBox();
	    b.add(label);
	    b.add(Box.createHorizontalGlue());
	    return b;
	}

	protected JButton makeButton(String text) {
	    JButton button = new JButton(text);
	    button.addActionListener(CatchingActionListener.listener(this));
	    return button;
	}

	/**
	 * Returns the item being edited / viewed in the panel,
	 * if any.  If the user is creating a new item, the result
	 * will be null;
	 */
	AgendaItem getEditingItem() {
	    return editingItem;
	}

	/**
	 * Records an existing item that is being viewed in the panel.
	 * Always call this method rather than directly assigning
	 * to the field.
	 */
	void setEditingItem(AgendaItem i) {
	    editingItem = i;
	}

	/**
	 * Sets up to edit an existing item.
	 */
	void showItem(AgendaItem i) {

	    setEditingItem(i);
	    i.addItemListener(this);

	    itemText.setText(PatternParser.unparse(i.getPattern()));
	    itemText.setEditable(false); // was setEnabled(false);

	    commentText.setText(i.getComments());
	    commentText.setEnabled(true);

	    ensureButtons(editItemButtons);

	    Status status = i.getStatus();
	    if (status == Status.BLANK || status == Status.POSSIBLE) {
		modifyButton.setEnabled(true);
	    }
	    else {
		// /\/: Instead allow modification of e.g. comments?
		//      Yes.
		// modifyButton.setEnabled(false);
		// commentText.setEnabled(false);
	    }

	    frame.validate();

	    showViewsIfVisible(i);

	}

	/**
	 * Sets up to edit a new, initially empty item
	 */
	void showNewItem() {

	    setEditingItem(null);

	    itemText.setText("");
	    itemText.setEditable(true);	// was setEnabled(true);

	    commentText.setText("");
	    commentText.setEnabled(true);

	    ensureButtons(newItemButtons);
	    frame.validate();

	    clearViews();

	}

	void ensureButtons(Box buttons) {
	    Component lastComponent = getComponent(getComponentCount() - 1);
	    Debug.expect(lastComponent == newItemButtons
			 || lastComponent == editItemButtons);
	    if (lastComponent != buttons) {
		remove(lastComponent);
		add(buttons);
		repaint();
	    }
	}


	/**
	 * Action interpreter.  Note that this object will be wrapped in
	 * a CatchingActionListener when used as an ActionListener.
	 *
	 * @see ix.iface.util.CatchingActionListener
	 */
	public abstract void actionPerformed(ActionEvent e);


	/**
	 * Tells the item viewer to add a new item
	 */
	void addItem() {
	    Debug.expect(editingItem == null);
	    try {
		AgendaItem i = getItem();
		if (i != null) {
		    agendaViewer.addItem(i);
		    showItem(i);	// now editing that item
		}
	    }
	    catch (Exception e) {
		Debug.displayException("Cannot add the item", e);
	    }
	}

	/**
	 * Creates an issue or activity from the current state of the
	 * panel.
	 */
	AgendaItem getItem() {
	    if (editingItem != null) {
		// /\/: For now we can just return it, because it will
		// only be viewed, not modified.  This case is only for
		// the XML text and tree views.
		return editingItem;
	    }
	    String text = itemText.getText().trim();
	    String comments = commentText.getText();
	    if (text.length() == 0) {
		JOptionPane.showMessageDialog(frame,
	            "Empty item text",
		    "Invalid item",
		    JOptionPane.ERROR_MESSAGE);
		return null;
	    }
	    LList pattern = PatternParser.parse(text);
	    if (pattern.length() < 1)
		throw new RuntimeException("Empty item pattern");
	    AgendaItem i = agendaViewer.makeItem(pattern);
	    if (!comments.trim().equals(""))
		i.setComments(comments);
	    return i;
	}

	/**
	 * Modifies a existing item.
	 */
	void modifyItem() {
	    Debug.expect(editingItem != null);
            // /\/: Maybe this should be an undoable transaction
            // since we might eventually(?) modify the item in
            // other ways?
	    editingItem.setComments(commentText.getText());
	}

	/**
	 * Starts editing a new item based on an existing one.
	 */
	void copyItem() {
	    Debug.noteln("Copying item");
	    Debug.expect(editingItem != null, "no item being edited");
	    Debug.expect(editingItem.getAbout() != null,
			 "no edited item about");
	    // Go back from any Variables to ?-vars.
	    LList pat = (LList)Variable
		.revertVars(editingItem.getAbout().getPattern());
	    // Set the pattern text.
	    // /\/: In Java 1.3, when we have a syntax combo box
	    // and init it to show "" when copying an activity,
	    // there's an event as if the user had selected "",
	    // and that clears itemText.  So we need to put
	    // the text back in even if it should already be
	    // right (because there were no Variables).
	    itemText.setText(PatternParser.unparse(pat));
	    setEditingItem(null);
	    itemText.setEditable(true);	// was setEnabled(true);
	    commentText.setEnabled(true);
	    ensureButtons(newItemButtons); // can't edit subactions
	    frame.validate();
	}

	/*
	 * AgendaItemListener methods - in case an Item changes while
	 * we're editing it.
	 */

	public void statusChanged(AgendaItemEvent e) {
	    AgendaItem i = (AgendaItem)e.getSource();
	    if (i != editingItem)
		return;
	    Debug.noteln("Status change while editing item", i);

	    // Since it only gets more restrictive, and always in the
	    // same way, we only have to check that we're not still in
	    // the most permissive state.
	    Status status = i.getStatus();
	    if (status != Status.BLANK && status != Status.POSSIBLE) {
		// modifyButton.setEnabled(false);
	    }
	    
	}

	public void priorityChanged(AgendaItemEvent e) {
	    AgendaItem i = (AgendaItem)e.getSource();
	    if (i != editingItem)
		return;
	    Debug.noteln("Priority change while editing item", i);
	}

	public void handlerActionsChanged(AgendaItemEvent e) {
	    AgendaItem i = (AgendaItem)e.getSource();
	    if (i != editingItem)
		return;
	    Debug.noteln("Handler actions changed while editing item", i);
	}

	public void newHandlerAction(AgendaItemEvent e, HandlerAction act) {
	    AgendaItem i = (AgendaItem)e.getSource();
	    if (i != editingItem)
		return;
	    Debug.noteln("New handler action while editing item", i);
	}

	public void newReport(AgendaItemEvent e, Report report) {
	    AgendaItem i = (AgendaItem)e.getSource();
	    if (i != editingItem)
		return;
	    Debug.noteln("New report while editing item", i);
            takeChangedComments();
	}

	public void agendaItemEdited(AgendaItemEvent e) {
	    AgendaItem i = (AgendaItem)e.getSource();
	    if (i != editingItem)
		return;
	    // Probably we did the editing ...
	    Debug.noteln("Item edited while editing", i);
            takeChangedComments();
	}

        private void takeChangedComments() {
            String text = editingItem.getComments();
            commentText.setText(text);
            // Try to get it to scoll to the end
	    commentText.setCaretPosition(text.length());
	    frame.validate();
        }

    }

    /**
     * A text area that looks disabled when it's no editable.
     * It is used to display the item's pattern text.
     */
    protected static class ItemTextArea extends JTextArea {
	ItemTextArea(int rows, int cols) {
	    super(rows, cols);
	}
	public void setEditable(boolean v) {
	    super.setEditable(v);
	    setForeground(v ? MetalLookAndFeel.getBlack()
			    : MetalLookAndFeel.getControlDisabled());
	}
    }

}

// Issues:
// * The itemText is editable for a new item and not when viewing
//   an item that's already in the panel, but it used to be enabled
//   and disabled instead.  This was changed so that the text could
//   be mousable in both states, but then it no longer looked
//   disabled and there was no visual cue that it was not editable.
//   Hence the ItemTextArea subclass.  But is that the right answer?
