/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Sep 13 21:03:35 2007 by Jeff Dalton
 * Copyright: (c) 2004 - 2007, AIAI, University of Edinburgh
 */

package ix.ip2;

import javax.swing.*;

import java.awt.event.*;

import java.io.*;
import java.net.URL;
import java.util.*;

import org.jdom.Namespace;
import org.jdom.output.XMLOutputter;

// Imports needed by the xmlSyntaxClasses method
// and for some file syntaxes.
import ix.icore.*;
import ix.icore.domain.*;
import ix.icore.plan.*;
import ix.ichat.ChatMessage;

// More file syntax imports.
import ix.iface.domain.LTF_Parser;
import ix.iface.domain.LTF_Writer;
import ix.iface.domain.TF_Writer;
import ix.iface.domain.ChecklistParser;
import ix.iface.plan.InitLoader;
import ix.iface.plan.InitSaver;
import ix.iface.plan.HtmlPlanWriter;
import ix.iface.plan.TextPlanWriter;
import ix.icore.plan.PlanAsDomain;
import ix.icore.plan.DomainAsPlan;

import ix.iface.util.CatchingActionListener;

import ix.iscript.*;

import ix.util.*;
import ix.util.xml.*;
import ix.util.lisp.*;
import ix.util.reflect.*;

/**
 * XML tool configuration for I-P2 and similar systems.
 */
public class Ip2XMLConfig extends XMLConfig {

    /* * * Constructors * * */

    public Ip2XMLConfig() {
	super();
	setHomeNamespaceURI("http://www.aiai.ed.ac.uk/project/ix/");
    }

    /* * * Factories * * */

    public ClassFinder makeClassFinder() {
	return new DashSyntaxClassFinder();
    }

    public XMLTranslator makeXMLTranslator(ClassSyntax syntax) {
	XMLTranslator xmlt = new XMLTranslator(syntax);
	xmlt.setOmitImplAttributes(true);
	return xmlt;
    }

    public FileSyntaxManager makeFileSyntaxManager() {
	FileSyntaxManager fsm = new FileSyntaxManager();
	fsm.addSyntax(new LTFFiles());
	fsm.addSyntax(new TFFiles());
	fsm.addSyntax(new ChecklistFiles());
	fsm.addSyntax(new InitFiles());
	fsm.addSyntax(new LPADFiles());	// plans as domains
//  	fsm.addSyntax(new HtmlFiles());
	fsm.addSyntax(new TextFiles());
	fsm.addSyntax(new OwlSFiles());
	return fsm;
    }

    /* * * Default instances * * */

    protected void makeDefaultInstances() {
	super.makeDefaultInstances();
	addInitialImports();
    }

    /* * * ClassFinder imports * * */

    /**
     * Adds an initial set of imports to the default ClassFinder.
     */
    protected void addInitialImports() {
	ClassFinder cf = defaultClassFinder;
	cf.addImport("ix.icore.*");
	cf.addImport("ix.icore.domain.*");
	cf.addImport("ix.icore.plan.*");
	cf.addImport("ix.icore.log.*");
	cf.addImport("ix.ichat.ChatMessage");
	cf.addImport("ix.ip2.test.*");
	cf.addImport("ix.ip2.log.*");
	cf.addImport("ix.ip2.ObjectView");
	cf.addImport("ix.ip2.ObjectViewProperty");
	cf.addImport("ix.ip2.ObjectViewProperty$Syntax");
	cf.addImport("ix.iplan.PlanTest");
	cf.addImport("ix.iplan.PlanTestDefaults");
	cf.addImport("ix.iplan.PlanTestGroup");
	cf.addImport("ix.iplan.ExpandPlanTestGroup");
	cf.addImport("ix.util.xml.LiteralDocument");
	cf.addImport("ix.util.Name");
	cf.addImport("ix.util.Duration");
	cf.addImport("ix.util.lisp.*");
	cf.addImport("ix.iscript.*");
	cf.addImport("ix.test.xml.*");	// for experiments and testing
	cf.addImport("java.awt.Color");

	// /\/: Add this for now, until we work out a general
	// mechansim for classes that may not always be known.
	cf.addImport("ix.iview.igraph.NodePosition");
    }

    /* * * File syntaxes * * */

    /** LTF file syntax. */
    public static class LTFFiles extends AbstractFileSyntax {
	private List types = Lisp.list("lsp", "ltf");
	public LTFFiles() { }
	public List getFileTypes() {
	    return types;
	}
	public String getFileTypeDescription() {
	    return "List-TF files";
	}
	public boolean canRead() {
	    return true;
	}
	public boolean canWrite() {
	    return true;
	}
	public List readableClasses() {
	    return Collections.singletonList(Domain.class);
	}
	public List writableClasses() {
	    return Collections.singletonList(Domain.class);
	}
	public Object readObject(URL url) throws IOException {
	    return new LTF_Parser(url).readDomain();
	}
	public void writeObject(Object obj, File file) throws IOException {
	    new LTF_Writer(file).writeDomain((Domain)obj);
	}
    }

    /** TF file syntax. */
    public static class TFFiles extends AbstractFileSyntax {
	private List types = Lisp.list("tf");
	public TFFiles() { }
	public List getFileTypes() {
	    return types;
	}
	public String getFileTypeDescription() {
	    return "O-Plan TF files";
	}
	public boolean canWrite() {
	    return true;
	}
	public List writableClasses() {
	    return Collections.singletonList(Domain.class);
	}
	public void writeObject(Object obj, File file) throws IOException {
	    new TF_Writer(file).writeDomain((Domain)obj);
	}
    }

    /** Checklist file syntax. */
    public static class ChecklistFiles extends AbstractFileSyntax {
	private List types = Lisp.list("checklist");
	public ChecklistFiles() { }
	public List getFileTypes() {
	    return types;
	}
	public String getFileTypeDescription() {
	    return "Checklist files as domains";
	}
	public boolean canRead() {
	    return true;
	}
	public List readableClasses() {
	    return Collections.singletonList(Domain.class);
	}
	public Object readObject(URL url) throws IOException {
	    return new ChecklistParser(url).readDomain();
	}
    }

    /** ".init" (world state) file syntax. */
    public static class InitFiles extends AbstractFileSyntax {
	private List types = Lisp.list("init");
	public InitFiles() { }
	public List getFileTypes() {
	    return types;
	}
	public String getFileTypeDescription() {
	    return "Initial world-state files";
	}
	public boolean canRead() {
	    return true;
	}
	public boolean canWrite() {
	    return true;
	}
	public List readableClasses() {
	    return Collections.singletonList(Plan.class);
	}
	public List writableClasses() {
	    return Collections.singletonList(Plan.class);
	}
	public Object readObject(URL url) throws IOException {
	    return new InitLoader().readInit(url);
	}
	public void writeObject(Object obj, File file) throws IOException {
	    new InitSaver().savePlan((Plan)obj, file);
	}
    }

    /** File syntax for plans represented as domains. */
    public static class LPADFiles extends AbstractFileSyntax {
	private List types = Lisp.list("lpad");
	public LPADFiles() { }
	public List getFileTypes() {
	    return types;
	}
	public String getFileTypeDescription() {
	    return "Plans represented as LTF domains";
	}
	public boolean canRead() {
	    return true;
	}
	public boolean canWrite() {
	    return true;
	}
	public List readableClasses() {
	    return Collections.singletonList(Plan.class);
	}
	public List writableClasses() {
	    return Collections.singletonList(Plan.class);
	}
	public Object readObject(URL url) throws IOException {
	    // Read an LTF domain
	    Domain d = new LTF_Parser(url).readDomain();
	    // Convert it to a plan
	    return new DomainAsPlan(d);
	}
	public void writeObject(Object obj, File file) throws IOException {
	    // Convert the plan to a domain
	    Domain d = new PlanAsDomain((Plan)obj);
	    // Write the domain as LTF
	    new LTF_Writer(file).writeDomain(d);
	}
    }

    /** HTML file syntax for plan output. */
    public static class HtmlFiles extends AbstractFileSyntax {
	private List types = Lisp.list("html");
	public HtmlFiles() { }
	public List getFileTypes() {
	    return types;
	}
	public String getFileTypeDescription() {
	    return "HTML descriptions of plans";
	}
	public boolean canWrite() {
	    return true;
	}
	public List writableClasses() {
	    return Collections.singletonList(Plan.class);
	}
	public void writeObject(Object obj, File file) throws IOException {
	    new HtmlPlanWriter(file).writePlan((Plan)obj);
	}
    }

    /** Text file syntax for plan output. */
    public static class TextFiles extends AbstractFileSyntax {
	private List types = Lisp.list("txt");
	public TextFiles() { }
	public List getFileTypes() {
	    return types;
	}
	public String getFileTypeDescription() {
	    return "Text descriptions of plans";
	}
	public boolean canWrite() {
	    return true;
	}
	public List writableClasses() {
	    return Collections.singletonList(Plan.class);
	}
	public void writeObject(Object obj, File file) throws IOException {
	    new TextPlanWriter(file).writePlan((Plan)obj);
	}
    }

    /** OWL-S file syntax. */
    public static class OwlSFiles extends AbstractFileSyntax {
	private List types = Lisp.list("owls");
	public OwlSFiles() { }
	public List getFileTypes() {
	    return types;
	}
	public String getFileTypeDescription() {
	    return "OWL-S files";
	}
	public boolean isAvailable() {
	    return existsClass("ix.util.owls.OwlSTranslator")
		&& existsClass("org.mindswap.owls.service.Service");
	}
	public boolean canRead() {
	    return true;
	}
	public boolean canWrite() {
	    return false;	// for now /\/
	}
	public List readableClasses() {
	    return Collections.singletonList(Domain.class);
	}
	public List writableClasses() {
	    // /\/: Can't yet do it from a Plan object.
	    return Collections.singletonList(Plan.class);
	}
	public Object readObject(URL url) throws IOException {
	    Loader loader = (Loader)getTranslator();
	    return loader.readObject(url);
	}
	public void writeObject(Object obj, File file) throws IOException {
	    Saver saver = (Saver)getTranslator();
	    saver.writeObject(obj, file);
	}
	protected Object getTranslator() {
	    return makeInstance("ix.util.owls.OwlSTranslator");
	}
    }

    /* * * Tree editor * * */

    public Class[] treeEditorTemplateClassRoots() {
	return new Class[] {
	    ix.icore.Activity.class,
	    ix.icore.Issue.class,
	    ix.icore.Report.class,
	    ix.ichat.ChatMessage.class,
	    ix.icore.domain.Domain.class,
	    ix.icore.domain.PatternAssignment.class,
	    ix.icore.plan.Plan.class,
	    ix.ip2.test.TestItem.class
	};
    }

    public JMenu makeTreeEditorLookAtMenu(XMLTreeEditFrame frame) {
	JMenu lookMenu = new JMenu("Look At");
	lookMenu.add(new LookItem(frame, "Domain") {
	    Object getObject() {
		return ip2.getDomain();
	    }
	});
	lookMenu.add(new LookItem(frame, "Plan") {
	    Object getObject() {
		return ip2.getPlan();
	    }
	});
	return lookMenu;
    }

    abstract class LookItem extends JMenuItem implements ActionListener {
	XMLTreeEditFrame frame;
	Ip2 ip2;
	LookItem(XMLTreeEditFrame frame, String text) {
	    super(text);
	    this.frame = frame;
	    this.ip2 = (Ip2)frame.getAgent();
	    addActionListener(CatchingActionListener.listener(this));
	}
	public void actionPerformed(ActionEvent e) {
	    Debug.noteln("LookItem command", e.getActionCommand());
	    frame.editObject(getObject());
	}
	abstract Object getObject();
    }

    /* * * XMLSyntax * * */

    // Note that we keep the I-Script classes separate form the rest.
    // Really, they ought to be in a separate namespace.  /\/

    public List xmlSyntaxClasses(ClassSyntax classSyntax, Class rootClass) {
	// /\/: Here's where we try to make up for the limitations
	// of the methods used in the ClassSyntax class.
	if (rootClass == Object.class)
	    return ixClasses(classSyntax);
	else if (Plan.class.isAssignableFrom(rootClass))
	    return planClasses(classSyntax, rootClass);
	else if ("ix.iscript".equals(rootClass.getPackage().getName()))
	    return classSyntax
		      .relevantClasses(Arrays.asList(ixIScriptClasses));
	else
	    return classSyntax.relevantClasses(rootClass);
    }

    protected List planClasses(ClassSyntax classSyntax, Class planClass) {
	List classes = classSyntax.relevantClasses(new Class[] {
	    planClass,
	    PlanVariable.class,
	    Constraint.class,
	    Ordering.class,
	    LiteralDocument.class
	});
	return (List)Collect.union(classes, Arrays.asList(atomicClasses));
    }

    protected List ixClasses(ClassSyntax classSyntax) {
	Class[] roots = (Class[])Util.appendArrays(new Object[] {
	    basicIXClasses,
	    atomicClasses,
	    ixDomainClasses,
	    ixPlanClasses
	    // ixIScriptClasses
	});
	return classSyntax.expandRelevantClasses(Arrays.asList(roots));
    }

    protected final Class[] ixDomainClasses = {
	Domain.class
	// ix.iview.igraph.NodePosition.class
    };

    protected final Class[] ixPlanClasses = {
	Plan.class, PlanVariable.class
    };

    protected final Class[] basicIXClasses = {
	Issue.class, Activity.class,
	Constraint.class, Ordering.class, PatternAssignment.class,
	ItemVar.class,
	Report.class,
	ChatMessage.class,
	LiteralDocument.class
    };

    protected final Class[] atomicClasses = {
	Symbol.class, String.class,
	Integer.class, Long.class, Float.class, Double.class
    };

    protected final Class[] ixIScriptClasses = {
	And.class,
	Assignment.class,
	Binding.class,
	Call.class,
	Expression.class,
	IScriptSource.class,
	IScriptXmlSource.class,
	IScriptLispSource.class,
	If.class,
	Lambda.class,
	Let.class,
	Literal.class,
	Or.class,
	Sequence.class,
	VarRef.class,
	While.class
    };

}
