/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sun Feb 24 13:55:07 2008 by Jeff Dalton
 * Copyright: (c) 2001 - 2007, AIAI, University of Edinburgh
 */

package ix.ip2;

import javax.swing.*;		// but shouldn't act as part of UI /\/

import java.util.*;

import ix.icore.*;
import ix.icore.process.*;
import ix.icore.plan.*;
import ix.icore.domain.*;
import ix.icore.log.BindingEvent;

import ix.iplan.TimePointNet;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;
import ix.util.context.*;

public class Ip2ModelManager extends AbstractPMM {

    public static final Symbol
	S_WORLD_STATE = Symbol.intern("world-state"),
	S_CONDITION   = Symbol.intern("condition"),
	S_EFFECT      = Symbol.intern("effect");

    protected static final Symbol
	S_COMPUTE_SUPPORT_CODE = Symbol.intern("compute-support-code");

    protected Ip2 ip2;

    protected LLQueue<ActivityItem> nodes = new LLQueue<ActivityItem>();

    protected ContextHashMap varEnv = new ContextHashMap(); // name -> Variable

    protected LLQueue sentinels = new LLQueue();

    protected ConstraintAssociator constraintAssociator =
	new ConstraintAssociator(this);

    protected VariableManager variableManager = makeVariableManager();

    protected Ip2WorldStateManager worldStateCM = makeWorldStateManager();

    // /\/: S.b. able to have declared class just be ConstraintManager.
    protected TimePointNet tpnm = makeTPNManager();

    protected AdviceManager adviceManager = makeAdviceManager();

    protected LLQueue otherConstraints = new LLQueue();
	// /\/ was a LinkedListOfConstrainer

    protected ContextMultiMap nodeFilterConstraints =
	new ContextMultiHashMap() {
	    public Collection makeValueCollection(Object firstItem) {
		Collection c = new ContextListOfConstraint();
		c.add(firstItem);
		return c;
	    }
	};

    protected ContextMultiMap nodeTimeConstraints =
	new ContextMultiHashMap();

    protected ContextMultiMap otherNodeConstraints =
	new ContextMultiHashMap();

    // We keep a record of the most recently loaded code so that
    // it won't be loaded again and again.  /\/
    protected Object computeSupportCode = null;

    public Ip2ModelManager(Ip2 ip2) {
	super();
	this.ip2 = ip2;
	Context c = Context.getContext();
	if (c != Context.rootContext && c.getParent() != Context.rootContext) {
	    Context.printContextTree();
	    throw new ConsistencyException
		("initial context isn't root or root-child", c);
	}
	Context.pushContext();	// protect root context
	// Add plug-in constraint managers.
	addConstraintManager(new Scrum());
	addConstraintManager(new UseCM(this));
    }

    public void reset() {
	nodes.clearCompletely();
	varEnv.clearCompletely();
	sentinels.clearCompletely();
	variableManager.reset();
	worldStateCM.reset();
	tpnm.reset();
	adviceManager.reset();
	otherConstraints.clearCompletely();
	nodeFilterConstraints.clearCompletely();
	nodeTimeConstraints.clearCompletely();
	otherNodeConstraints.clearCompletely();
	// /\/: Annotations might be handled directly, rather than inherited
	// from AbstractAnnotatedObject.
	clearAnnotationsCompletely();

	Context.clearContexts();
	Context.pushContext();	// protect root context
    }

    public void clear() {
	nodes.clear();
	varEnv.clear();
	sentinels.clear();
	constraintAssociator.clear();
	variableManager.clear();
	worldStateCM.clear();
	tpnm.clear();
	// /\/ Clear advice CM?
	otherConstraints.clear();
	nodeFilterConstraints.clear();
	nodeTimeConstraints.clear();
	otherNodeConstraints.clear();
	clearAnnotations();
    }

    public Ip2 getIp2() {
	return ip2;
    }

    public VariableManager getVariableManager() {
	return variableManager;
    }

    protected VariableManager makeVariableManager() {
	return new VariableManager(this);
    }

    protected Ip2WorldStateManager makeWorldStateManager() {
	return new Ip2WorldStateManager(this);
    }

    protected TimePointNet makeTPNManager() {
	return new TimePointNet();
    }

    public TimePointNet getTPNManager() {
	return tpnm;
    }

    protected AdviceManager makeAdviceManager() {
	return new AdviceManager(this);
    }

    public AdviceManager getAdviceManager() {
	return adviceManager;
    }

    protected ComputeInterpreter makeComputeInterpreter() {
	return new LispComputeInterpreter(this);
    }

    public ComputeInterpreter getComputeInterpreter() {
	return getVariableManager().getComputeInterpreter();
    }

    public void addConstraintManager(ConstraintManager cm) {
	cm.registerWith(constraintAssociator);
    }

    public List<ConstraintManager> getConstraintManagers(Constraint c) {
	return constraintAssociator.getConstraintManagers(c);
    }

    public void addVariable(Variable v) {
	Debug.expect(varEnv.get(v.getName()) == null,
		     "Variable already exists");
	varEnv.put(v.getName(), v);
    }

    public Variable getVariable(Object name) {
	return (Variable)varEnv.get(name);
    }

    public Map getVarEnv() {	// for ix.ip2.PlanMaker /\/
	return varEnv;
    }

    public void addNode(PNode node) {
	// This is called by the agenda's addItem method.
	Debug.noteln("PMM adding node", node);
	node.modelManager = this;
	nodes.add((ActivityItem)node);
	// Put a trivial time constraint between the ends of the node.
	tpnm.addTimeConstraintElseFail(node.getBegin(), node.getEnd());
	// /\/: Eval any new compute support code.  This is an odd
	// time, but we don't want to check every time we evaluate
	// a compute condition, and doing it here will usually work.
	loadAnyNewComputeSupportCode();
    }

    public void removeNode(PNode node) {
	if (node.isExpanded())
	    throw new IllegalArgumentException
		("Attempt to delete a node that has been expanded " + node);
	Debug.noteln("PMM removing node", node);
	node.unlink();
	nodes.remove(node);
    }

    public void addNodesBefore(PNode at, List addList) {
	// This lets us add some nodes before a given node
	// rather than at the end.  "at" should occur only
	// once.  In any case, we add before the first occurrence.
	Debug.noteln("PMM adding nodes", addList);
	Debug.noteln("before", at);
	for (Iterator i = addList.iterator(); i.hasNext();) {
	    PNode n = (PNode)i.next();
 	    n.modelManager = this;
	    // /\/: we can't do this now because this method is used
	    // to add goal nodes in Slip, and we don't want to have to
	    // delete the temporal constraints, because we don't get
	    // have a good way to do it.
	    // tpnm.addTimeConstraintElseFail(n.getBegin(), n.getEnd());
	}
	LList oldNodes = nodes.contents();
	LList newNodes = 
	    (LList)Collect.insertBeforeFirst(at, oldNodes, addList);
	nodes.setContents(newNodes);
    }

    public List<ActivityItem> getNodes() {
	// Returns a snapshot of the current state
	return nodes.contents();
    }

    public void walkNodes(Proc p) {
	for (Iterator i = getNodes().iterator(); i.hasNext();) {
	    ActivityItem item = (ActivityItem)i.next();
	    p.call(item);
	}
    }

    // walkTopNodes and walkNodeChildren can be used together to
    // recursively visit the nodes in various orders.  By not building
    // in the recursion, we let the walker decide whether to visit
    // the children and whether to visit them before or after
    // their parent.

    public void walkTopNodes(Proc walker) {
	for (Iterator i = getNodes().iterator(); i.hasNext();) {
	    ActivityItem item = (ActivityItem)i.next();
	    if (item.getParent() == null)
		walker.call(item);
	}
    }

    public void walkNodeChildren(ActivityItem item, Proc walker) {
	// Recursion utility
	for (Iterator i = item.getChildren().iterator(); i.hasNext();) {
	    ActivityItem child = (ActivityItem)i.next();
	    walker.call(child);
	}
    }

    /* ---- deleted ----
    public void walkNodes(Proc p) {
	for (Iterator i = getNodes().iterator(); i.hasNext();) {
	    ActivityItem item = (ActivityItem)i.next();
	    p.call(item);
	}
    }
    ---- end deleted ---- */

    public void walkNodeEnds(Proc p) {
	for (Iterator i = getNodes().iterator(); i.hasNext();) {
	    ActivityItem item = (ActivityItem)i.next();
	    p.call(item.getBegin());
	    p.call(item.getEnd());
	}
    }

    public List<PNodeEnd> getNodeEnds() {
	// Returns a snapshot of the current state
	List result = new LinkedList<PNodeEnd>();
	for (Iterator i = getNodes().iterator(); i.hasNext();) {
	    ActivityItem item = (ActivityItem)i.next();
	    result.add(item.getBegin());
	    result.add(item.getEnd());
	}
	return result;
    }

    /**
     * Assigns values to {@link Variable}s.  Assignments must
     * go through this method rather than directly calling
     * {@link Variable#setValue(Object)}.
     */
    public void bindVariables(Map bindings) {
	beginUndoableTransaction("Bind variables");
	try {
	    do_bindVariables(bindings);
	}
	finally {
	    endUndoableTransaction("Bind variables");
	}
    }

    private void do_bindVariables(Map bindings) {
	// /\/: It's possible a Variable might have been bound
	// between the time something got a list of unbound vars
	// and the time when this was called.  Should we check
	// all vars before giving any of them a value?
	variableManager.tryBindings(bindings);
	for (Iterator i = bindings.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    Variable var = (Variable)e.getKey();
	    Object val = e.getValue();
	    // /\/: Sometimes in SlipExpander the var already
	    // has the desired value.
	    if (var.getValue() == val)
		Debug.noteln("Already has desired value", var);
	    else
		var.setValue(val);
	}
	fireNewBindings(bindings);
	runSentinels();
	// We call the variable manager at the end, because this
	// might result in more bindings, and we want the earlier
	// bindings to have already happened.
	variableManager.newBindings(bindings);
    }

    protected void forcedBindings(MatchEnv forced) {
	// /\/: For now, at least, we want to tell the user,
	// but this class shouldn't be doing UI stuff.
	if (Parameters.isInteractive())
	    Util.displayAndWait(null,
	      Strings.foldLongLine("Discovered necessary bindings " + forced));
	bindVariables(forced);
    }

    public void applyEnv(MatchEnv env) {
	Map bindings = env.getVariableBindings(Variable.class);
	if (bindings != null)
	    bindVariables(bindings);
    }

    public void logBindings(Map bindings) {
	if (ip2.getOptionManager().canTakeInput())
	    ip2.log(new BindingEvent
		    ((Map)Variable.revertAllVars(bindings)));
    }

    public void addConstraint(Constrainer c) {
	addConstraint(null, c);
    }

    /// Called by a PlanInstaller for top-level constraints.
    public void addConstraint(Map idToItemMap, Constrainer c) {
	Symbol type = c.getType();
	Symbol relation = c.getRelation();
	if (type == S_WORLD_STATE) {
	    if (ObjectWalker.findIf(c, Fn.isInstanceOf(ItemVar.class))
		    != null)
		throw new IllegalArgumentException
		    ("Variables are not allowed in " + c);
	    if (relation == S_EFFECT) {
		// /\/: Now treat as Constraint rather than Constrainer
		Constraint w = (Constraint)c;
		PatternAssignment pv = (PatternAssignment)w.getParameter(0);
		handleEffects(Lisp.list(pv));
		return;
	    }
	}
	if (type == Ordering.S_TEMPORAL) {
	    if (relation == Ordering.S_BEFORE) {
		Debug.expect(idToItemMap != null, "no idToItemMap for", c);
		PNodeEnd.addOrdering(idToItemMap, (Ordering)c);
		tpnm.addOrdering(idToItemMap, (Ordering)c);
		return;
	    }
	}
	if (type == MatchChoice.S_VARIABLE) {
	    if (relation == MatchChoice.S_MATCH_CHOICE) {
		Constraint mc = (Constraint)c;
		variableManager.add(new MatchChoice(mc));
		// /\/: Have to recalculate...  However, for now
		// we get here only when loading a plan, and a
		// recalc is done there.
		return;
	    }
	}
	Debug.noteln("Can't handle " + c);
	
	if (!otherConstraints.contains(c)) {
	    // Don't add if already present?  \/\
	    otherConstraints.add(c);
	}
    }

    public void addConstraint(PNode node, Constraint c) {
        markUndoPoint("Add constraint");
	Symbol type = c.getType();
	if (type == S_WORLD_STATE) {
	    worldStateCM.addConstraint(node, c);
	    if (c.getRelation() != S_EFFECT)
		nodeFilterConstraints.addValue(node, c);
	    return;
	}
	if (type == AdviceManager.S_ADVICE) {
	    adviceManager.addConstraint(node, c);
	    otherNodeConstraints.addValue(node, c); // /\/
	    return;
	}
	if (type == Refinement.S_COMPUTE) {
	    nodeFilterConstraints.addValue(node, c);
	    return;
	}
	if (type == Ordering.S_TEMPORAL) {
	    tpnm.addConstraint(node, c);
	    // /\/: Record constraints centrally rather than in CMs.
	    nodeTimeConstraints.addValue(node, c);
	    return;
	}
	constraintAssociator.addConstraint(node, c);
	otherNodeConstraints.addValue(node, c);
    }

    public void evalAtBegin(PNodeEnd ne, List<Constraint> constraints) {
	for (Constraint c: constraints) {
	    constraintAssociator.evalAtBegin(ne, c);
	}
    }

    public void evalAtEnd(PNodeEnd ne, List<Constraint> constraints) {
	for (Constraint c: constraints) {
	    constraintAssociator.evalAtEnd(ne, c);
	}
    }

    public void linkBefore(PNodeEnd from, PNodeEnd to) {
	from.linkBefore(to);
	tpnm.addTimeConstraintElseFail(from, to);
    }

    public void linkAfter(PNodeEnd to, PNodeEnd from) {
	linkBefore(from, to);
    }

    public void addOrderingsAsTimeConstraints(PNode node,
					      Map nameToChild,
					      ListOfOrdering orderings) {
	// /\/: Should we give them all to the TPN at once?
	if (orderings != null) {
	    for (Iterator i = orderings.iterator(); i.hasNext();) {
		Ordering ord = (Ordering)i.next();
		PNodeEnd from = PNodeEnd.fromRef(nameToChild, ord.getFrom());
		PNodeEnd to = PNodeEnd.fromRef(nameToChild, ord.getTo());
		tpnm.addTimeConstraintElseFail(from, to);
	    }
	}
	// Ensure children linked to parent.
	// This is at the node-end level.
	// The children should be new, and so not have links
	// with anything other than siblings.
	// /\/: It's a pain to have to do this explicitly when
	// there's a PNode method that already does it for orderings.
	PNodeEnd p_b = node.getBegin();	// node == parent
	PNodeEnd p_e = node.getEnd();
	for (Iterator i = node.getChildren().iterator(); i.hasNext();) {
	    PNode child = (PNode)i.next();
	    PNodeEnd c_b = child.getBegin();
	    PNodeEnd c_e = child.getEnd();
	    if (c_b.getPreConstraints().isEmpty())
		// c_b.linkAfter(p_b);
		tpnm.addTimeConstraintElseFail(p_b, c_b);
	    if (c_e.getPostConstraints().isEmpty())
		// c_e.linkBefore(p_e);
		tpnm.addTimeConstraintElseFail(c_e, p_e);
	}
    }

    public void deleteConstraint(Constraint c) {
	Debug.noteln("Asked to delete", c);
	Symbol type = c.getType();
	Symbol relation = c.getRelation();
	if (type == S_WORLD_STATE) {
	    if (relation == S_EFFECT) {
		PatternAssignment pv = (PatternAssignment)c.getParameter(0);
		deleteEffect(pv);
		return;
	    }
	}
	// /\/: What if constraints contain variables?
	if (otherConstraints.contains(c)) {
	    otherConstraints.remove(c);
	    return;
	}
	throw new IllegalArgumentException
	    ("No matching constraint to delete: " + c);
    }

    public ListOfConstraint getNodeConditions(PNode node) {
	// return worldStateCM.getNodeConditions(node);
	return getNodeFilters(node);
    }

    public ListOfConstraint getNodeFilters(PNode node) {
	// /\/: Should try to keep the original order rather
	// then in effect move all compute conds to last.
	ListOfConstraint filters =
	    (ListOfConstraint)nodeFilterConstraints.get(node);
	if (filters == null)
	    filters = new ContextListOfConstraint();
	return filters;
    }

    public List getNodeEffects(PNode node) {
	return worldStateCM.getNodeEffects(node);
    }

    public List getNodeTimeConstraints(PNode node) {
	return (List)nodeTimeConstraints.get(node);
    }

    public List<Constraint> getOtherNodeConstraints(PNode node) {
	return (List<Constraint>)otherNodeConstraints.get(node);
    }

    public List getOtherConstraints() {	// for PlanMaker /\/
	return otherConstraints;
    }

    public Map getWorldStateMap() {	// for PlanMaker and the simulator /\/
	return worldStateCM.getWorldStateMap();
    }

    public void setWorldStateMap(Map m) { // for Slip /\/
	// Not implemented until the IPlanModelManager \/\
	throw new UnsupportedOperationException();
    }

    public Object getWorldStateValue(LList pattern) {
	return worldStateCM.getPatternValue(pattern);
    }

    public void loadAnyNewComputeSupportCode() {
	Object code = getAnyNewComputeSupportCode();
	if (code != null)
	    variableManager.computeInterpreter
		.loadSupportCode(code);
    }

    protected Object getAnyNewComputeSupportCode() {
	Object code = ip2.domain.getAnnotation(S_COMPUTE_SUPPORT_CODE);
	if (code != computeSupportCode)
	    return computeSupportCode = code;
	else
	    return null;
    }

    // We have to override at least one annotation method
    // so that adding or changing an annotation on the model
    // can be undone.  /\/: Any others?

    public void setAnnotation(Object key, Object value) {
        markUndoPoint("Set annotation");
        super.setAnnotation(key, value);
    }

    /*
     * Undo
     */

    // Undo is handled by the option-manager because it involves contexts.

    // There are a number of things that can - and should - stop "undo"
    // processing.  The first is that there may not be an option-manager;
    // and that generally happens in agents (such as the mini ones
    // used when planning) where we don't want undo-points to affect
    // contexts.  /\/: However, it would be better to make that reason
    // be more explicit, for instance by having agents have to
    // explicitly enable "undo".

    // Transactions are used to ensure there's only one undo-point
    // for a "big" action - such as expanding an activity - that
    // should be treated as a unit.  /\/: That could be simplified
    // (and perhaps transactions could be eliminated) if undo-points
    // were always associated with particular actions initiated
    // by the user.

    // For more, see the option-manager.

    public boolean undoIsEnabled() {
        return ip2.getOptionManager() != null; // for now /\/
    }

    public void undo() {
        if (undoIsEnabled())
            ip2.getOptionManager().undo();
    }

    public void undo(String noteToMatch) {
	if (undoIsEnabled()) {
	    List<UndoAction> trail = getUndoTrail();
	    if (!trail.isEmpty() && trail.get(0).getNote().equals(noteToMatch))
		undo();
	    else {
		ip2.getOptionManager().printUndoTrail();
		throw new UndoException
		    ("Didn't find expected undo record for " + noteToMatch);
	    }
	}
    }

    public List<UndoAction> getUndoTrail() {
	return undoIsEnabled() ? ip2.getOptionManager().getUndoTrail() : null;
    }

    public void markUndoPoint(String note) {
        if (undoIsEnabled())
            ip2.getOptionManager().markUndoPoint(note);
    }

    public void saveUndoAction(UndoAction un) {
        // In this case, the note is assumed to be part of the UndoAction.
        if (undoIsEnabled())
            ip2.getOptionManager().saveUndoAction(un);
    }

    /**
     * Packages an undoable transaction as one method call.
     *
     * <p>An alternative is to write a try-finally "by hand".
     * It should look like this:
     * <pre>
     *  Ip2ModelManager mm = ...;
     *  ...
     *
     *  mm.beginUndoableTransaction("Note");
     *  try {
     *      ... undoable operations ...
     *  }
     *  finally {
     *      mm.endUndoableTransaction("Note");
     *  }
     * </pre>
     * Note that the call to beginUndoableTransaction must be
     * directly before the "try".</p>
     */
    public void undoableTransaction(String note, Runnable r) {
        if (undoIsEnabled())
            ip2.getOptionManager().undoableTransaction(note, r);
        else
            r.run();
    }

    public void beginUndoableTransaction(String note) {
        if (undoIsEnabled())
            ip2.getOptionManager().beginUndoableTransaction(note);
    }

    public void endUndoableTransaction(String note) {
        if (undoIsEnabled())
            ip2.getOptionManager().endUndoableTransaction(note);
    }

    /*
     * Filters
     */

    public List evalFilters(ListOfConstraint conds, MatchEnv env) {
	Map worldStateMap = worldStateCM.getWorldStateMap();
	return variableManager.evalFilters(conds, worldStateMap, env);
    }

    public ListOfConstraint testFilters(ListOfConstraint conds, MatchEnv env) {
	Map worldStateMap = worldStateCM.getWorldStateMap();
	return variableManager.testFilters(conds, worldStateMap, env);
    }

    public List reevaluateFilters(ListOfConstraint conds) {
	Debug.noteln("");
	Debug.noteln("Re-evaluating filters", conds);
	Map worldStateMap = worldStateCM.getWorldStateMap();
	MatchEnv initialEnv = new MatchEnv();
	List envs = variableManager.evalFilters(conds, worldStateMap,
						initialEnv);
	MatchChoice mc = addMatchChoice(envs);
	Debug.noteln("");
	return mc.getLiveBranches();
    }

    public MatchChoice addMatchChoice(List envs) {
	List branches = Bindings.mapsToBindings(envs);
	MatchChoice mc = new MatchChoice(branches);
	variableManager.add(mc);
	variableManager.recalculate();
	variableManager.showState();
	return mc;
    }

    public void statusChanged(PNode node) {
	Debug.noteln("PMM sees new status of", node);
	if (node.getStatus() == Status.COMPLETE) {
	    handleCompletion(node);
	}
    }

    protected void handleCompletion(PNode node) {
	List effects = getNodeEffects(node);
	if (effects == null)
	    return;
	Set vars = Variable.varsAnywhereIn(effects);
	Set unbound = Variable.unboundVarsIn(vars);
	if (!unbound.isEmpty()) {
	    Debug.noteln("Unbound vars on completion of", node);
	    Debug.noteln("Vars are ", unbound);
	    if (Parameters.isInteractive())
		JOptionPane.showMessageDialog(null,
	            new Object[] {"Unbound vars on completion of " + node,
			          "Vars are: " + unbound},
		    "Warning",
		    JOptionPane.WARNING_MESSAGE);
	    node.setStatus(Status.IMPOSSIBLE);
	    addSentinel(new BindingSentinel(node, unbound));
	}
	else
	    handleEffects(node, effects);
    }

    public void handleEffects(PNode node, List effects) {
	// N.B. All Variables in the effects must be bound.
	// /\/: Takes a list of PatternAssignments, NOT a list of Constraints.
        if (effects.isEmpty()) return;
        beginUndoableTransaction("Effects at " + node);
        try {
            Map delta = worldStateCM.handleEffects(node, effects);
            fireStateChange(delta);
            runSentinels();
        }
        finally {
            endUndoableTransaction("Effects at " + node);
        }
    }

    protected void handleEffects(List effects) {
        if (effects.isEmpty()) return;
        beginUndoableTransaction("Handle effects");
        try {
            Map delta = worldStateCM.handleEffects(effects);
            fireStateChange(delta);
            runSentinels();
        }
        finally {
            endUndoableTransaction("Handle effects");
        }
    }

    // /\/: Should the "fire" calls etc be included in the undoable t.?

    protected void deleteEffect(PatternAssignment pv) {
        beginUndoableTransaction("Delete effect");
        try {
            worldStateCM.deleteEffect(pv);
        }
        finally {
            endUndoableTransaction("Delete effect");
        }
	Map delta = new HashMap();
	delta.put(pv.getPattern(), pv.getValue());
	fireStateDeletion(delta);
    }

    /*
     * Simple condition-testing and execution.
     */

    /**
     * Returns a list of {@link MatchEnv}s containing one env for each
     * way in which all of the refinement's preconditions can be satisfied
     * by the curent world state.  The preconditions are the <tt>world-state
     * condition</tt> constraints, and the <tt>compute</tt> constraints,
     * listed in the refinement.  Each MatchEnv will also contain any
     * bindings obtained by matching the activity's pattern to to
     * the refinement's.  Note that the list will be empty if
     * the preconditions could not be satisfied.
     *
     * @return  null if the refinement's pattern does not match
     *    the activity's; otherwise, a possibly empty list of MatchEnvs
     *    as described above.
     */
    public List satisfyRefinementPreconditions(Activity act, Refinement r) {
	LList pattern = act.getPattern();
	pattern = putVariablesInPattern(pattern); // just in case? /\/
	MatchEnv e = Matcher.match(r.getPattern(), pattern);
	if (e == null)
	    return null;
	else {
	    ListOfConstraint preConds = r.getFilterConstraints();
	    if (preConds.isEmpty())
		return Collections.singletonList(e);
	    List envs = evalFilters(preConds, e);
	    Debug.noteln("satisfyPreconditions(" + act + ", " + r + ") ==> " +
			 envs);
	    return envs;
	}
    }

    /**
     * Returns a copy of the refinement in which each {@link ItemVar}
     * that appears in the refinement is replaced the value, if any,
     * that it has in the MatchEnv.
     *
     * @throws MissingValuesException if any variables in the
     *    refinement did not have values.
     */
    public Refinement fillInRefinement(Refinement r, MatchEnv env) {
	// Fill in variable values in the refinement.
	return r.instantiate(env);
    }

    /**
     * Applies the <tt>world-state effect</tt>s specified by the
     * refinement, changing the current world state.  The
     * MatchEnv should be one that satisfies the refinement's
     * preconditions, such as one obtained by calling
     * {@link #satisfyRefinementPreconditions(Activity, Refinement)}.
     * All variables in the refinement should have been replaced
     * by values as specified by the MatchEnv.  That can be done
     * by calling {@link #fillInRefinement(Refinement, MatchEnv)}
     * on the original refinement as obtained from a domain.
     *
     * <p>In addition to changing the world state, this method
     * will assign values to any Variables listed in the MatchEnv.
     * </p>
     */
    public void executeRefinementEffects(Refinement r, MatchEnv env) {
	// Bind any variables that now have bindings;
	Map bindings = env.getVariableBindings(Variable.class);
	if (bindings != null)
	    this.bindVariables(bindings);
	// Execute the effects from the refinement;
	List effects = r.getEffectConstraints();
	if (!effects.isEmpty()) {
	    this.addConstraints(effects);
	}
    }

    /**
     * A sentinel that lets an activity become complete when all the
     * variables in its effects have values.
     */
    protected class BindingSentinel implements Sentinel {

	PNode node;
	Set unbound;

	BindingSentinel(PNode node, Set unbound) {
	    this.node = node;
	    this.unbound = new HashSet(unbound); // copy
	}

	public boolean isReady() {
	    Debug.noteln("Testing BindingSentinel for", node);
	    Set stillUnbound = Variable.unboundVarsIn(unbound);
	    return stillUnbound.isEmpty();
	}

	public void run() {
	    Debug.noteln("Able to complete", node);
	    Debug.noteln("Because all variables bound.");
	    node.setStatus(Status.COMPLETE);
	}

    }

    /*
     * State
     */

    public Plan getPlan() {
	return new PlanMaker(ip2).getPlan();
    }

    public void setPlan(Plan plan) {
	// /\/ S.b. called loadPlan, because it adds to whatever
	// plan is already in the MM.
        beginUndoableTransaction("Load plan");
        try {
            PlanInstaller pi = new PlanInstaller(ip2, plan);
            pi.installPlan();
            postProcessInstalledPlan(pi);
        }
        finally {
            endUndoableTransaction("Load plan");
        }
    }

    protected void postProcessInstalledPlan(PlanInstaller pi) {
	pi.walkInstalledPlan(new PlanInstaller.PlanWalker() {
	    public void visitAgendaItem(AgendaItem item) {
		if (!(item instanceof ActivityItem))
		    return;
		if (item.getStatus() == Status.POSSIBLE
		      && !Collect.isEmpty(item.getChildren())
		      && Collect.isEmpty(getNodeConditions(item))) {
		    item.setStatus(Status.EXECUTING);
		}
	    }
	});
    }

    /*
     * Sentinels
     */

    public void addSentinel(Sentinel r) {
	sentinels.add(r);
    }

    public void removeSentinel(Sentinel r) {
	sentinels.remove(r);
    }

    protected List getSentinels() {
	return sentinels;
    }

}

// Issues:
// * There's only the singular deleteCondition and deleteEffect.
// * The whole idea of deleting constraints is very questionable.
