/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri Oct 12 16:22:52 2007 by Jeff Dalton
 * Copyright: (c) 2001 - 2007, AIAI, University of Edinburgh
 */

package ix.ip2;

import javax.swing.*;
import java.util.*;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.net.MalformedURLException;

import java.lang.reflect.Constructor;

import ix.ip2.event.AgendaListener;
import ix.ip2.test.TestElement;

import ix.iface.domain.DomainParser;
import ix.iface.util.Reporting;
import ix.iface.util.LogoPanel;
import ix.iface.util.ToolController;
import ix.iface.util.KeyValueTable; // for LexicographicComparator /\/

import ix.ispace.*;

import ix.iplan.IPlanOptionManager;

import ix.icore.domain.*;
import ix.icore.domain.event.*;
import ix.icore.process.ProcessModelManager;
import ix.icore.plan.Plan;
import ix.icore.*;

import ix.ichat.ChatMessage;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.xml.*;


/** 
 * The generic I-P2 class and application main program 
 */

public class Ip2 extends IXAgent {

    protected Domain domain = new Domain();

    protected ProcessModelManager modelManager;
    protected PanelController controller;

    protected IPlanOptionManager optionManager;

    protected List resetHooks = new LinkedList(); // List of Runnable

    protected Ip2Frame frame;

    boolean classic;		// use "old" viewers?

    protected AgendaViewer activityViewer;
    protected AgendaViewer issueViewer;
    protected StateViewer stateViewer;
    protected AnnotationViewer annotationViewer;

    // Some customizable fields
    protected String agentTypeName = "I-X Process Panel";
    protected String logoLine1 = agentTypeName;
    protected String logoLine2 = "Based on I-X Technology";
    protected String logoImage = "ip2-logo.gif";
    protected boolean showOptions = false; // default value

    public Ip2() {
	super();
	// Change some defaults
	displayName = "Process Panel";
    }

    /** Constructor for subclasses that need to creat a second agent. */
    protected Ip2(boolean setAgent) {
	super(setAgent);
	displayName = "Process Panel";
    }

    /**
     * Main program.
     */
    public static void main(String[] argv) {
	Util.printGreeting("I-P2");
	new Ip2().mainStartup(argv);
    }

    public String getAgentTypeName() {
	return agentTypeName;
    }

    public PanelController getController() {
	return controller;
    }

    public ProcessModelManager getModelManager() {
	return modelManager;
    }

    public Ip2ModelManager getIp2ModelManager() {
	return (Ip2ModelManager)modelManager;
    }

    public Domain getDomain() {
	return domain;
    }

    public Plan getPlan() {
	return modelManager.getPlan();
    }

    public IPlanOptionManager getOptionManager() {
	return optionManager;
    }

    public Ip2Frame getFrame() {
	return frame;
    }

    public SortedSet getActivityPatternSyntaxes() {
	SortedSet result =
	    new TreeSet(new KeyValueTable.LexicographicComparator());
	result.addAll(controller.getActivityAgenda().getSyntaxList());
	result.addAll(Collect.map
		       (new LinkedList(),
			getDomain().getRefinements(),
			Fn.accessor(Refinement.class, "getPattern")));
	return result;
    }

    /**
     * Called to create this agent's model-manager.
     */
    protected ProcessModelManager makeModelManager() {
	return new Ip2ModelManager(this);
    }

    /**
     * Called to create this agent's controller.
     */
    protected PanelController makeController() {
	return new PanelController(this);
    }

    /**
     * Command-line argument processing for arguments used by all
     * versions of I-P2.
     */
    @Override
    protected void processCommandLineArguments() {
	super.processCommandLineArguments();

	logoLine1   = Parameters.getParameter("logo-line-1", logoLine1);
	logoLine2   = Parameters.getParameter("logo-line-2", logoLine2);
	logoImage   = Parameters.getParameter("logo-image", logoImage);

	showOptions = Parameters.haveParameter("option-directory")
	           || Parameters.getBoolean("show-options", showOptions);

	classic = Parameters.getBoolean("classic", false);

    }

    /**
     * Completes basic I-P2 setup and initialization.
     */
    @Override
    public void startup() {

	// Make controller and model-manager
	modelManager = makeModelManager();
	controller = makeController();
	controller.connectTo(modelManager);

	// Make the option manager.
	// The model-manager and the controller's agendas
	// must exist before it can be connected.
	optionManager = new IPlanOptionManager(this);
	optionManager.connectYourself();

	// Create the frame without filling in anything interesting.
	// This allows the viewers to add tools, for example.
	frame = makeIp2Frame();
        Debug.recordMainFrame(frame);

	// Make the viewers
	activityViewer = makeActivityViewer();
	issueViewer = makeIssueViewer();
	stateViewer = makeStateViewer();
	annotationViewer = makeAnnotationViewer();

	// Connect the viewers as listeners
	controller.addActivityListener((AgendaListener)activityViewer);
	controller.addIssueListener((AgendaListener)issueViewer);
	modelManager.addProcessStatusListener(stateViewer);
	modelManager.addAnnotationListener(annotationViewer);

	// Add handlers
  	addHandlers();
	addHandlers(Parameters.getList("additional-handlers"));

	// Read any domain that should be present from the start,
	// /\/ Not clear where it's best to do this.
	readDomain(domain);

	// Complete the frame
	frame.setUp();
	Util.swingAndWait(new Runnable() {
	    public void run() {
		// /\/: We do some adjustment after making the
		// frame visible, so we switch threads here.
		completeStartup();
	    }
	});

    }

    /**
     * Called in the AWT event thread to make the main GUI frame
     * visible and perform any additional setup that ought to be
     * done in that thread.
     *
     * @see Ip2Frame#becomeVisible()
     */
    protected void completeStartup() {
	frame.becomeVisible();

	// /\/: It's not clear where it's best to do this
	installAgentExtensions();

	initOptions();

	// Load a plan if desired
	// We have to do this at the end, because viewers may not be
	// able to deal with any pre-existing information.
	loadInitialPlan();

    }

    protected void initOptions() {
	if (showOptions)
	    optionManager.initOptions();
	else
	    optionManager.initOneOption();
    }

    /**
     * Called to restore the initial state.
     */
    public void reset() {
	// /\/: Things should register for reset instead of being known here.
	modelManager.reset();
	controller.reset();
	resetViewers();
        Util.runHooks("reset", resetHooks);
    }

    public void resetViewers() {
	// /\/: This should be done via the agendas and model-manager.
	// They should tell their listeners about the reset.
	if (activityViewer != null) activityViewer.reset();
	if (issueViewer != null) issueViewer.reset();
	if (stateViewer != null) stateViewer.reset();
	if (annotationViewer != null) annotationViewer.reset();
	if (frame != null) frame.validate();
    }

    public void resetAllButState() {
	Ip2ModelManager mm = (Ip2ModelManager)modelManager;
	// Get a copy of the current state.  The Map returned
	// by the model manager is not a copy and will be modified
	// during the reset().  /\/
	List state = PatternAssignment.mapToAssignments(mm.getWorldStateMap());
	reset();
	mm.handleEffects(state);
    }

    /**
     * Remove everything in the model in the current context.
     */
    public void clearModel() {
	modelManager.clear();
	controller.clearModel();
    }

    public void clearAllButState() {
	// Compare resetAllButState().
	Ip2ModelManager mm = (Ip2ModelManager)modelManager;
	List state = PatternAssignment.mapToAssignments(mm.getWorldStateMap());
	clearModel();
	mm.handleEffects(state);
    }

    /**
     * Records an object that will be run when this panel is asked
     * to reset.  The reset hooks run after other things have been
     * reset.
     */
    public void addResetHook(Runnable hook) {
	resetHooks.add(hook);
    }

    /**
     * Called when the agent should cease execution.
     */
    public void exit() {
	//\/ Call super method?
	System.exit(0);		// should let some things know?  /\/
    }

    /**
     * Called to make the main user interface frame.  This merely
     * creates the frame, with almost nothing filled in.  The frame's
     * {@link Ip2Frame#setUp()} is later called to do most of the work.
     */
    protected Ip2Frame makeIp2Frame() {
	return classic
	    ? new Ip2Frame(this)
	    : new NewIp2Frame(this);
    }

    /**
     * Called to add "About" information.  The method provided by
     * the Ip2 class does nothing; it should be overridden in subclasses
     * that have something to add.
     */
    public void addAboutInfo(List about) {
    }

    /**
     * Called to create the activity viewer.
     */
    protected AgendaViewer makeActivityViewer() {
	if (classic)  return new ActivityViewTable(this);
	return (AgendaViewer)
	    makeViewer("activity-viewer-class", ActivityTableViewer.class);
    }

    /**
     * Called to create the issue viewer.
     */
    protected AgendaViewer makeIssueViewer() {
	if (classic) return new IssueViewTable(this);
	return (AgendaViewer)
	    makeViewer("issue-viewer-class", IssueTableViewer.class);
    }

    /**
     * Called to create the state viewer.
     */
    protected StateViewer makeStateViewer() {
	return (StateViewer)
	    makeViewer("state-viewer-class", StateViewTable.class);
    }

    /**
     * Called to create the annotation viewer.
     */
    protected AnnotationViewer makeAnnotationViewer() {
	return (AnnotationViewer)
	    makeViewer("annotation-viewer-class", AnnotationViewTable.class);
    }

    /**
     * Utility use to construct viewers.
     *
     * @param parameterName  the name of a paramter whose value,
     *    if given, is a class name.
     * @param defaultClass  the class to use if the parameter was
     *    not given.
     * @return an instance of the specified class obtained by
     *    calling its 1-argument (Ip2) constructor.
     *
     * @see Parameters#getClass(String, Class)
     */
    protected Object makeViewer(String parameterName, Class defaultClass) {
	Class viewerClass = Parameters.getClass(parameterName, defaultClass);
	Class[] sig = {Ip2.class};
	try {
	    Constructor cons = viewerClass.getConstructor(sig);
	    return cons.newInstance(new Object[] {this});
	}
	catch (Exception e) {
	    Debug.noteException(e);
	    throw new RethrownException
		(e, "Cannot instantiate " + parameterName +
		    " class " + viewerClass.getName() +
		    " because " + Debug.describeException(e));
	}
    }


    // Reloading viewers the hard way.  :(

    // /\/: This is a (hopefully) temporary expedient, because
    // the viewers don't have a proper reload function.
    // /\/: There will be a problem if viewers treat newBindings
    // events as something more than a reason to redisplay.
    // If any viewer tries to keep track of variables and their
    // values, it will probably become confused by context changes.
    // /\/: Note that in_reloadViewers and IPlanOptionManager's
    // noticePlanChangeEvents have very similar functions.

    public void reloadViewers() {
        try {
            in_reloadViewers = true;
            do_reloadViewers();
        }
        finally {
            in_reloadViewers = false;
        }
    }

    private boolean in_reloadViewers = false;

    public boolean isReloadingViewers() {
        return in_reloadViewers;
    }

    protected void do_reloadViewers() {
        Debug.noteln("Realoding viewers");
	Ip2 ip2 = this;
	// ip2.resetViewers() should already have been called.
	Ip2ModelManager mm = (Ip2ModelManager)ip2.getModelManager();
	PanelController controller = ip2.getController();
	// Issues
	Agenda issueAgenda = controller.getIssueAgenda();
	List issueItems = issueAgenda.getItems();
	for (Iterator i = issueItems.iterator(); i.hasNext();) {
	    IssueItem item = (IssueItem)i.next();
	    issueAgenda.fireItemAdded(item);
	}
	// Activities
	Agenda activityAgenda = controller.getActivityAgenda();
	List activityItems = activityAgenda.getItems();
	for (Iterator i = activityItems.iterator(); i.hasNext();) {
	    ActivityItem item = (ActivityItem)i.next();
	    activityAgenda.fireItemAdded(item);
	}
	// World state
	Map worldState = mm.getWorldStateMap();
	mm.fireStateChange(worldState);
	// Annotations
	Annotations annotations = mm.getAnnotations();
	if (annotations != null) {
	    for (Iterator i = annotations.entrySet().iterator()
		     ; i.hasNext();) {
		Map.Entry e = (Map.Entry)i.next();
		mm.fireSetAnnotation(e.getKey(), e.getValue());
	    }
	}
    }


    /**
     * Read in any initial domain descriptions.
     */
    protected void readDomain(Domain domain) {
	List domains = Parameters.getList("domain");
	for (Iterator i = domains.iterator(); i.hasNext();) {
	    String domainName = (String)i.next();
	    try {
		readDomain(domain, domainName);
	    }
	    catch (Exception e) {
		Debug.displayException
		    ("Problem with domain " + domainName, e);
		    if (!Parameters.isInteractive())
			throw new RethrownException(e);
	    }
	}
    }

    /**
     * Adds definitions to a domain.
     */
    public void readDomain(Domain domain, String resourceName) {
	// /\/: This used to be so simple ...
	// DomainParser.makeParser(domainName).readDomain(domain);
	Debug.noteln("Loading a domain from", resourceName);
	URL url = XML.toURL(resourceName);
	if (url == null) {
	    // /\/: For backwards compatibility, try it as a file
	    // in the library directory.
	    Debug.noteln("Trying " + resourceName + " as file in lib dir");
	    File file = new File(resourceName);
	    if (file.getParentFile() == null) {
		File libDir = DomainParser.getLibraryDirectory();
		Debug.noteln("Resolving " + file + " against " + libDir);
		file = new File(libDir, file.getPath());
		resourceName = file.getPath(); // for error message
		if (file.exists()) {
		    try {
			url = file.toURL();
		    }
		    catch (MalformedURLException e) {
			Debug.noteException(e);
			throw new ConsistencyException("Unexpected", e);
		    }
		}
	    }
	    if (url == null)
		throw new IllegalArgumentException
		    ("Can't find a domain named " +
		     Strings.quote(resourceName));
	}
	Domain tempDomain = (Domain)XML.readObject(Domain.class, url);
	tempDomain.checkConsistency();
	domain.takeFrom(tempDomain);
    }

    public void loadDomain(Domain dom) {
	dom.checkConsistency();
	domain.takeFrom(dom);
    }

    /**
     * Makes the LogoPanel for the application's main frame.
     * This method is in this class to make it easier to define
     * versions that have a different logo panel.
     */
    public JPanel makeLogoPanel() {
	ImageIcon icon = null;
	try {
	    icon = Parameters.haveParameter("logo-image")
		? Util.getImageIcon(logoImage) // allow files, URLs, etc
		: Util.resourceImageIcon(logoImage); // only class-loader
	}
	catch (Exception e) {
	    Debug.displayException(e);
	}
	return new LogoPanel(symbolName, logoLine1, logoLine2, icon);
    }


    protected void loadInitialPlan() {
	if (Parameters.haveParameter("plan")) {
	    List plans = Parameters.getList("plan");
	    for (Iterator i = plans.iterator(); i.hasNext();) {
		String resourceName = (String)i.next();
		try {
		    loadPlan(resourceName);
		}
		catch (Throwable t) {
		    Debug.displayException(t);
		    if (!Parameters.isInteractive())
			throw new RethrownException(t);
		}
	    }
	}
    }

    public void loadPlan() {
	Plan plan = (Plan)new XMLLoader(frame, Plan.class).loadObject();
	if (plan != null)
	    loadPlan(plan);
    }

    public void loadPlan(String resourceName) {
	Debug.noteln("Loading a plan from", resourceName);
	Plan plan = (Plan)XML.readObject(Plan.class, resourceName);
	loadPlan(plan);
    }

    public void loadPlan(Plan plan) {
	modelManager.setPlan(plan);
    }

    public void savePlanAs() {
	Plan plan = modelManager.getPlan();
	new XMLSaver(frame, Plan.class).saveObject(plan);
    }

    public void handleInput(IPC.InputMessage message) {
	Object contents = message.getContents();
	// /\/: Automatic handling is a special case for now. :(
	if (contents instanceof Issue || contents instanceof Activity) {
	    TaskItem issueOrActivity = (TaskItem)contents;
	    if (controller.canHandleAutomatically(issueOrActivity)) {
		handleReceivedReport(issueOrActivity);
		controller.handleAutomatically(issueOrActivity);
		return;
	    }
	}
	if (message.getAnnotation("is-external") != Boolean.TRUE)
	    // Internal messages are handled in the current option.
	    handleInputDirectly(message);
	else if (contents instanceof ChatMessage)
	    // Chat messages can also be handled directly.
	    handleInputDirectly(message);
	else if (contents instanceof Report) {
	    // Reports are an irritating special case.
	    optionManager.handleReportWhenOptions(message);
	}
	else {
	    // External message.
	    // /\/: Load-plan activities are a special case, at least for now.
	    Plan p = getPlanIfLoadPlanRequest(message);
	    if (p != null)
		new LoadPlanFrame(this, (Activity)contents, p);
	    // Otherwise, see if we can handle the input now or
	    // have to delay it.
	    else if (optionManager.canTakeInput())
		handleInputDirectly(message);
	    else
		optionManager.recordDelayedInput(message);
	}
    }

    public void handleInputDirectly(IPC.InputMessage message) {
        // We can't mark the undo point here because some things
        // add items further in - for example, the issue and activity
        // editors go via the viewer directly to the agenda.
        // getIp2ModelManager().markUndoPoint();
	super.handleInput(message);
    }

    private Plan getPlanIfLoadPlanRequest(IPC.InputMessage message) {
	Object contents = message.getContents();
	if (contents instanceof Activity) {
	    Activity act = (Activity)contents;
	    LList pat = act.getPattern();
            String verb = pat.get(0).toString();
	    if ((verb.equals("load-plan") || verb.equals("load-state"))
		  && pat.get(1) instanceof Plan) {
		Plan plan = (Plan)pat.get(1);
		return plan;
	    }
	}
	return null;
    }

    /**
     * Handles new issues from external sources.
     */
    @Override
    public void handleNewIssue(Issue issue) {
	handleReceivedReport(issue);
	IssueItem iss = controller.addIssue(issue);
        iss.setIsNew(true);
    }

    /**
     * Handles new activities from external sources.
     */
    @Override
    public void handleNewActivity(Activity activity) {
	handleReceivedReport(activity);
	ActivityItem act = controller.addActivity(activity);
        act.setIsNew(true);
    }

    /**
     * Handles new constraints from external sources.
     */
    @Override
    public void handleNewConstraint(Constraint constraint) {
	modelManager.addConstraint(constraint);
    }

    /**
     * Handles new reports from external  sources.
     */
    @Override
    public void handleNewReport(Report report) {
	controller.newReport(report);
	List constraints = report.getConstraints();
	if (constraints != null) {
	    for (Iterator i = constraints.iterator(); i.hasNext();) {
		Constraint c = (Constraint)i.next();
		handleNewConstraint(c);
	    }
	}
    }

    /**
     * Handles new chat messages.
     */
    @Override
    public void handleNewChatMessage(ChatMessage message) {
	frame.getChatFrameVisible().newMessage(message);
    }

    /**
     * Records the tool and adds an entry to the main frame's "Tools" menu.
     *
     * @see Ip2Frame#addTool(ToolController)
     */
    @Override
    public void addTool(ToolController tc) {
	frame.addTool(tc);
    }

    /**
     * Returns the tool of the specified name, causing it to be created
     * if it does not already exist.  Note that it returns the tool,
     * not its tool-controller, and that it does not change the tool's
     * visibility.
     *
     * @throws IllegalArgumentException if there's no tool of the
     *    specified name.
     */
    @Override
    public Object ensureTool(String toolName) {
	if (frame == null)
	    throw new IllegalArgumentException
		("There is no tool named " + Strings.quote(toolName));
	// The next line may also throw an IllegalArgumentException.
	ToolController tc = frame.toolManager.findToolElseError(toolName);
	return tc.ensureTool();
    }

    /**
     * Called when the main frame is set up to add items to the frame
     * menu bar's "Test" menu.  The method is in this class, rather than
     * in the frame class, to make it easier to define versions that have
     * different test items.  This method adds tests from the resources
     * specified by the "test-menu" parameter, if one was given;
     * otherwise it does nothing.
     *
     * @see ix.util.Parameters
     * @see PanelFrame#addTests(String)
     */
    protected void addTestMenuItems() {
	frame.addTestResources(Parameters.getList("test-menu"));
    }

    /**
     * Adds the specified test to the main frame's "Test" menu.
     */
    public void addTest(TestElement test) {
	test.addToMenu(frame);
    }

    /**
     * Add one instance of each of the specified classes as a
     * handler for issues, activities, or both, as appropriate.
     *
     * @see PanelController#addHandler(ItemHandler handler)
     */
    protected void addHandlers(List classNames) {
	for (Iterator i = classNames.iterator(); i.hasNext();) {
	    String name = i.next().toString();
	    ItemHandler handler = null;
	    // Make the handler
	    try {
		handler = ItemHandler.makeHandler(Ip2.this, name);
	    }
	    catch (Exception e) {
		Debug.displayException
		    ("Cannot make a handler of class " + name,
		     e);
		continue;
	    }
	    // Install it
	    Debug.expect(handler != null);
	    controller.addHandler(handler);
	}
    }

    /**
     * Install any built-in issue and activity handlers.  Note that the
     * order in which handlers are added here is also the order in which
     * they are asked to add handler-actions.
     */ 
    protected void addHandlers() {

	// Handler that lets the user do nothing or make items complete.
	controller
//  	    .addItemHandler(new SimpleCompletionHandler());
	    .addItemHandler(new CompletionHandler(this));

	// Condition satisfaction for pre-expanded activities.
        // This is in effect a kind of completion handler.  /\/
// 	controller
// 	    .addActivityHandler(new ConditionHandler(this));

	// Handler that lets issues be converted to activities
	controller
  	    // .addIssueHandler(new PerformAsActivityHandler(this));
	    .addIssueHandler(new TransformToActivityHandler(this));

	// Expansion
	controller
	    .addActivityHandler(new ExpandHandler(this));

	// Handler that provides forwarding to agents that have
	// required capabilities
	controller
	    .addItemHandler(new InvokeHandler(this));

	// Now all forwarding handlers require report-back.
	// Before, "Escalate" to a superior did not, and
	// "pass" to a peer had both "with report-back"
	// and "without report-back" handlers.

	// Escalate to superior
	addForwardingHandler
	    ("Escalate", AgentRelationship.SUPERIOR, true);

	// Pass to peer
	addForwardingHandler
	    ("Pass", AgentRelationship.PEER, true);

	// Delegate to subordinate
	addForwardingHandler
	    ("Delegate", AgentRelationship.SUBORDINATE, true);

	// Connecting an agent
	controller
	    .addActivityHandler(new ConnectHandler(this));

	// Setting capabilities
	controller
	    .addActivityHandler(new SetCapabilitiesHandler(this));

	// Adding handlers
	controller
	    .addActivityHandler(new AddHandlersHandler());

	// Adding extensions
	controller
	    .addActivityHandler(new AddExtensionsHandler());

	// Sending a report
	controller
	    .addActivityHandler(new SendReportHandler());

	// Querying an agent
	controller
	    .addActivityHandler(new QueryHandler());
	controller
	    .addActivityHandler(new QueryHandler.AnswerHandler());

	// Loading a plan
	controller
	    .addActivityHandler(new LoadPlanHandler(this));

	// Loading state
	controller
	    .addActivityHandler(new LoadStateHandler(this));

	// New or include domain
	controller
	    .addActivityHandler(new LoadDomainHandler(this));

	// Show URL
	controller
	    .addActivityHandler(new ShowURLHandler(this));

	// Match world-state
	controller
	    .addActivityHandler
	        (new ix.test.MatchStateHandler(this, "select"));

	// Synchronize simulation time
	controller
	    .addActivityHandler(new SyncSimulationTimeHandler(this));

    }

    public void addForwardingHandler(String verb,
				     AgentRelationship rel,
				     boolean reportBack) {
	controller.addItemHandler
	    (new ForwardingHandler(this, verb, rel, reportBack));
    }


    /*
     * Some useful handler classes
     */

    public class AddHandlersHandler extends ActivityHandler {
	Symbol S_ADD_HANDLERS = Symbol.intern("add-handlers");
	public AddHandlersHandler() {
	    super("Add issue or activity handlers");
	}
	@Override
	public  List getSyntaxList() {
	    return (LList)Lisp.readFromString
		("((add-handlers ?class-name ...))");
	}
	@Override
    	public boolean appliesTo(AgendaItem item) {
	    LList pattern = item.getPattern();
	    return pattern.length() > 1
		&& pattern.get(0) == S_ADD_HANDLERS;
	}
	@Override
	public void addHandlerActions(AgendaItem item) {
	    item.addAction
		(new HandlerAction.AutomaticWhenBound(item, this));
	}
	@Override
	public void handle(AgendaItem item) {
	    LList pattern = (LList)Variable.removeVars(item.getPattern());
	    LList classNames = pattern.cdr();
	    Ip2.this.addHandlers(classNames);
	    // /\/: We make it complete even if there were
	    // problems adding the handlers, because no exception
	    // makes it out this far.
	    item.setStatus(Status.COMPLETE);	
	}
    }

    public class AddExtensionsHandler extends ActivityHandler {
	Symbol S_ADD_EXTENSIONS = Symbol.intern("add-extensions");
	public AddExtensionsHandler() {
	    super("Add agent extensions");
	}
	@Override
	public  List getSyntaxList() {
	    return (LList)Lisp.readFromString
		("((add-extensions ?class-name ...))");
	}
	@Override
    	public boolean appliesTo(AgendaItem item) {
	    LList pattern = item.getPattern();
	    return pattern.length() > 1
		&& pattern.get(0) == S_ADD_EXTENSIONS;
	}
	@Override
	public void addHandlerActions(AgendaItem item) {
	    item.addAction
		(new HandlerAction.AutomaticWhenBound(item, this));
	}
	@Override
	public void handle(AgendaItem item) {
	    LList pattern = (LList)Variable.removeVars(item.getPattern());
	    LList classNames = pattern.cdr();
	    Ip2.this.installAgentExtensions(classNames);
	    // /\/: We make it complete even if there were
	    // problems adding the handlers, because no exception
	    // makes it out this far.
	    item.setStatus(Status.COMPLETE);	
	}
    }

}

// Issues:
// * Need a good way to get ahold of the new IssueItem or ActivityItem
//   when a new Issue or Activity is added.
