/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Sep  5 23:28:09 2007 by Jeff Dalton
 * Copyright: (c) 2001 - 2002, 2004, 2007, AIAI, University of Edinburgh
 */

package ix.ip2;

import javax.swing.*;

import java.awt.Color;
import java.awt.Container;
import java.awt.Component;
import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.*;

import java.util.*;

import ix.ip2.event.*;

import ix.iview.SimpleDomainEditor;
import ix.iface.util.XColors;

import ix.icore.domain.Refinement;

import ix.icore.process.PNode;

import ix.icore.*;
import ix.icore.domain.*;
import ix.util.*;
import ix.util.lisp.*;

/**
 * A simple editor for entering and viewing Activities
 */
class ActivityEditor extends ItemEditor {

    protected JMenuItem saveAsRefinementItem;

    ActivityEditor(AgendaViewer agendaViewer, String title) {
	super(agendaViewer, title);
	itemType = "Activity";
    }

    protected EditPanel makeEditPanel() {
	return new ActivityEditPanel();
    }

    /**
     * Constructs the menu bar
     */
    protected JMenuBar makeMenuBar() {
	saveAsRefinementItem = makeMenuItem("Save As Refinement");
        saveAsRefinementItem.setEnabled(false);
        fileMenu.add(saveAsRefinementItem);
        fileMenu.addSeparator();

        return super.makeMenuBar();
    }

    /**
     * Action interpreter
     */
    public void actionPerformed(ActionEvent e) {
	String command = e.getActionCommand();
	Debug.noteln("ActivityEditor action:", command);
	if (command.equals("Save As Refinement")) {
	    Refinement r = 
		((ActivityEditPanel)editPanel).expansionPanel
		    .makePartialRefinement();
	    agendaViewer.saveExpansion(r);
	}
	else if (handleViewCommand(e)) {
	    // Already handled
	}
	else if (command.equals("Close")) {
	    frame.setVisible(false);
	}
	else
	    Debug.noteln("Nothing to do for", command);
    }

    class ActivityEditPanel extends EditPanel {

	// Buttons that can be disabled or have their text changed.
	JButton expansionButton;

	ExpansionPanel expansionPanel = new ExpansionPanel();

	SyntaxComboBox syntaxChoice;

	ActivityEditPanel() {
	    super();
	    modifyButton = makeButton("Modify Activity");
	    expansionButton = makeButton("Specify Subactivities");
	    expansionButton.setForeground(Color.blue);
	    syntaxChoice = new SyntaxComboBox();
	    setUp();
	}

	protected void setUp() {

	    setBorder(BorderFactory.createTitledBorder("Activity"));

	    syntaxChoice.addActionListener(new SyntaxChoiceListener());

	    add(makeLeftLabel("Activity Pattern"));
	    add(syntaxChoice);
	    add(new JScrollPane(itemText));

	    add(makeLeftLabel("Annotations"));
	    add(new JScrollPane(commentText));

	    newItemButtons.add(makeButton("Add Activity"));
	    newItemButtons.add(makeButton("Clear"));
	    newItemButtons.add(makeButton("Cancel"));

	    editItemButtons.add(modifyButton);
	    editItemButtons.add(makeButton("Copy Activity"));
	    editItemButtons.add(expansionButton);
	    editItemButtons.add(makeButton("Clear"));
	    editItemButtons.add(makeButton("Cancel"));

	    add(newItemButtons);
	}

	/**
	 * Sets up to edit an existing item.
	 */
	void showItem(AgendaItem i) {
	    ensureNoExpansionPanel();
	    syntaxChoice.off();
	    Status status = i.getStatus();
	    expansionButton
		.setEnabled(status == Status.BLANK
			    || status == Status.POSSIBLE);
	    super.showItem(i);
	}

	/**
	 * Sets up to edit a new, initially empty item
	 */
	void showNewItem() {
	    ensureNoExpansionPanel();
	    syntaxChoice.on();
	    super.showNewItem();
	}

	/**
	 * Action interpreter
	 */
	public void actionPerformed(ActionEvent e) {
	    String command = e.getActionCommand();
	    Debug.noteln("Activity action:", command);
	    if (command.equals("Add Activity")) {
		addItem();
	    }
	    else if (command.equals("Modify Activity")) {
		modifyItem();
	    }
	    else if (command.equals("Copy Activity")) {
		copyItem();
	    }
	    else if (command.equals("Specify Subactivities")) {
		addExpansionPanel();
	    }
	    else if (command.equals("Install Subactivities")) {
		addSubitems();
	    }
	    else if (command.equals("Modify Subactivities")) {
		throw new ConsistencyException
		    ("Modify Subactivities button should be disabled");
	    }
	    else if (command.equals("Clear")) {
		showNewItem();
	    }
	    else if (command.equals("Cancel")) {
		// Equiv to close + clear.
		frame.setVisible(false);
		showNewItem();
	    }
	    else
		throw new UnsupportedOperationException
		    ("Nothing to do for " + command);
	}

	AgendaItem getItem() {
	    AgendaItem item = super.getItem();
	    if (item == null ||
		hasValidSyntax(item.getPattern()) ||
		Util.dialogConfirms(frame, new String[] {
		    "The pattern doesn't match the selected syntax",
		    "Do you still want to add the activity?"}))
	      return item;
	    else
	      return null;
	}

	boolean hasValidSyntax(LList pattern) {
	    // When we run into something in the syntax that we
	    // don't know how to handle, we just return true.
	    String syntax = (String)syntaxChoice.getSelectedItem();
	    if (syntax.equals(""))
		return true;	// no syntax was specified
	    LListCollector tokens = new LListCollector(pattern);
	    List words = Strings.breakAt(" ", syntax);
	    Debug.noteln("Words", words);
	    for (Iterator i = words.iterator(); i.hasNext();) {
		String w = (String)i.next();
		if (w.equals("")) { // probably never happens /\/
		    return true;
		}
		else if (Character.isLetterOrDigit(w.charAt(0))) {
		    // A literal
		    if (tokens.isEmpty()
			  || !w.equals(tokens.popElement().toString()))
			return false;
		}
		else if (w.startsWith("?")) {
		    // Matches anything
		    if (tokens.isEmpty()) return false;
		    else tokens.popElement();
		}
		else if (w.equals("...")) {
		    // One or more of the previous syntax element,
		    // which we must have matced.
		    return tokens.size() >= 0;
		}
		else if (w.equals("&rest")) {
		    return true;
		}
		else {
		    // A word we don't know how to interpret.
		    return true;
		}
	    }
	    return tokens.isEmpty();
	}

	/**
	 * Starts editing a new item based on an existing one.
	 */
	void copyItem() {
	    ensureNoExpansionPanel();
	    syntaxChoice.on();
	    super.copyItem();
	}

	/**
	 * Adds subitems to an existing item.
	 */
	void addSubitems() {
	    Refinement s = expansionPanel.makePartialRefinement();
	    agendaViewer.expandItem(editingItem, s);
	    expansionPanel.setControlsEnabled(false);
	}

	/*
	 * AgendaItemListener methods - in case an Item changes while
	 * we're editing it.
	 */

	public void statusChanged(AgendaItemEvent e) {
	    AgendaItem i = (AgendaItem)e.getSource();
	    if (i != editingItem)
		return;
	    Debug.noteln("Status change while editing item", i);

	    // Since it only gets more restrictive, and always in the
	    // same way, we only have to check that we're not still in
	    // the most permissive state.
	    Status status = i.getStatus();
	    if (status != Status.BLANK && status != Status.POSSIBLE) {
		// Modification of comments always allowed.
		// modifyButton.setEnabled(false);

		// The panel might be in the "Install Subactivities"
		// state, with the subactivities panel visible and
		// perhaps even with some subactivity patterns
		// entered; the button is still disabled.
		expansionButton.setEnabled(false);
	    }
	    
	}

	/*
	 * Pattern syntax
	 */

	class SyntaxComboBox extends JComboBox {
	    SyntaxComboBox() {
		super(getSyntaxes());
	    }
	    void off() {
		setSelectedItem("");
		setEnabled(false);
	    }
	    void on() {
		setModel(new DefaultComboBoxModel(getSyntaxes()));
		setEnabled(true);
	    }
	}

	class SyntaxChoiceListener implements ActionListener {
	    public void actionPerformed(ActionEvent e) {
		SyntaxComboBox sbox = (SyntaxComboBox)e.getSource();
		String pat = (String)sbox.getSelectedItem();
		if (pat != null) { // check needed in Java 1.3 /\/
		    Debug.noteln("Selected syntax:", pat);
		    itemText.setText(constantPart(pat));
		}
	    }
	    String constantPart(String pat) {
		List words = Strings.breakAt(" ", pat);
		String result = "";
		for (Iterator i = words.iterator(); i.hasNext();) {
		    String w = (String)i.next();
		    if (Character.isLetterOrDigit(w.charAt(0)))
			result += w + " ";
		}
		return result;
	    }
	}

	String[] getSyntaxes() {
	    // This method needs to be outside the SyntaxComboBox class
	    // so that it can be called in super(getSyntaxes()).  /\/
	    Ip2 ip2 = (Ip2)IXAgent.getAgent();
	    Set syntaxes = ip2.getActivityPatternSyntaxes();
	    List strings = new LinkedList();
	    strings.add("");
	    for (Iterator i = syntaxes.iterator(); i.hasNext();) {
		LList pattern = (LList)i.next();
		strings.add(PatternParser.unparse(pattern));
	    }
	    return Strings.toArray(strings);
	}

	/*
	 * Expansion / subitem panel
	 */

	/**
	 * Adds a panel for editing subitems and their ordering constraints.
	 */
	void addExpansionPanel() {
	    // Panel shouldn't already be there.
	    Debug.expect(!expansionPanelIsShown());

	    // The panel is only for editing existing items.
	    Debug.expect(editingItem != null,
			 "can't edit new item subitems");

	    expansionPanel.loadFromItem(editingItem);

	    // Allow the user to turn the item's expansion into a refinement.
	    saveAsRefinementItem.setEnabled(true);

	    if (editingItem.getChildren().isEmpty()) {
		// No existing subitems.
		expansionButton.setText("Install Subactivities");
		Status status = editingItem.getStatus();
		if (status == Status.BLANK || status == Status.POSSIBLE) {
		    expansionButton.setEnabled(true);
		    expansionPanel.setControlsEnabled(true);
		}
		else {
		    expansionButton.setEnabled(false);
		    expansionPanel.setControlsEnabled(false);
		}
	    }
	    else {
		// We don't yet allow existing subitems to be modified. /\/
		expansionButton.setText("Modify Subactivities");
		expansionButton.setEnabled(false);
		expansionPanel.setControlsEnabled(false);
	    }

	    // Add the panel just above the comments/annotations.
	    add(expansionPanel, getComponentCount() - 3);
	    frame.pack();
	}


	/**
	 * Removes the panel for editing subitems if it is being displayed.
	 */
	void ensureNoExpansionPanel() {
	    if (expansionPanelIsShown()) {
		saveAsRefinementItem.setEnabled(false);
		// The expansionButton can bring the subitem panel back
		expansionButton.setText("Specify Subactivities");
		expansionButton.setEnabled(true);
		remove(expansionPanel);
		frame.pack();
	    }
	}

	/**
	 * Determines whether the expansion panel is being displayed.
	 */
	boolean expansionPanelIsShown() {
	    return getComponent(getComponentCount() - 4) == expansionPanel;
	}


	/**
	 * A panel for editing subitems and their ordering constraints.
	 */
	class ExpansionPanel extends JPanel {

	    JTextArea subitemText = new JTextArea(5, textCols);

	    SimpleDomainEditor.TemporalConstraintPanel constraintPanel
		= new SimpleDomainEditor.TemporalConstraintPanel();

	    ExpansionPanel() {
		super();
		setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
		// setBorder(BorderFactory.createTitledBorder("Expansion"));
		setUp();
	    }

	    void setUp() {
		JLabel subitemsLabel = new JLabel("Subactivities");
		subitemsLabel.setForeground(Color.blue);
		add(makeLeftLabel(subitemsLabel));
		add(new JScrollPane(subitemText));
		subitemText.setForeground(Color.blue);

		add(makeLeftLabel("Constraints"));
		add(constraintPanel);
	    }

	    void setControlsEnabled(boolean v) {
		subitemText.setEnabled(v);
	    }

	    Refinement makePartialRefinement() {
		// Pattern
		LList pattern = 
		    // PatternParser.parse(itemText.getText().trim());
		    // (LList)Variable.revertVars(editingItem.getPattern());
		    Lisp.NIL;

		// Nodes
		String expansion = subitemText.getText().trim();
		ListOfNodeSpec nodes = 
		    SimpleDomainEditor.parseNodes(expansion);
		if (nodes.isEmpty())
		    throw new IllegalArgumentException
			("There are no subactivities");

		// Orderings
		String ordChoice = constraintPanel.getOrderingChoice();
		ListOfOrdering orderings = ordChoice.equals("Sequential")
		    ? SimpleDomainEditor.sequentialOrderings(nodes)
		    : null;
		
		// Refinement
		Refinement partial = new Refinement();
		partial.setPattern(pattern);
		partial.setNodes(nodes);
		partial.setOrderings(orderings);
		partial.setComments(commentText.getText());
		return partial;
	    }

	    void loadFromItem(AgendaItem item) {
		Debug.noteln("Editing orderings of", item);
		List subnodes = item.getChildren();
		List orderings = itemOrderings(item);
		Debug.noteln("Orderings = ", orderings);

		subitemText.setText(subitemsToText(subnodes));
		constraintPanel.loadFromOrderings(subnodes, orderings);
	    }

	    String subitemsToText(List children) {
		String lineSeparator = System.getProperty("line.separator");
		StringBuffer result = new StringBuffer();
		for (Iterator n = children.iterator(); n.hasNext();) {
		    AgendaItem child = (AgendaItem)n.next();
		    result.append(PatternParser.unparse(child.getPattern()));
		    result.append(lineSeparator);
		}
		return result.toString();
	    }

	    List itemOrderings(AgendaItem item) {
		return Lisp.NIL;

//  		// We have to turn the graph of the item's children
//  		// back into a list of orderings.  This is tedious
//  		// but straightforward.  Fortunately, we know there
//  		// are no links except to siblings.
//  		LList nodes = item.getChildren();
//  		HashMap nodeToNumber = new HashMap(nodes.size());
//  		LListCollector orderings = new LListCollector();

//  		// Give each subnode a number.
//  		int count = 1;
//  		for (Iterator n = nodes.iterator(); n.hasNext();) {
//  		    nodeToNumber.put(n.next(), new Long(count));
//  		    count++;
//  		}

//  		// Look at the direct successors of each subnode
//  		// and record a (subnode successor) ordering using
//  		// the corresponding node-numbers.
//  		for (Iterator n = nodes.iterator(); n.hasNext();) {
//  		    PNode node = (PNode)n.next();
//  		    for(Iterator p = node.postNodes.iterator(); p.hasNext();) {
//  			PNode post = (PNode)p.next();
//  			Long node_n = (Long)nodeToNumber.get(node);
//  			Long post_n = (Long)nodeToNumber.get(post);
//  			orderings.add(Lisp.list(node_n, post_n));
//  		    }
//  		}
//  		// Return list of orderings.
//  		return orderings.contents();
	    }

	}

    }

}

