/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue May 22 03:18:36 2007 by Jeff Dalton
 * Copyright: (c) 2001 - 2006, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.icore.*;
import ix.icore.domain.Refinement;
import ix.icore.process.ProcessModelManager;

import ix.ip2.log.ActivityHandledEvent;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.context.TypedContextValue;

/**
 * An agenda of activities
 */
public class ActivityAgenda extends Agenda {

    private TypedContextValue __openOtherReportsActivity =
	// Has a value when such an activity exists and its
	// status is not complete.
	new TypedContextValue(OtherReportsActivity.class, null);

    public ActivityAgenda(PanelController controller) {
	super(controller);
    }

    public void reset() {
	super.reset();
	__openOtherReportsActivity =
	    new TypedContextValue(OtherReportsActivity.class, null);
    }

    public void clear() {
        super.clear();
        setOpenOtherReportsActivity(null);
    }

    public AgendaItem makeItem(LList pattern) {
	return new ActivityItem(new Activity(pattern));
    }

    public AgendaItem makeItem(TaskItem activity) {
	Util.mustBe(Activity.class, activity);
	return new ActivityItem((Activity)activity);
    }

    public void addItem(AgendaItem item) {
	Ip2ModelManager mm = controller.getModelManager();
        mm.beginUndoableTransaction("Add activity");
        try {
            item.setPattern(mm.putVariablesInPattern(item.getPattern()));
            mm.addNode(item);
            item.computeStatus();
            super.addItem(item);
        }
        finally {
            mm.endUndoableTransaction("Add activity");
        }
    }

    public void addItemsBefore(AgendaItem at, List addList) {
	Ip2ModelManager mm = (Ip2ModelManager)controller.getModelManager();
	// /\/: For the outermost undoable transaction, see
	// ActivityInsertionEditor.
	mm.beginUndoableTransaction("Add activities before");
	try {
	    for (Iterator i = addList.iterator(); i.hasNext();) {
		AgendaItem item = (AgendaItem)i.next();
		item.setPattern(mm.putVariablesInPattern(item.getPattern()));
		item.computeStatus();
	    }
	    mm.addNodesBefore(at, addList);
	    super.addItemsBefore(at, addList);
	}
	finally {
	    mm.endUndoableTransaction("Add activities before");
	}
    }

    public void removeItem(AgendaItem item) {
	// /\/: Need to tell the model mamager to remove the node
	// and also any variables that are no longer needed?
	// We can at least try ...
	Ip2ModelManager mm = (Ip2ModelManager)controller.getModelManager();
	mm.removeNode(item);
	super.removeItem(item);
    }

    void logItemHandled(AgendaItem item, HandlerAction act) {

	// Log in the issue or activity.
	String description = act.getActionDescription();
	ActivityHandledEvent event = new ActivityHandledEvent();
	event.setAction(description);
	item.addHistoryEvent(event);

	// Maybe log in the agent's log as well.
	super.logItemHandled(item, act);

    }

    public boolean acceptReport(Report report) {
	// Pass 1 - skipping any OtherReportsActivity.
	for (Iterator i = items.iterator(); i.hasNext(); ) {
	    AgendaItem item = (AgendaItem)i.next();
	    if (item instanceof OtherReportsActivity)
		continue;
	    if (item.wantsReport(report)) {
		item.addReport(report);
		return true;
	    }
	}
	// Else use an OtherReportsActivity.
	if (openOtherReportsActivity() == null) {
            Ip2ModelManager mm = controller.getModelManager();
            mm.beginUndoableTransaction("Add other-reports activity");
            try {
                setOpenOtherReportsActivity
                    (new OtherReportsActivity("Note other reports"));
                addItem(openOtherReportsActivity());
            }
            finally {
                mm.endUndoableTransaction("Add other-reports activity");
            }
	}
	openOtherReportsActivity().addReport(report);
	return true;
    }

    protected OtherReportsActivity openOtherReportsActivity() {
	return (OtherReportsActivity)__openOtherReportsActivity.get();
    }

    protected void setOpenOtherReportsActivity(OtherReportsActivity act) {
        Debug.noteln("Setting open other reports to", act);
	__openOtherReportsActivity.set(act);
    }

    public class OtherReportsActivity extends ActivityItem {

	public OtherReportsActivity(String text) {
	    super(new Activity(text));
	    setStatus(Status.EXECUTING);
	}

	public boolean wantsReport(Report report) {
	    Debug.noteln("Other reports activity asked if it wants", report);
	    return getStatus() != Status.COMPLETE;
	}

	public boolean wantsActionsFrom(ItemHandler handler) {
	    return handler instanceof SimpleCompletionHandler;
	}

	public boolean actionCanBeTakenNow(HandlerAction act) {
	    return getStatus() == Status.POSSIBLE 	//  ? /\/
		|| (act instanceof HandlerAction.Manual
		    && getStatus() == Status.EXECUTING);
	}

	public void setStatus(Status status) {
	   if (status == Status.COMPLETE
	         && this == ActivityAgenda.this.openOtherReportsActivity())
	       ActivityAgenda.this.setOpenOtherReportsActivity(null);
	   super.setStatus(status);
	}

	public void computeStatus() {
	    // /\/: Do nothing.  If we let the PNode computeStatus()
	    // method run, a consistency check would complain that
	    // this item had status EXECUTING without having children.
	}

	public void addReport(Report report) {
	    super.addReport(report);
	    // Take priority from Report if it's higher.
	    if (getPriority().compareTo(report.getPriority()) < 0)
		setPriority(report.getPriority());
	}

	protected void setStatusBasedOn(Report report) {
	    // We don't want to become COMPLETE or IMPOSSIBLE
	    // because of a report, so we override the inherited
	    // method.
	}

    }

    public void expandItem(AgendaItem item, Refinement instructions) {
	// This is really a kind of issue handler.  /\/
	// Avoid null pointer exceptions in debugging code:  /\/
	Refinement r = instructions;
	if (r.getName() == null)
	    r.setName("temporary partial schema");
	// Note that we don't add the action to the issue.  /\/
 	// Instead we handle "directly".
	ExpandHandler exp =
	    (ExpandHandler)
	        Collect.findIf(handlers,
			       Fn.isInstanceOf(ExpandHandler.class));
	HandlerAction act = exp.makeManualExpandAction(item, r);
	item.addAction(act);	// ??? but we do add it ??? /\/
	// act.handle(item);
	handleItem(item, act);
	fireItemHandled(item, act); // shouldn't need this /\/
    }

}
