/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri Sep 12 15:03:39 2003 by Jeff Dalton
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 */

package ix.iface.util;

import java.text.SimpleDateFormat;
import java.util.Date;

import java.util.*;

import ix.icore.*;
import ix.icore.domain.Constraint;

import ix.ichat.*;

import ix.util.xml.XML;
import ix.util.lisp.*;
import ix.util.*;


/**
 * Class of static methods useful when reporting incoming messages etc.
 */
public class Reporting {

    static SimpleDateFormat dateFormat =
	new SimpleDateFormat("dd-MMM-yy HH:mm:ss");

    // We sometimes have to explicitly set the default timezone;
    // static { dateFormat.setTimeZone(TimeZone.getDefault()); }

    private Reporting() { }	// no instantiation

    public static String messageDescription(IPC.InputMessage message) {
	Object contents = message.getContents();
	return dateString() + " " + description(contents);
    }

    public static String dateString() {
	return dateFormat.format(new Date());
    }

    public static String description(Object obj) {
	if (obj instanceof Issue)
	    return issueDescription((Issue)obj);
	else if (obj instanceof Activity)
	    return activityDescription((Activity)obj);
	else if (obj instanceof Report)
	    return reportDescription((Report)obj);
	else if (obj instanceof Constraint)
	    return constraintDescription((Constraint)obj);
	else if (obj instanceof ChatMessage)
	    return chatMessageDescription((ChatMessage)obj);
	else
	    return obj.toString();
    }

    public static String issueDescription(Issue issue) {
	Name senderId = issue.getSenderId();
	Name ref = issue.getRef();
	YesNo reportBack = issue.getReportBack();
	return 
	    "Issue from " + senderId
	    + (ref == null ? "" : " ref=" + ref)
	    + (reportBack == null ? "" : " report-back=" + reportBack)
	    + " priority=" + issue.getPriority()
	    + ": " + Util.quote(patternDescription(issue.getPattern()));
    }

    public static String activityDescription(Activity activity) {
	Name senderId = activity.getSenderId();
	Name ref = activity.getRef();
	YesNo reportBack = activity.getReportBack();
	return
	    "Activity from " + senderId
	    + (ref == null ? "" : " ref=" + ref)
	    + (reportBack == null ? "" : " report-back=" + reportBack)
	    + " priority=" + activity.getPriority()
	    + ": " + Util.quote(patternDescription(activity.getPattern()));
    }

    public static String patternDescription(LList pattern) {
        return PatternParser.unparse(pattern);
    }

    public static String constraintDescription(Constraint constraint) {
	// /\/: Don't have a nice, general way to describe constraints,
	// so just use toString();
	Name senderId = constraint.getSenderId();
	return "Constraint from " + senderId + ": " + constraint;
    }

    public static String reportDescription(Report report) {
	Name senderId = report.getSenderId();
	Name ref = report.getRef();
	ReportType type = report.getReportType();
	return
	    "Report from " + senderId
	    + (ref == null ? "" : " ref=" + shortRef(ref))
	    + (type == null ? "" : " type=" + type)
	    + ": " + Util.quote(report.getText());
    }

    private static String shortRef(Name ref) {
	return Strings.afterLast("/", ref.toString());
    }

    public static String chatMessageDescription(ChatMessage message) {
	String lineSeparator = System.getProperty("line.separator");
	Name senderId = message.getSenderId();
	List lines = Strings.breakIntoLines(message.getText());
	StringBuffer result = new StringBuffer();

	result.append("Message from ")
              .append(senderId)
              .append(":");

	for (Iterator i = lines.iterator(); i.hasNext();) {
	    String line = (String)i.next();
	    result.append(lineSeparator)
	          .append("   ")
                  .append(line);
	}
	return result.toString();
    }

}
