/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Sep  6 20:05:50 2005 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.iface.util;

import javax.swing.*;
import javax.swing.table.*;
import javax.swing.event.TableModelEvent;

import java.awt.Toolkit;
import java.awt.Dimension;
import java.awt.event.*;

import java.util.*;

import ix.util.*;
import ix.util.lisp.*;

/**
 * Packages a JTable together with a table model for showing
 * a list of checkable items.
 */
public class CheckTable {

    protected ViewJTable table;
    protected ViewTableModel model;

    public CheckTable(String valColName, Object[] items, Object[] selected) {
	super();
	model = new ViewTableModel(valColName, items, selected);
	table = new ViewJTable(model);
    }

    public JTable getJTable() {
	return table;
    }

    public Object[] getSelectedValues() {
	return model.getSelectedValues().toArray();
    }

    public void clearSelections() {
	model.clearSelections();
    }

    protected class ViewJTable extends JTable {

	public ViewJTable(TableModel model) {
	    super(model);
	    getTableHeader().setReorderingAllowed(false);
	    setCellSelectionEnabled(false); // ? /\/
	    setPreferredColumnWidths();
	    setPreferredScrollableViewportSize(getReasonableSize());
	}

	protected void setPreferredColumnWidths() {
	    // /\/: Widths are times assumed number of pixels per char
	    getColumnModel().getColumn(model.KEY_COL)
		.setPreferredWidth(12);
	    getColumnModel().getColumn(model.KEY_COL)
		.setMaxWidth(12);
	    getColumnModel().getColumn(model.VAL_COL)
		.setPreferredWidth(40 * 10);
	}

	protected Dimension getReasonableSize() {
	    Dimension screen = Toolkit.getDefaultToolkit().getScreenSize();
	    Dimension preferred = getPreferredSize();
	    double width = Math.min(screen.getWidth() - 200,
				    preferred.getWidth());
	    double height = Math.min(screen.getHeight() - 200,
				     preferred.getHeight());
	    return new Dimension((int)width, (int)height);
	}

    }

    /** Mediates between the data and the JTable. */
    protected class ViewTableModel extends AbstractTableModel {

	final int KEY_COL = 0, VAL_COL = 1;
	final String[] columnName;
	final Boolean[] isSelected;
	final Object[] items;

	public ViewTableModel(String valColName, Object[] items,
			                         Object[] selected) {
	    super();
	    this.columnName = new String[]{"", valColName};
	    this.items = items;
	    this.isSelected = new Boolean[items.length];
	    Arrays.fill(isSelected, Boolean.FALSE);
	    if (selected != null) {
		List toSelect = Arrays.asList(selected);
		for (int i = 0; i < isSelected.length; i++) {
		    isSelected[i] = Boolean.valueOf
			              (toSelect.contains(items[i]));
		}
	    }
	}

	public String getColumnName(int col) {
	    return columnName[col];
	}

	public int getColumnCount() {
	    return columnName.length;
	}

	public int getRowCount() {
	    return items.length;
	}

	public Class getColumnClass(int c) {
	    return getValueAt(0, c).getClass();
	}

	public Object getValueAt(int row, int col) {
	    switch(col) {
	    case KEY_COL: return isSelected[row];
	    case VAL_COL: return items[row];
	    default:
		throw new ConsistencyException("Bogus column " + col);
	    }
	}

	public boolean isCellEditable(int row, int col) {
	    return col == KEY_COL;
	}

	public void setValueAt(Object value, int row, int col) {
	    Debug.expect(col == KEY_COL, "attempt to edit", new Integer(col));
	    isSelected[row] = (Boolean)value;
	}

	public List getSelectedValues() {
	    List result = new ArrayList(items.length);
	    for (int i = 0; i < isSelected.length; i++) {
		if (isSelected[i].booleanValue() == true)
		    result.add(items[i]);
	    }
	    return result;
	}

	public void clearSelections() {
	    Arrays.fill(isSelected, Boolean.FALSE);
	}

    }

}
