/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri Oct 21 23:47:16 2005 by Jeff Dalton
 * Copyright: (c) 2004, 2005, AIAI, University of Edinburgh
 */

package ix.iface.plan;

import java.io.*;
import java.util.*;

import ix.icore.*;
import ix.icore.plan.*;
import ix.icore.plan.inspect.*;
import ix.icore.domain.*;
import ix.icore.process.PNode;
import ix.icore.process.PNodeEnd;
import ix.iface.util.KeyValueTable;
import ix.iplan.IPlanOptionManager; // for ModelHolder /\/
import ix.ip2.Ip2ModelManager;	// /\/
import ix.util.*;
import ix.util.lisp.*;

/**
 * Writes a plain text description of a plan to a file.
 */
public class TextPlanWriter {

    protected File outputFile = null;
    protected Writer out = null;

    protected int indent = 0;
    protected int indentStep = 3;

    public TextPlanWriter(File f) {
	this.outputFile = f;
    }

    public TextPlanWriter(Writer w) {
	this.out = w;
    }

    public void writePlan(final Plan plan) throws IOException {
	if (out != null) {
	    outPlan(plan);
	    return;
	}
	out = new BufferedWriter(new FileWriter(outputFile));
	Util.run(new WithCleanup() {
	    public void body() throws IOException {
		outPlan(plan);
		out.flush();
	    }
	    public void cleanup() throws IOException {
		out.close();
	    }
	});
    }

    void outPlan(Plan plan) {
	outputTableView(plan);
	outputTimeline(plan);
    }

    void outputTableView(Plan plan) {
	if (plan.getPlanIssues() != null) outIssues(plan);
	if (plan.getPlanNodes() != null) outActivities(plan);
	if (plan.getPlanNodes() != null) outActivitiesInFull(plan);
	if (plan.getWorldState() != null) outWorldState(plan);
	if (plan.getAnnotations() != null) outAnnotations(plan);
    }

    /*
     * Issues and activities
     */

    void outIssues(Plan plan) {
	outTitle("Issues");
	PlanInspector pi = new PlanInspector(plan).setIsAutoRecursive(false);
	pi.walkIssues(new TIVisitor(pi));
    }

    void outActivities(Plan plan) {
	outTitle("Activities");
	PlanInspector pi = new PlanInspector(plan).setIsAutoRecursive(false);
	pi.walkActivities(new TIVisitor(pi));
    }

    class TIVisitor implements TaskItemVisitor {
	PlanInspector inspector;
	TIVisitor(PlanInspector i) {
	    inspector = i;
	}
	public void visit(TaskItem item, TaskItem parent, List children) {
	    outTaskItem(item);
	    if (!children.isEmpty()) {
		try {
		    indent += indentStep;
		    inspector.walkTaskItemChildren(item, this);
		}
		finally {
		    indent -= indentStep;
		}
	    }
	}
	String patternSentence(TaskItem ti) {
	    LList pattern = (LList)removePlanVars(ti.getPattern());
	    return PatternParser.unparse(pattern);
	}
	void outTaskItem(TaskItem ti) {
	    outln(patternSentence(ti));
	}
    }

    public static Object removePlanVars(Object obj) {
	return new DeepCopier() {
	    public Object copy(Object obj) {
		if (obj instanceof PlanVariable) {
		    PlanVariable var = (PlanVariable)obj;
		    return Symbol.intern(varName(var));
		}
		else
		    return super.copy(obj);
	    }
	    String varName(PlanVariable var) {
		String id = Strings.afterFirst("var-", var.getId().toString());
		return var.getName() + "/" + id;
	    }
	}.copy(obj);
    }

    /*
     * Activities described in full
     */

    void outActivitiesInFull(Plan plan) {
	outTitle("Activities in Full");
	PlanInspector pi = new PlanInspector(plan).setIsAutoRecursive(false);
	pi.walkActivities(new TIVisitor(pi) {
	    void outTaskItem(TaskItem ti) {
		AbstractPlanItem pi = inspector.toPlanItem(ti);
		outln(activityId(pi) + " " + patternSentence(ti));
		Map m = ti.getAnnotations();
		if (m != null)
		    outMap(m);
		newline();
	    }
	});
    }

    String activityId(AbstractPlanItem pi) {
	// /\/: In the Plan, numbering starts with 0, hence node-0,
	// node-0-0-0-1, etc.  Rather than change that at this point,
	// we increment each number in the id by 1.
	String id = pi.getId().toString();
	String nodeNumber = Strings.afterFirst("node-", id);
	List numbers = Strings.breakAt("-", nodeNumber);
	List incremented = new LinkedList();
	for (Iterator i = numbers.iterator(); i.hasNext();) {
	    Long n = (Long)Lisp.readFromString((String)i.next());
	    incremented.add("" + (n.longValue() + 1));
	}
	return Strings.joinWith("-", incremented);
    }

    /*
     * World state
     */

    void outWorldState(Plan plan) {
	Map state = new TreeMap(new KeyValueTable.PatternObjectComparator());
	state.putAll(PatternAssignment.assignmentsToMap(plan.getWorldState()));
	outTitle("World State");
	for (Iterator i = state.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    // We know there are already no variables.
	    indent(indentStep);
	    out(PatternParser.unparse((LList)e.getKey()));
	    out(" = ");
	    out(e.getValue());
	    newline();
	}
    }

    /*
     * Annotations
     */

    void outAnnotations(Plan plan) {
	outTitle("Annotations");
	outMap(plan.getAnnotations());
    }

    void outMap(Map m) {
	for (Iterator i = m.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    Object k = e.getKey();
	    Object v = e.getValue();
	    if (Lisp.isFullyPrintable(k) && Lisp.isFullyPrintable(v)) {
		indent();
		out("   " + Lisp.printToString(k) + " = ");
		if (isTextBlock(v)) {
		    newline();
		    outTextBlock(3, (String)v);
		}
		else {
		    out(Lisp.printToString(v));
		    newline();
		}
	    }
	}
    }

    /*
     * Timeline
     */

    void outputTimeline(Plan plan) {
	outTitle("Timeline");
	try {
	    indent += indentStep;
	    do_outputTimeline(plan);
	}
	finally {
	    indent -= indentStep;
	}
    }

    void do_outputTimeline(Plan plan) {
	// We have to put the plan back into a model.
	IPlanOptionManager.ModelHolder m =
	    IPlanOptionManager.ModelHolder.newInstance();
	Ip2ModelManager modelManager = (Ip2ModelManager)m.getModelManager();
	TopologicalSorter sorter = new TopologicalSorter() {
	    protected Collection getChildren(Object nodeEnd) {
		// Return node-ends linked directly after this one.
		return ((PNodeEnd)nodeEnd).getSuccessors();
	    }
	};
	m.loadPlan(plan);
	List allNodeEnds = modelManager.getNodeEnds();
	List sorted = sorter.sort(allNodeEnds);
	for (ListIterator i = sorted.listIterator(); i.hasNext();) {
	    PNodeEnd ne = (PNodeEnd)i.next();
	    if (i.hasNext()) {
		if (((PNodeEnd)i.next()).getNode() == ne.getNode()) {
		    // The next node-end is the other end of of ne
		    outln(patternSentence(ne.getNode()));
		    continue;
		}
		else {
		    // Pretend we hadn't looked ahead.
		    i.previous();
		}
	    }
	    outln(ne.getEnd() + " " + patternSentence(ne.getNode()));
	}
    }

    String patternSentence(PNode n) {
	LList pattern = (LList)Variable.removeVars(n.getPattern());
	return PatternParser.unparse(pattern);
    }

    /*
     * Utilities
     */

    void out(String s) {
	try { out.write(s); }
	catch (IOException e) {
	    throw new RethrownException(e);
	}
    }

    void out(Object o) {
	out(Lisp.printToString(o));
    }

    void indent() {
	out(Strings.repeat(indent, " "));
    }

    void indent(int n) {
	out(Strings.repeat(n, " "));
    }

    void newline() {
	out("\n");
    }

    void outln(String line) {
	indent();
	out(line);
	newline();
    }

    void outTitle(String text) {
	newline();
	out(text); newline();
	out(Strings.repeat(text.length(), "-")); newline();
	newline();
    }

    boolean isTextBlock(Object o) {
	return o instanceof String
	    && ((String)o).indexOf("\n") >= 0;
    }

    void outTextBlock(int extraIndent, String block) {
	try {
	    indent += extraIndent;
	    String[] lines = Strings.toArray(Strings.breakIntoLines(block));
	    lines[0] = "\"" + lines[0];
	    lines[lines.length-1] += "\"";
	    for (int i = 0; i < lines.length; i++) {
		indent(); 
		outln(lines[i]);
	    }
	}
	finally {
	    indent -= extraIndent;
	}
    }

}
