/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Nov  8 23:43:49 2004 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.iface.plan;

import java.io.*;
import java.util.*;

import ix.icore.*;
import ix.icore.plan.*;
import ix.icore.plan.inspect.*;
import ix.icore.domain.*;
import ix.iface.util.KeyValueTable;
import ix.util.*;
import ix.util.lisp.LList;

/**
 * Writes an HTML description of a plan to a file.
 */
public class HtmlPlanWriter {

    // /\/: This is a very simple version which directly outputs
    // HTML as text, with some line breaks but no indentation.
    // A more sophisticated version would use JDOM to construct
    // a representation of an XHTML description and then output
    // that as HTML.

    File outputFile;
    Writer out;
    LinkedList tagContext;	// a stack

    public HtmlPlanWriter(File f) {
	this.outputFile = f;
    }

    public void writePlan(Plan plan) throws IOException {
	out = new BufferedWriter(new FileWriter(outputFile));
	tagContext = new LinkedList();
	outTag("html"); outln("");
	outTag("body", "bgcolor=\"#ffffff\""); outln("");
	outPlan(plan); outln("");
	closeTag("body"); outln("");
	closeTag("html"); outln("");
	out.flush();
    }

    void outPlan(Plan plan) {
	if (plan.getPlanIssues() != null) outIssues(plan);
	if (plan.getPlanNodes() != null) outActivities(plan);
	if (plan.getWorldState() != null) outWorldState(plan);
	if (plan.getAnnotations() != null) outAnnotations(plan);
    }

    /*
     * Issues and activities
     */

    void outIssues(Plan plan) {
	tagln("h2", "Issues");
	outTag("ul");
	outln("");
	PlanInspector pi = new PlanInspector(plan).setIsAutoRecursive(false);
	pi.walkIssues(new TIVisitor(pi));
	closeTag("ul");
	outln("");
    }

    void outActivities(Plan plan) {
	tagln("h2", "Activities");
	outTag("ul");
	outln("");
	PlanInspector pi = new PlanInspector(plan).setIsAutoRecursive(false);
	pi.walkActivities(new TIVisitor(pi));
	closeTag("ul");
	outln("");
    }

    class TIVisitor implements TaskItemVisitor {
	PlanInspector inspector;
	TIVisitor(PlanInspector i) {
	    inspector = i;
	}
	public void visit(TaskItem item, TaskItem parent, List children) {
	    outTaskItem(item);
	    if (!children.isEmpty()) {
		outTag("ul");
		inspector.walkTaskItemChildren(item, this);
		closeTag("ul");
		outln("");
	    }
	}
    }

    void outTaskItem(TaskItem ti) {
	LList pattern = (LList)Variable.removeVars(ti.getPattern());
	String sentence = PatternParser.unparse(pattern);
	outTag("li");
	out(sentence);
	closeTag("li");
	outln("");
    }

    /*
     * World state
     */

    void outWorldState(Plan plan) {
	Map state = new TreeMap(new KeyValueTable.PatternObjectComparator());
	state.putAll(PatternAssignment.assignmentsToMap(plan.getWorldState()));
	tagln("h2", "State");
	for (Iterator i = state.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    // We know there are already no variables.
	    out(PatternParser.unparse((LList)e.getKey()));
	    out(" = ");
	    out(e.getValue());
	    outln("<br />");
	}
    }

    /*
     * Annotations
     */

    void outAnnotations(Plan plan) {
	tagln("h2", "Annotations");
    }

    /*
     * Utilities
     */

    void outTag(String tag) {
	out("<" + tag + ">");
	tagContext.addFirst(tag);
    }

    void outTag(String tag, String attributes) {
	out("<" + tag + " " + attributes + ">");
	tagContext.addFirst(tag);
    }

    void closeTag(String tag) {
	Debug.expectEquals(tagContext.removeFirst(), tag,
			   "Closing the wrong tag");
	out("</" + tag + ">");
    }

    void tagln(String tag, String contents) {
	outTag(tag); out(contents); closeTag(tag); outln("");
    }

    void out(String s) {
	try { out.write(s); }
	catch (IOException e) {
	    throw new RethrownException(e);
	}
    }

    void out(Object o) {
	out(o.toString());
    }

    void outln(String line) {
	out(line);
	out("\n");
    }

}
