/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sun Nov  4 14:09:08 2007 by Jeff Dalton
 * Copyright: (c) 2002, 2005, 2007, AIAI, University of Edinburgh
 */

package ix.iface.domain;

import java.io.*;
import java.util.*;

import ix.icore.domain.*;
import ix.icore.*;
import ix.util.*;
import ix.util.lisp.*;

/**
 * A writer for domains described in LTF syntax.  This is written in
 * a fairly straightforward recursive-descent style and hence has
 * minimal documentation.  It is slightly more complex than strictly
 * necessary, so that subclass TF_Writer can be simpler.
 *
 * @see LTF_Parser
 * @see TF_Writer
 */
public class LTF_Writer extends DomainWriter {

    File domainName;		// N.B. may be null
    Writer out;

    public LTF_Writer(File domainName) {
	this.domainName = domainName;
    }

    public LTF_Writer(Writer out) {
	this.out = out;
    }

    public LTF_Writer() {
    }

    //
    // Writing a domain description
    //

    public void writeDomain(final Domain domain) throws IOException {
	// Get a stream to write to, if one's needed.
	if (out == null)
	    out = new BufferedWriter(new FileWriter(domainName));
	Util.run(new WithCleanup() {
	    public void body() throws IOException {
		outDomain(domain);
		out.flush();
	    }
	    public void cleanup() throws IOException {
		out.close();
	    }
	});
    }

    public void writeDomain(Domain domain, Writer w) throws IOException {
	out = w;
	outDomain(domain);
	out.flush();
    }

    public void writeRefinement(Refinement r, Writer w) throws IOException {
	out = w;
	outRefinement(r);
	out.flush();
    }

    void outDomain(Domain domain) {
	// Output domain header
	outDomainHeader(domain);
	// Output object class definitions
	for (Iterator i = Collect.iterator(domain.getObjectClasses())
		 ; i.hasNext();) {
	    ObjectClass c = (ObjectClass)i.next();
	    outObjectClass(c);
	    outln("");
	}
	// Output refinement definitions
	for (Iterator i = domain.getRefinements().iterator(); i.hasNext();) {
	    Refinement r = (Refinement)i.next();
	    outRefinement(r);
	    if (i.hasNext()) outln("");
	}
	// Output annotations
	Annotations ann = domain.getAnnotations();
	if (ann != null) {
	    for (Iterator i = ann.entrySet().iterator() ; i.hasNext();) {
		Map.Entry e = (Map.Entry)i.next();
		outDomainAnnotation(e.getKey(), e.getValue());
	    }
	}
    }

    void outDomainHeader(Domain domain) {
	// For now, all that's in the header is the domain's name if it has one.
	String name = domain.getName();
	if (name != null) {
	    outln("(domain (name " + Strings.quote(name) + "))");
	    outln("");
	}
    }

    void outDomainAnnotation(Object key, Object value) {
	// This is defined before outRefinement because it is simpler.  /\/
	if (!(isFullyPrintable(key) && isFullyPrintable(value))) {
	    outln("");
	    outln("; Cannot output annotation with key " + key);
	    return;
	}
	out(0, "(annotations");
	out(2, "(");
	outObject(3, key);	// 3 ??? /\/  doesn't really work
	out(" = ");
	outAnnotationValue(value);
	out(")");
	outln(")");
    }

    void outAnnotationValue(Object v) {
	outObject(4, v);
    }

    void outObject(int indent, Object obj) {
	// /\/: What we really need here is general object output
	// using a ClassSyntax.  Now it's all special cases.
	// /\/: The indent is needed only if we start a new line.
	if (obj instanceof Map)
	    outMap(indent, (Map)obj);
	else if (obj instanceof String)
	    // { out("\""); out(obj); out("\""); }
	    out(Lisp.quotedAndEscaped('"', (String)obj));
	else if (obj instanceof Name) {
	    // /\/: Output names as symbols
	    // /\/: This happens when outputting duration constraints.
	    outObject(indent, Symbol.intern(obj.toString()));
	}
	else
	    out(Lisp.printToString(obj)); // will often work /\/
    }

    void outMap(int indent, Map m) {
	// /\/: Works properly only for keys and values that don't
	// need special output or are strings.
	out(indent, "(Map");
	for (Iterator i = m.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    out(indent + 2, "(");
	    outObject(indent + 3, e.getKey());
	    out(" = ");
	    outObject(indent + 4, e.getValue());
	    out(")");
	}
	out(")");
    }

    void outObjectClass(ObjectClass c) {
	out("(object-class ");
	out(safeName(c.getName()));
	out(" ");
	out(LList.newLList(c.getSuperClassNames()));
	for (Iterator i = c.getObjectProperties().iterator(); i.hasNext();) {
	    ObjectProperty p = (ObjectProperty)i.next();
	    out(2, "(");
	    out(p.getName());
	    if (p.getValueClassName() != null) {
		out(" ");
		out(p.getValueClassName());
	    }
	    if (p.getSyntax() != null) {
		out(" :syntax ");
		out(p.getSyntax());
	    }
	    out(")");
	}
	outln(")");
    }

    void outRefinement(Refinement r) {
	outRefinementHeader(r);

	List varDcls = r.getVariableDeclarations();
	List nodes = r.getNodes();
	List orderings = r.getOrderings();
	List constraints = r.getConstraints();
	List issues = r.getIssues();
	String comments = r.getComments();
	Annotations annotations = r.getAnnotations();

	// Comments is an annotation, but a subclass that can't
	// handle annotations might still want to output comments. /\/

	if (varDcls != null) outVarDcls(varDcls);
	if (nodes != null) outNodes(nodes);
	if (orderings != null) outOrderings(orderings);
	if (constraints != null) outConstraints(constraints);
	if (issues != null) outIssues(issues);
	if (comments != null && !comments.equals("")) outComments(comments);
	// if (annotations != null) outAnnotations(annotations);
	if (usefulAnnotationsP(r, annotations)) outAnnotations(annotations);

	outRefinementClose();
    }

    boolean usefulAnnotationsP(Refinement r, Annotations a) {
	// /\/: Hack to avoid useless comments = ""
	// annotations that are somehow produced by I-DE.
	// Handles only the case where that's the only annotation.
	return a != null
	    && !a.isEmpty()
	    && !(a.size() == 1 && a.get(COMMENTS) != null);
    }

    private static final Symbol COMMENTS = Symbol.intern("comments"); // /\/

    void outRefinementHeader(Refinement r) {
	// Output: (refinement name pattern
	out("(refinement ");
	out(safeName(r.getName()));
	out(" ");
	out(r.getPattern());
    }

    void outRefinementClose() {
	outln(")");
    }

    private String safeName(String name) {
	// Since a Refinement's name is a String, it might contain
	// characters that prevent it from being written and reread
	// correctly as a Symbol; still, we'd like to use a Symbol
	// in simple cases if we can.  Hence this method.
	// /\/: This is not perfect but will do for now.
	// /\/: We could always use a Symbol and hope the |...|
	// notation handles it correctly.
	return isSafeAsSymbolName(name)
	    ? name
	    : Lisp.quotedAndEscaped('"', name);
    }

    private boolean isSafeAsSymbolName(String name) {
	// /\/: Very imperfect.  See also the way the Symbol
	// class determines whether |...| notation is needed.
	char start = name.charAt(0);
	if (!(Character.isJavaIdentifierStart(start)))
	    return false;
	int i = 0, len = name.length();
	while (i < len) {
	    char c = name.charAt(i);
	    if (!(Character.isJavaIdentifierPart(c)
		  || c == '-'))
		return false;
	    i++;
	}
	return true;
    }

    void outClauseStart(String name) {
	out(2, "("); out(name);
    }

    void outClauseFinish() {
	out(")");
    }

    void outBigSeparator() {
	// Called to separate major elements of a clause when they
	// should be on separate lines.  Here it's a no-op because
	// it's always handled by indentation of next line
	// when out(int, whatever) is called.
    }

    void outSmallSeparator() {
	// Called to separate major elements of a clause when they
	// should be on the same line.
	out(" ");
    }

    void outVarDcls(List varDcls) {
	outClauseStart("variables");
	for (Iterator i = varDcls.iterator(); i.hasNext();) {
	    VariableDeclaration dcl = (VariableDeclaration)i.next();
	    outSmallSeparator();
	    out(dcl.getName());
	}
	outClauseFinish();
    }

    void outNodes(List nodes) {
	outClauseStart("nodes");
	for (Iterator i = nodes.iterator(); i.hasNext();) {
	    NodeSpec spec = (NodeSpec)i.next();
	    outNode(spec);
	    if (i.hasNext())
		outBigSeparator();
	}
	outClauseFinish();
    }

    void outNode(NodeSpec spec) {
	out(4, "(");
	out(spec.getId());
	out(" ");
	out(spec.getPattern());
	out(")");
    }

    void outOrderings(List orderings) {
	outClauseStart("orderings");
	out(4, "");
	for (Iterator i = orderings.iterator(); i.hasNext();) {
	    Ordering ord = (Ordering)i.next();
	    outOrdering(ord);
	    if (i.hasNext()) outSmallSeparator();
	}
	outClauseFinish();
    }

    void outOrdering(Ordering ord) {
	NodeEndRef from = ord.getFrom();
	NodeEndRef to = ord.getTo();
	out("(");
	outNodeEndRef(from, End.END);
	out(" ");
	outNodeEndRef(to, End.BEGIN);
	out(")");
    }

    void outNodeEndRef(NodeEndRef ref, End defaultEnd) {
	End end = ref.getEnd();
	if (end != defaultEnd)
	    out(end == End.BEGIN ? "b/" : "e/");
	out(ref.getNode());
    }

    void outConstraints(List constraints) {
	outClauseStart("constraints");
	for (Iterator i = constraints.iterator(); i.hasNext();) {
	    Constraint c = (Constraint)i.next();
	    outConstraint(c);
	    if (i.hasNext()) outBigSeparator();
	}
	outClauseFinish();
    }

    void outConstraint(Constraint c) {
	out(4, "(");
	out(c.getType());
	if (c.getRelation() != null) {
	    out(" ");
	    out(c.getRelation());
	}
	for (Iterator i = c.getParameters().iterator(); i.hasNext();) {
	    out(" ");
	    outConstraintParameter(i.next());
	}
	out(")");
    }

    void outConstraintParameter(Object p) {
	if (p instanceof PatternAssignment)
	    outPV((PatternAssignment)p);
	else if (p instanceof TimeWindow)
	    outTimeWindow((TimeWindow)p);
	else
	    // Debug.expect(false, "Unexpected constraint parameter", p);
	    outObject(4+2, p); // hope this works /\/
    }

    void outPV(PatternAssignment pv) {
	out(pv.getPattern());
	out(" = ");
	out(Lisp.printToString(pv.getValue()));	// /\/
    }

    void outTimeWindow(TimeWindow w) {
	// /\/: We're cheating a bit here by including the " = ".
	out("= " + w.getMin().toISOString()
	          + " .. " + w.getMax().toISOString());
    }

    void outIssues(List issues) {
	outClauseStart("issues");
	for (Iterator i = issues.iterator(); i.hasNext();) {
	    Issue issue = (Issue)i.next();
	    outIssue(issue);
	    if (i.hasNext()) outBigSeparator();
	}
	outClauseFinish();
    }

    void outIssue(Issue issue) {
	out(4, Lisp.list(Symbol.intern("issue"),
			 issue.getPattern()));
    }

    void outComments(String comments) {
	// We don't have to do anything, because the comments
	// will be handled as an annotation.
    /*
	// /\/: Comments clause has to be commented-out for now
	List lines = Strings.breakIntoLines(comments);
	out(2, ";(comments");
	out(2, ";  \"");
	for (Iterator i = lines.iterator(); i.hasNext();) {
	    String line = (String)i.next();
	    out(line);
	    if (i.hasNext()) out(2, ";");
	}
	out("\"");
	out(")");
	outln("");
    */
    }

    void outAnnotations(Annotations annotations) {
	// Outputs the annotations of a refinement.
	outClauseStart("annotations");
	for (Iterator i = annotations.entrySet().iterator() 
		 ; i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    Object k = e.getKey(), v = e.getValue();
	    if (k != null) {
		if (isFullyPrintable(k) && isFullyPrintable(v)) {
		    out(4, "(");
		    outObject(5, k); // 5 ??? /\/
		    out(" = ");
		    outObject(6, v);
		    out(")");
		}
		else {
		    out(4, "; Cannot output annotation with key " + k);
		    outln("");
		}
	    }
	}
	outClauseFinish();
    }

    // Output utilities

    boolean isFullyPrintable(Object a) {
	// Like Lisp.isFullyPrintable(Object) except that it
	// allows maps.
	return Lisp.isFullyPrintable(a)
	    || (a instanceof Map && isFullyPrintableMap((Map)a));
    }

    private boolean isFullyPrintableMap(Map m) {
	for (Iterator i = m.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    if (!isFullyPrintable(e.getKey()) ||
		!isFullyPrintable(e.getValue()))
	      return false;
	}
	return true;
    }

    void out(String s) {
	try { out.write(s); }
	catch (IOException e) {
	    throw new RethrownException(e);
	}
    }

    void out(Object o) {
	out(toString(o));
    }

    void out(int indent, String s) {
	outln("");
	out(Strings.repeat(indent, " "));
	out(s);
    }

    void out(int indent, Object o) {
	out(indent, toString(o));
    }

    void outln(String line) {
	out(line);
	out("\n");
    }

    String toString(Object o) {
	return o.toString();	// N.B. may be different in subclass
    }

}
