/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Sep 18 02:42:44 2007 by Jeff Dalton
 * Copyright: (c) 2000 - 2007, AIAI, University of Edinburgh
 */

package ix.iface.domain;

import java.io.*;
import java.net.URL;
import java.net.MalformedURLException;
import java.util.*;

import ix.icore.*;
import ix.icore.domain.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;
import ix.util.xml.XML;

/**
 * A parser for domains described in a Lisp-like syntax. <p>
 *
 * The parser parses a file of refinement definitions to populate 
 * a Domain. <p>
 *
 * In the syntax definitions below, all-upper-case names are literals
 * and lower- and mixed-case names are syntactic variables.  Literals
 * should actually be written in lower case in domain definitions.
 *
 * <pre>
 *    domain ::= {header | refinement | object-class | annotations | include}*
 *
 *    header ::= (DOMAIN domain-property*)
 *
 *    domain-property ::= (NAME string)
 *
 *    include ::= (INCLUDE file-name-or-URL-as-string)
 *
 *    refinement ::= (REFINEMENT name pattern clause*)
 *
 *    name ::= symbol | string
 *    pattern ::= (item*)
 *    item ::= number | symbol | string | pattern | variable
 *    variable ::= ?symbol
 *    clause ::= (VARIABLES variable-declaration*)
 *            | (ISSUES issue*)
 *            | (NODES node*)
 *            | (ORDERINGS ordering*)
 *            | (CONSTRAINTS constraint*)
 *            | annotations
 *    variable-declaration = variable
 *    issue ::= (ISSUE pattern)
 *    node ::= (node-id pattern)
 *    ordering ::= ({node-end-ref | (node-end-ref*)}*)
 *    constraint ::= (WORLD-STATE CONDITION pattern = value)
 *                |  (WORLD-STATE EFFECT pattern = value)
 *                |  (COMPUTE [MULTIPLE-ANSWER] pattern = value)
 *                |  (ADVICE EXPANSION-REFINEMENT verb (symbol*))
 *                |  (TEMPORAL DURATION SELF = min-duration .. max-duration)
 *                |  (RESOURCE operation pattern = value)
 *                |  other-constraint
 *    other-constraint ::= (type subtype pattern = value)
 *    node-id ::= number | symbol | string
 *    node-end-ref ::= node-id | B/node-id | E/node-id
 *    value ::= item
 *    verb, operation, type, subtype ::= symbol
 *    min-duration, max-duration ::= duration
 *    <i>A duration has ISO 8601 syntax for days, hours, minutes, seconds
 *          and milliseconds.</i>
 *
 *    object-class ::= (OBJECT-CLASS class-name supers object-property*)
 *    supers = (class-name*)
 *    object-property ::= (property-name [class-name] [:SYNTAX value-syntax])
 *    class-name ::= symbol | string
 *    property-name ::= symbol | string
 *    value-syntax ::= NUMBER | SYMBOL | STRING | LIST | OBJECT | DEFAULT
 *
 *    annotations :: = (ANNOTATIONS annotation*)
 *    annotation ::= (aKey = aValue)
 *    map ::= (MAP map-entry*)
 *    map-entry ::= (aKey = aValue)
 *
 *    <i>aKey and aValue can be anything the Lisp reader will accept.
 *    aValue can also be a map.  The word map must be written
 *    "</i>Map<i>".</i>
 * </pre>
 *
 * @see Domain
 * @see Duration
 * @see #expandAllOrderings(LList)
 */

public class LTF_Parser extends DomainParser implements LTF_Symbols {

    protected String sourceName;
    protected LispReader lin;

    protected Object lastRead = "Start of file";

    protected boolean forceNumericNodeIds =
	Parameters.getBoolean("force-numeric-node-ids", false);

    protected boolean badAnnotationsAreErrors = false;

    protected Set badAnnotations;

    protected Set otherConstraints;

    public LTF_Parser(String filename) throws FileNotFoundException {
	this.sourceName = filename;
	this.lin = new LispFileReader(filename);
	addConstraintParsers();
    }

    public LTF_Parser(File file) throws FileNotFoundException {
	this(file.getPath());
    }

    public LTF_Parser(URL url) throws IOException {
	this.sourceName = url.toString();
	Reader reader = Util.openURLReader(url);
	this.lin = new LispReader(new BufferedReader(reader));
	addConstraintParsers();
    }

    public LTF_Parser() {
	// Used e.g. if someone just wants to call some methods,
	// such as the constraint parser.
	addConstraintParsers();
    }

    public Domain readDomain() {
	return readDomain(new Domain());
    }

    public Domain readDomain(final Domain dom) {
	Util.run(new WithCleanup() {
	    public void body() {
		do_readDomain(dom);
	    }
	    public void cleanup() throws IOException {
		lin.close();
	    }
	});
	return dom;
    }

    protected void do_readDomain(Domain dom) {
	Debug.noteln("Reading definitions from", sourceName);
	badAnnotations = new TreeSet(); // list of keys as strings
	otherConstraints = new TreeSet();
	try {
	    Object item;
	    while ((item = lin.readObject()) != Lisp.EOF) {
		if (!(item instanceof Cons)) {
		    throw syntaxError("Found " + item +
				      " when expecting a definition");
		}
		LList def = theLList(item);
		lastRead = item;
		Symbol defType = (Symbol)def.elementAt(0);
		if (defType == S_DOMAIN) {
		    processDomainHeader(dom, def);
		}
		else if (defType == S_REFINEMENT) {
		    Refinement r = makeRefinement(def);
		    dom.addRefinement(r);
		}
		else if (defType == S_OBJECT_CLASS) {
		    ObjectClass c = makeObjectClass(def);
		    dom.addObjectClass(c);
		}
		else if (defType == S_ANNOTATIONS) {
		    dom.takeAnnotations(makeAnnotations(def));
		}
		else if (defType == S_INCLUDE) {
		    doInclude((Cons)item);
		}
		else {
		    throw syntaxError("Illegal definition type " + defType);
		}
	    }
	    if (!otherConstraints.isEmpty())
		Debug.warn("Constraints of unfamiliar types: " +
			   otherConstraints);
	    if (!badAnnotations.isEmpty())
		Debug.warn("Some annotations were not (key = value);" +
			   " the keys were: " +
			   Strings.conjunction(badAnnotations));
	}
        catch (SyntaxException e) {
	    Debug.noteln("Rethrowing syntax exception.");
	    throw e;
	}
	catch (LispReadException e) {
	    Debug.noteException(e, false);
	    throw syntaxError(Debug.describeException(e));
	}
        catch (Exception e) {
	    Debug.noteException(e);
	    throw syntaxError(Debug.describeException(e));
	}
	// /\/: Shouldn't check consistency so soon
	// dom.checkConsistency();
    }

    protected SyntaxException syntaxError(String message) {
	return new SyntaxException
	    (message
	     + (lin instanceof IncludeReader
		? " while parsing included " + Util.quote(sourceName)
		: "")
	     + " in or after " + lastRead);
    }

    protected static Cons theCons(Object o) {
	return (Cons)Util.mustBe(Cons.class, o);
    }
    protected static LList theLList(Object o) {
	return (LList)Util.mustBe(LList.class, o);
    }
    protected static Symbol theSymbol(Object o) {
	return (Symbol)Util.mustBe(Symbol.class, o);
    }
    protected static String theString(Object o) {
	return (String)Util.mustBe(String.class, o);
    }
    protected static String theName(Object o) {
	// Doesn't give quite the right error message /\/
	if (o instanceof String) return (String)o;
	else return theSymbol(o).toString();
    }
    protected static Name theNodeId(Object o) {
	if (o instanceof Number || o instanceof Symbol || o instanceof String)
	    return Name.valueOf(o);
	else
	    throw new ClassCastException
		("Attempt to use " + Util.aClass(o.getClass()) + " " +
		 o + " as a node-id.");
    }
    protected static NodeEndRef theNodeEndRef(Name nodeId, End defaultEnd) {
	String chars = nodeId.toString();
	if (chars.startsWith("b/"))
	    return new NodeEndRef(End.BEGIN, theNodeId(chars.substring(2)));
	else if (chars.startsWith("e/"))
	    return new NodeEndRef(End.END, theNodeId(chars.substring(2)));
	else
	    return new NodeEndRef(defaultEnd, nodeId);
    }

    /*
     * Domain header
     */

    void processDomainHeader(Domain dom, LList header) {
	Debug.expectSame(S_DOMAIN, header.get(0));
	for (LList clauses = header.cdr(); !clauses.isEmpty()
		 ; clauses = clauses.cdr()) {
	    LList clause = theLList(clauses.car());
	    Symbol property = theSymbol(clause.car());
	    if (property == S_NAME) {
		String name = theString(clause.get(1));
		dom.setName(name);
	    }
	    else
		throw syntaxError("Invalid domain header clause " + clause);
	}

    }

    /*
     * Refinements
     */

    public Refinement makeRefinement(LList def) {
	Debug.expectSame(S_REFINEMENT, def.get(0));

	Refinement r = new Refinement();
	r.setName(theName(def.get(1)));
	r.setPattern(theLList(def.get(2)));
	for (LList body = def.drop(3); !body.isEmpty(); body = body.cdr()) {
	    Cons clause = theCons(body.car());
	    Symbol name = theSymbol(clause.car());
	    LList items = clause.cdr();
	    if (name == S_VARIABLES)
		r.setVariableDeclarations(makeVarDcls(items));
	    else if (name == S_NODES)
		r.setNodes(makeNodes(items));
	    else if (name == S_ORDERINGS)
		r.setOrderings(makeOrderings(items));
	    else if (name == S_CONSTRAINTS)
		r.setConstraints(makeConstraints(items));
	    else if (name == S_ISSUES)
		r.setIssues(makeIssues(items));
	    else if (name == S_ANNOTATIONS)
		r.takeAnnotations(makeAnnotations(clause));
	    else
		throw syntaxError("Invalid clause " + clause);
	}
	return forceNumericNodeIds ? useNumericNodeIds(r) : r;
    }

    public ListOfVariableDeclaration makeVarDcls(LList items) {
	ListOfVariableDeclaration result =
	    new LinkedListOfVariableDeclaration();
	for (Iterator i = items.iterator(); i.hasNext();) {
	    ItemVar name = (ItemVar)Util.mustBe(ItemVar.class, i.next());
	    result.add(new VariableDeclaration(name));
	}
	return result;
    }

    public ListOfNodeSpec makeNodes(LList items) {
	ListOfNodeSpec result = new LinkedListOfNodeSpec();
	for (Iterator i = items.iterator(); i.hasNext();) {
	    LList pair = theCons(i.next());
	    Name id = Name.valueOf(pair.get(0));
	    LList pattern = theLList(pair.get(1));
	    result.add(new NodeSpec(id, pattern));
	}
	return result;
    }

    public ListOfOrdering makeOrderings(LList items) {
	ListOfOrdering result = new LinkedListOfOrdering();
	for (LList ords = expandAllOrderings(items);
	     !ords.isEmpty(); ords = ords.cdr()) {
	    Cons pair = theCons(ords.car());
	    Name from = theNodeId(pair.get(0));
	    Name to = theNodeId(pair.get(1));
	    result.add(new Ordering(theNodeEndRef(from, End.END), 
				    theNodeEndRef(to, End.BEGIN)));
	}
	return result;
    }

    public ListOfConstraint makeConstraints(LList items) {
	ListOfConstraint result = new LinkedListOfConstraint();
	for (Iterator i = items.iterator(); i.hasNext();) {
	    Object spec = i.next();
	    Constraint c = (Constraint)parsers.match(spec);
	    if (c == null)
		throw syntaxError("Invalid constraint " + spec);
	    else
		result.add(c);
	}
	return result;
    }

    public ListOfIssue makeIssues(LList items) {
	ListOfIssue result = new LinkedListOfIssue();
	for (Iterator i = items.iterator(); i.hasNext();) {
	    LList issue = theLList(i.next());
	    if (issue.get(0) != S_ISSUE)
		throw syntaxError("Invalid issue syntax in " + issue);
	    else {
		LList pattern = theLList(issue.get(1));
		result.add(new Issue(pattern));
	    }
	}
	return result;
    }

    /*
     * Forcing numeric node ids
     */

    Refinement useNumericNodeIds(Refinement r) {
	if (r.getNodes() != null) {
	    Map idMap= makeNumericIdMap(r.getNodes());
	    ListOfNodeSpec nodes = new LinkedListOfNodeSpec();
	    for (Iterator i = r.getNodes().iterator(); i.hasNext();) {
		NodeSpec n = (NodeSpec)i.next();
		Name id = (Name)idMap.get(n.getId());
		Debug.expect(id != null);
		nodes.add(new NodeSpec(id, n.getPattern()));
	    }
	    r.setNodes(nodes);
	    if (r.getOrderings() != null) {
		ListOfOrdering ords = new LinkedListOfOrdering();
		for (Iterator i = r.getOrderings().iterator(); i.hasNext();) {
		    Ordering ord = (Ordering)i.next();
		    NodeEndRef from = ord.getFrom();
		    NodeEndRef to = ord.getTo();
		    Name fromId = (Name)idMap.get(from.getNode());
		    Name toId = (Name)idMap.get(to.getNode());
		    Debug.expect(fromId != null, "no node", fromId);
		    Debug.expect(toId != null, "no node", toId);
		    ords.add(new Ordering
			           (new NodeEndRef(from.getEnd(), fromId),
				    new NodeEndRef(to.getEnd(), toId)));
		}
		r.setOrderings(ords);
	    }
	}
	return r;
    }

    Map makeNumericIdMap(ListOfNodeSpec nodes) {
	Map map = new HashMap();
	int n = 1;
	for (Iterator i = nodes.iterator(); i.hasNext();) {
	    NodeSpec spec = (NodeSpec)i.next();
	    map.put(spec.getId(), Name.valueOf(new Long(n++)));
	}
	return map;
    }

    /*
     * Orderings
     */

    /**
     * Expands a list of orderings into a list of simple orderings.
     * Each simple ordering is a pair of node ids: (before after). <p>
     *
     * Each ordering is a list representing a sequence.  Consider
     * adjacent elements A and B in such a list.  Each node in A is
     * linked before each node in B.  For example, (1 (2 3) 4) is
     * equivalent to the four separate lists (1 2), (1 3), (2 4), (3 4).
     */
    public static LList expandAllOrderings(LList orderings) {
	LListCollector result = new LListCollector();
	for (LList ords = orderings; ords != Lisp.NIL; ords = ords.cdr()) {
	    result.concLList(expandOrdering(theLList(ords.car())));
	}
	return result.contents();
    }

    public static LList expandOrdering(LList ordering) {
	LListCollector result = new LListCollector();
	// for adjacent elements left and right ...
	for (LList ords = ordering; ords != Lisp.NIL; ords = ords.cdr()) {
	    Object left = ords.car();
	    Object right = ords.cdr().car();
	    result.concLList(expandOrderPair(left, right));
	}
	return result.contents();
    }

    static LList expandOrderPair(Object left, Object right) {
	LListCollector result = new LListCollector();
	for (LList l = ensureList(left); l != Lisp.NIL; l = l.cdr()) {
	    for (LList r = ensureList(right); r != Lisp.NIL; r = r.cdr()) {
		Object from = l.car();
		Object to = r.car();
		result.addElement(Lisp.list(from, to));
	    }
	}
	return result.contents();
    }

    static LList ensureList(Object a) {
	return a instanceof LList ? (LList)a : Lisp.list(a);
    }


    /*
     * Constraint parsing
     */

    protected MatchTable parsers = new MatchTable();

    public Constraint parseConstraint(Object spec) {
	// May be call from outside the LTF parser.
	// /\/ Which is why it doesn't throw syntaxError.
	return (Constraint)parsers.match(spec);
    }

    public MatchTable getConstraintParsers() {
	return parsers;
    }

    public static abstract class ConstraintParser extends MatchCase {
	ConstraintParser(String pat) {
	    this.pattern = Lisp.readFromString(pat);
	}
	public Object tryMatch(Object spec) {
	    // Debug.noteln("Matching against", pattern);
	    MatchEnv env = SimpleMatcher.match(pattern, spec);
	    // Debug.noteln("Result:", env);
	    return env;
	}
	public Object ifSelected(Object spec, Object match) {
	    return makeConstraint((LList)spec, (MatchEnv)match);
	}
	public abstract Constraint makeConstraint(LList spec, MatchEnv env);
	public abstract Constraint makeTemplate();
    }

    protected void addConstraintParsers() {

        parsers.addCase(
	    new ConstraintParser(
	        "(world-state condition ?pattern = ?value)") {

	    public Constraint makeConstraint(LList spec, MatchEnv env) {
		LList pat = theLList(env.get(Q_PATTERN));
		Object val = env.get(Q_VALUE);
		return new Constraint(
		    theSymbol(spec.get(0)), 	// world-state
		    theSymbol(spec.get(1)),	// condition
		    Lisp.list(
		        new PatternAssignment(pat, val)));
	    }


	    public Constraint makeTemplate() {
		return new Constraint("world-state", "conditiion",
				      Lisp.list(new PatternAssignment()));
	    }

	});

        parsers.addCase(
	    new ConstraintParser(
	        "(world-state effect ?pattern = ?value)") {

	    public Constraint makeConstraint(LList spec, MatchEnv env) {
		LList pat = theLList(env.get(Q_PATTERN));
		Object val = env.get(Q_VALUE);
		return new Constraint(
		    theSymbol(spec.get(0)), 	// world-state
		    theSymbol(spec.get(1)),	// effect
		    Lisp.list(
		        new PatternAssignment(pat, val)));
	    }

	    public Constraint makeTemplate() {
		return new Constraint("world-state", "effect",
				      Lisp.list(new PatternAssignment()));
	    }

	});

        parsers.addCase(
	    new ConstraintParser("(compute ?pattern = ?value)") {

	    public Constraint makeConstraint(LList spec, MatchEnv env) {
		LList pat = theLList(env.get(Q_PATTERN));
		Object val = env.get(Q_VALUE);
		return new Constraint(
		    theSymbol(spec.get(0)), 	// compute
		    null,			// null
		    Lisp.list(
		        new PatternAssignment(pat, val)));
	    }

	    public Constraint makeTemplate() {
		return new Constraint(Symbol.intern("compute"), null,
				      Lisp.list(new PatternAssignment()));
	    }

	});

        parsers.addCase(
	    new ConstraintParser(
		"(compute multiple-answer ?pattern = ?value)") {

	    public Constraint makeConstraint(LList spec, MatchEnv env) {
		LList pat = theLList(env.get(Q_PATTERN));
		Object val = env.get(Q_VALUE);
		return new Constraint(
		    theSymbol(spec.get(0)), 	// compute
		    theSymbol(spec.get(1)),     // multiple-answer
		    Lisp.list(
		        new PatternAssignment(pat, val)));
	    }

	    public Constraint makeTemplate() {
		return new Constraint("compute", "multiple-answer",
				      Lisp.list(new PatternAssignment()));
	    }

	});

//	parsers.addCase(
//  	    new ConstraintParser(
//  		"(temporal before ?from ?to ?time-window)") {

//  	    public Constraint makeConstraint(LList spec, MatchEnv env) {
//  		LList pat = theLList(env.get(Q_PATTERN));
//  		Object val = env.get(Q_VALUE);
//  		return new Constraint(
//  		    theSymbol(spec.get(0)), 	// temporal
//  		    theSymbol(spec.get(1)),     // before
//  		    Lisp.list(new NodeEndRef(),
//  			      new NodeEndRef(),
//  			      new TimeWindow());
//  	    }

//  	    public Constraint makeTemplate() {
//  		return new Constraint
//  		    ("temporal", "before",
//  		     Lisp.list(new NodeEndRef(),
//  			       new NodeEndRef(),
//  			       new TimeWindow()));
//  	    }

//  	});

        parsers.addCase(
	    new ConstraintParser(
		"(temporal duration self = ?min .. ?max)") {

	    public Constraint makeConstraint(LList spec, MatchEnv env) {
		String min = env.get(Q_MIN).toString();
		String max = env.get(Q_MAX).toString();
		return new Constraint(
		    theSymbol(spec.get(0)), 	// temporal
		    theSymbol(spec.get(1)),     // duration
		    Lisp.list(Name.valueOf("self"),
			      new TimeWindow(new Duration(min),
					     new Duration(max))));
	    }

	    public Constraint makeTemplate() {
		return new Constraint
		    ("temporal", "duration",
		     Lisp.list(Name.class,
			       new TimeWindow()));
	    }

	});

        parsers.addCase(
	    new ConstraintParser(
		"(resource ?operation ?pattern = ?value)") {

	    public Constraint makeConstraint(LList spec, MatchEnv env) {
		LList pat = theLList(env.get(Q_PATTERN));
		Object val = env.get(Q_VALUE);
		return new Constraint(
		    theSymbol(spec.get(0)), 	// resource
		    theSymbol(spec.get(1)),	// the ?operation
		    Lisp.list(
		        new PatternAssignment(pat, val)));
	    }

	    public Constraint makeTemplate() {
		return new Constraint
		    ("resource", "?operation",
		     Lisp.list(new PatternAssignment()));
	    }

	});

	parsers.addCase(
	    new ConstraintParser(
		"(advice expansion-refinement ?verb ?names)") {

	    public Constraint makeConstraint(LList spec, MatchEnv env) {
		return new Constraint(
		    theSymbol(spec.get(0)),	// advice
		    theSymbol(spec.get(1)),	// expansion-refinement
		    Lisp.list(
			theSymbol(spec.get(2)),	// ?verb
		        spec.get(3)));		// ?names
	    }

	    public Constraint makeTemplate() {
		return new Constraint(
		    "advice", "expansion-refinement",
		    Lisp.list(Symbol.class,
			      ListOfSymbol.class));
	    }

	});

        parsers.addCase(	// "Other constraints"

	    new ConstraintParser(
                // Use (&rest ?pattern) so that it will match only lists.
	        // It's still just one pattern.
		"(?type ?subtype (&rest ?pattern) = ?value)") {

	    public Constraint makeConstraint(LList spec, MatchEnv env) {
		LList pat = theLList(env.get(Q_PATTERN));
		Object val = env.get(Q_VALUE);
		Constraint c =
		    new Constraint(
		        theSymbol(spec.get(0)),	// type
			theSymbol(spec.get(1)),	// subtype
			Lisp.list(new PatternAssignment(pat, val)));
		otherConstraints.add(c);
		return c;
	    }

	    public Constraint makeTemplate() {
		return new Constraint
		    ("?type", "?subtype",
		     Lisp.list(new PatternAssignment()));
	    }

	});

    }

    /*
     * Object classes
     */

    ObjectClass makeObjectClass(LList def) {
	Debug.expectEquals(S_OBJECT_CLASS, def.get(0));
	ObjectClass c = new ObjectClass();
	c.setName(theName(def.get(1)));
	c.setSuperClassNames(makeSuperNames(theLList(def.get(2))));
	c.setObjectProperties(makeObjectProperties(def.drop(3)));
	return c;
    }

    ListOfSymbol makeSuperNames(LList names) {
	ListOfSymbol result = new LinkedListOfSymbol();
	for (Iterator i = names.iterator(); i.hasNext();) {
	    result.add(theSymbol(i.next()));
	}
	return result;
    }

    ListOfObjectProperty makeObjectProperties(LList props) {
	ListOfObjectProperty result = new LinkedListOfObjectProperty();
	for (Iterator i = props.iterator(); i.hasNext();) {
	    result.add(makeObjectProperty(theLList(i.next())));
	}
	return result;
    }

    ObjectProperty makeObjectProperty(LList spec) {
	Symbol name = theSymbol(spec.get(0));
	Symbol objClass = null;
	ObjectProperty.Syntax syntax = null;
	switch (spec.length()) {
	case 3:
	    // (name :syntax syntax)
	    syntax = ObjectProperty.Syntax.valueOf(theSymbol(spec.get(2)));
	    if (spec.get(1) != K_SYNTAX)
		throw syntaxError("Expected :syntax in " + spec);
	    break;
	case 4:
	    // (name class :syntax syntax)
	    syntax = ObjectProperty.Syntax.valueOf(theSymbol(spec.get(3)));
	    if (spec.get(2) != K_SYNTAX)
		throw syntaxError("Expected :syntax in " + spec);
	    objClass = theSymbol(spec.get(1));
	case 2:
	    objClass = theSymbol(spec.get(1));
	case 1:
	    break;
	default:
	    throw syntaxError("Invalid object property " + spec);
	}
	return new ObjectProperty(name, objClass, syntax);
    }

    /*
     * Annotations
     */

    private LList annotationSyntax =
	(LList)Lisp.readFromString("(?key = ?value)");

    Map makeAnnotations(LList def) {
	Debug.expect(def.get(0) == S_ANNOTATIONS);
	return makeMap(def);
    }

    Map makeMap(LList def) {
	// def.car() might be S_ANNOTATIONS or S_MAP
	Map result = new StableHashMap();
	for (Iterator i = def.cdr().iterator(); i.hasNext();) {
	    LList kvPair = theLList(i.next());
	    if (SimpleMatcher.match(annotationSyntax, kvPair) == null)
		badAnnotation(kvPair);
	    Object key = kvPair.get(0);
	    Object value = kvPair.get(2);
	    if (isMapSyntax(value))
		value = makeMap((LList)value);
	    result.put(key, value);
	}
	return result;
    }

    boolean isMapSyntax(Object obj) {
	return obj instanceof LList
	    && ((LList)obj).car() == S_MAP;
    }

    void badAnnotation(LList kvPair) {
	if (badAnnotationsAreErrors)
	    throw syntaxError("Annotation " + kvPair +
			      " does not have the form (key = value).");
	else
            badAnnotations.add(kvPair.get(0).toString());
    }

    /*
     * Include
     */

    void doInclude(LList form) throws IOException {
	Debug.expectSame(S_INCLUDE, form.car());
	String name = theString(form.get(1));
	URL url = defaultedIncludeURL(name);
	Debug.noteln("Will try LTF include from", url);
	Reader r = Util.openURLReader(url);
	IncludeReader in = new IncludeReader(url, r, lin);
	in.pushState();
    }

    URL defaultedIncludeURL(String name) throws MalformedURLException {
	// String base = Strings.beforeLast("/", this.sourceName);
	String base = this.sourceName;
	URL baseURL = XML.toURL(base);
	Debug.noteln("Base URL for LTF include", baseURL);
	if (baseURL == null)
	    throw new MalformedURLException
		("Cannot treat as URL: " + base);
	return new URL(baseURL, name);
    }

    /**
     * Wrapper class used to read included resources.
     */
    class IncludeReader extends LispReader {

	String resourceName;
	LispReader wrappedReader;

	LispReader pushedReader;
	String pushedSourceName;
	Object pushedLastRead;

	IncludeReader(URL resourceURL,
		      Reader resourceReader,
		      LispReader readerToWrap) {
	    super(resourceReader);
	    this.resourceName = resourceURL.toString();
	    this.wrappedReader = readerToWrap;
	}

	public Object readObject() {
	    try {
		Object result = super.readObject();
		if (result != Lisp.EOF)
		    return result;
	    }
	    catch (Throwable t) {
		Debug.noteException(t);
		SyntaxException e = syntaxError(Debug.describeException(t));
		popState();
		throw e;
	    }
	    // We arrive here when we reach the end of the included
	    // resource without error.
	    popState();
	    return wrappedReader.readObject();
	}

	void pushState() {
	    Debug.noteln("Pushing LTF_Parser state to", resourceName);
	    pushedSourceName = sourceName;
	    pushedLastRead = lastRead;
	    pushedReader = lin;
	    sourceName = resourceName;
	    lastRead = "Start of file";
	    lin = this;
	}

	void popState() {
	    Debug.noteln("Popping LTF_Parser state back to", pushedSourceName);
	    lin = pushedReader;
	    sourceName = pushedSourceName;
	    lastRead = pushedLastRead;
	}

    }

}

// Issues:
// * Add schemas one at a time or pass a list to new Domain()?
// * Tell the domain to analyse itself or leave it "raw"?
// * If we gave the domain a list of all schemas, it could
//   do the analysis automaticlly.  When we give it schemas
//   one at a time, it needs to be told it has them all.
// * Should the constructor take a String or a File?
// * Need to do more syntax-checking.
