/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Nov  7 16:28:50 2007 by Jeff Dalton
 * Copyright: (c) 2001 - 2007, AIAI, University of Edinburgh
 */

package ix.icore.process;

import java.util.*;

import ix.icore.domain.*;
import ix.icore.Variable;
import ix.icore.Status;
import ix.icore.HasStatus;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;
import ix.util.context.*;

/**
 * A node that represents an action in a process.  This class
 * supports expansion into subactions and the propagation of status
 * changes, thus acting as a simple process model / simulator and
 * expansion planner.
 */
public abstract class PNode extends TimePoint implements HasStatus {

    public static boolean computeStatus = true; // changed while loading a plan

    // /\/ Shouldn't be public.
    public ProcessModelManager modelManager;

    protected int level = 0;		// expansion level, 1 + the parent's

    protected PNode parent = null;

    protected PNodeEnd beginEnd;
    protected PNodeEnd endEnd;

    // Fields that might have different values in different expansions
    // should be private to hide the way they use contexts.

    private LLQueue<PNode> children = new LLQueue<PNode>();

    private ContextValue expansionVarTable =
	new TypedContextValue(Map.class, null);
    // when instantiating a refinement
    // maps ?name to instance of Variable.

    private /* Refinement */ ContextValue expansionRefinement = 
	new TypedContextValue(Refinement.class, null);

    public PNode(PNode parent) {

	this.parent = parent;
	if (parent != null)
	    level = parent.level + 1;

	beginEnd = new PNodeEnd(End.BEGIN, this);
	endEnd = new PNodeEnd(End.END, this);
	
	beginEnd.linkBefore(endEnd);

	// computeStatus();	// may be too soon from subclass POV /\/

    }

    public abstract LList getPattern();

    public int getLevel() {
	return level;
    }

    public PNodeEnd getBegin() {
	return beginEnd;
    }

    public PNodeEnd getEnd() {
	return endEnd;
    }

    // getParent() is defined in subclasses to have a more specific
    // return type.

    public PNode getParentPNode() {
	return parent;
    }

    public List<PNode> getChildren() {
	return children.contents();
    }

    public boolean hasChildren() {
	return !children.isEmpty();
    }

    public void setChildren(List l) {
	children.setContents(l);
    }

    public ProcessModelManager getModelManager() {
	return modelManager;
    }

    public Map getVarTable() {
	Map vt = (Map)expansionVarTable.get();
	if (vt == null)
	    return null;
	else
	    return Collections.unmodifiableMap(vt);
    }

    public void setVarTable(Map vt) { // used when loading a plan /\/
	expansionVarTable.set(vt);
    }

    public Refinement getRefinement() {
	return (Refinement)expansionRefinement.get();
    }

    public void setRefinement(Refinement r) {
	expansionRefinement.set(r);
    }

    public abstract Status getStatus();

    public boolean isSiblingOf(PNode n) {
	return parent == n.parent;
    }

    public boolean isAncestorOf(PNode n) {
	for (PNode p = n.parent; p != null; p = p.parent) {
	    if (this == p)
		return true;
	}
	return false;
    }

    public boolean isDescendentOf(PNode n) {
	return n.isAncestorOf(this);
    }

    public boolean isExpanded() {
	return getVarTable() != null; // /\/
    }

    public void unlink() {
	beginEnd.unlink();
	endEnd.unlink();
    }

    public void addChild(PNode child, PNode after) {
	// Called when inserting a node, maybe someday for achieve /\/
	// Puts the new child before its sibling, after.
	Debug.expect(child.parent == null, "already has parent", child);
	if (after != null)
	    Debug.expect(children.contains(after), "not a sibling", after);
	child.parent = this;
	child.level = this.level + 1;
	// Put the new child in the right place in the list of children.
	if (after == null)
	    children.add(child);
	else
	    setChildren(Collect.insertBeforeFirst(after, getChildren(), 
						  Lisp.list(child)));
	// Ensure child is linked.
	PNodeEnd c_b = child.getBegin();
	PNodeEnd c_e = child.getEnd();
	if (c_b.getPredecessors().isEmpty())
	    c_b.linkAfter(this.getBegin());
	if (c_e.getSuccessors().isEmpty())
	    c_e.linkBefore(this.getEnd());
    }

    /**
     * Creates child nodes as specified by a refinement and installs
     * ordering links.  The subnodes are constructed by a factory method
     * so that they can be instances of a subclass of Node and to
     * allow other, related objects to be constructed at the same time.
     *
     * @return a map from the ids of child nodes to those children.
     *    The ids are taken from the node-specs in the refinement.
     */
    public Map expandOneLevel(Refinement sourceRefinement,
			       MatchEnv env) {

	Map varTable = new HashMap();
	Map nameToChildMap = new HashMap();

	setVarTable(varTable);

	// Instantiate the refinement
	Debug.expect(getRefinement() == null, "already expanded?", this);
	Function1 ifUnbound = new MakeVarIfUnbound(varTable);
	Refinement refinement = sourceRefinement.instantiate(env, ifUnbound);
	setRefinement(refinement);
	Debug.noteln("varTable after expansion", varTable);

	// Create the child nodes specified by the refinement.
	LListCollector c = new LListCollector();
	for (Iterator ni = Collect.iterator(refinement.getNodes());
	     ni.hasNext();) {
	    NodeSpec spec = (NodeSpec)ni.next();
	    LList childPattern = (LList)spec.getPattern();
	    PNode child = makePNode(PNode.this, childPattern);
	    c.add(child);
	    nameToChildMap.put(spec.getId(), child);
	}
	setChildren(c.contents());

	// Add the orderings
	// N.B. uses fields children and nameToChildMap.
	if (refinement.getOrderings() != null)
	    processOrderings(refinement.getOrderings(), nameToChildMap);
	ensureChildrenLinkedToParent();

	// Have all the children determine their status
	allComputeStatus(children);

	return nameToChildMap;

    }

    protected abstract PNode makePNode(PNode parent, LList pattern);

    public class MakeVarIfUnbound implements Function1 {

	protected Map nameToVarMap;

	public MakeVarIfUnbound(Map nameToVarMap) {
	    this.nameToVarMap = nameToVarMap;
	};

	public Object funcall(Object name) {
	    Variable v = (Variable)nameToVarMap.get(name);
	    if (v == null) {
		v = new Variable(name);
		Debug.noteln("Creating variable " + v + " for " + name);
		v.setSourceNode(PNode.this);
		nameToVarMap.put(name, v);
	    }
	    return v;
	}

    }
	
    public void processOrderings(List orderings, Map nameToChildMap) {
	// Public because called when loading a plan.
	for (Iterator ords = orderings.iterator(); ords.hasNext();) {
	    Ordering ord = (Ordering)ords.next();
	    PNodeEnd.addOrdering(nameToChildMap, ord);
	}
    }

    public void ensureChildrenLinkedToParent() {
	// This is at the node-end level.
	// The children should be new, and so not have links
	// with anything other than siblings.
	PNodeEnd p_b = getBegin();	// this node == parent
	PNodeEnd p_e = getEnd();
	for (Iterator i = children.iterator(); i.hasNext();) {
	    PNode child = (PNode)i.next();
	    PNodeEnd c_b = child.getBegin();
	    PNodeEnd c_e = child.getEnd();
	    if (c_b.getPredecessors().isEmpty())
		c_b.linkAfter(p_b);
	    if (c_e.getSuccessors().isEmpty())
		c_e.linkBefore(p_e);
	}
    }

    /**
     * computeStatus() is used to change the status of related nodes
     * when a node changes its status.
     */
    public void computeStatus() {
	// We assume that the current status may not be correct,
	// but only certain transitions are handled.

	if (!computeStatus) return;

	beginEnd.computeStatus();
	endEnd.computeStatus();

	if (getStatus() == Status.EXECUTING) {
            if (!children.isEmpty()) {
                if (allHaveStatus(children, Status.COMPLETE))
                    setStatus(Status.COMPLETE);
            }
        }

    }

    /**
     * Changes the node's status and then that of related nodes.
     */
    public void setStatus(Status status) {
	// A change in a node's status may affect the parent,
	// successor siblings, or children.
	// When this method is called, the status should already
	// have been changed (in a subclass).

	Debug.noteln("Setting status of " + this + " to " + status);

	// /\/: The status value might not even be stored in this object.
	// this.status = status;

	Debug.expect(computeStatus);

	// Let the model manager see the status change
	if (modelManager != null)	// it's null for Issues /\/
	    modelManager.statusChanged(this);

	// Propagate status change to related nodes.
	if (status == Status.BLANK) {
	    // This happens only when we manually insert a node
	    // before another and have to change the later node's
	    // status from POSSIBLE.  /\/
	    beginEnd.setStatus(Status.BLANK);
	    endEnd.setStatus(Status.BLANK);
	}
	else if (status == Status.COMPLETE) {
	    beginEnd.setStatus(Status.COMPLETE);
	    endEnd.setStatus(Status.COMPLETE);
	    if (parent != null)
		parent.computeStatus();		// may -> COMPLETE
	}
	else if (status == Status.EXECUTING) {
	    beginEnd.setStatus(Status.COMPLETE);
	}
	else if (status == Status.POSSIBLE) {
	    // When a node that already had children becomes POSSIBLE,
	    // we automatically make it start EXECUTING
	    // /\/: May cause problems in the future.
//  	    if (!children.isEmpty())
//  		setStatus(Status.EXECUTING);
	    if (!children.isEmpty()
		&& (modelManager == null /* eg IssueItem /\/ */ ||
		    Collect.isEmpty(modelManager.getNodeConditions(this))))
		setStatus(Status.EXECUTING);
	}
    }

    public void assignStatus(Status status) {
	// Adjust node-ends.
	if (status == Status.BLANK) {
	    beginEnd.setStatus(Status.BLANK);
	    endEnd.setStatus(Status.BLANK);
	}
	else if (status == Status.POSSIBLE) {
	    beginEnd.setStatus(Status.POSSIBLE);
	    endEnd.setStatus(Status.BLANK);
//  	    if (!children.isEmpty())
//  		assignStatus(Status.EXECUTING);
	}
	else if (status == Status.EXECUTING) {
	    beginEnd.setStatus(Status.COMPLETE);
	    endEnd.computeStatus();
	}
	else if (status == Status.COMPLETE) {
	    beginEnd.setStatus(Status.COMPLETE);
	    endEnd.setStatus(Status.COMPLETE);
	}
	else if (status == Status.IMPOSSIBLE) {
	    beginEnd.setStatus(Status.IMPOSSIBLE); // ??? /\/
	}
	else {
	    throw new ConsistencyException();
	}
    }

    public Status statusFromNodeEnds() {
	Status begin = beginEnd.getStatus();
	Status end = endEnd.getStatus();
	if (begin == Status.BLANK || begin == Status.POSSIBLE
	      || begin == Status.IMPOSSIBLE)
	    return begin;
	else if (begin == Status.COMPLETE) {
	    if (end == Status.COMPLETE)
		return Status.COMPLETE;
	    else
		return Status.EXECUTING;
	}
	throw new ConsistencyException();
    }

    public static boolean allHaveStatus(List l, Status status) {
	Iterator i = l.iterator();
	while(i.hasNext()) {
	    HasStatus n = (HasStatus)i.next();
	    if (n.getStatus() != status)
		return false;
	}
	return true;
    }

    public static void allComputeStatus(List l) {
	Iterator i = l.iterator();
	while (i.hasNext()) {
	    HasStatus n = (HasStatus)i.next();
	    n.computeStatus();
	}
    }

}

