/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Sep 19 01:56:14 2007 by Jeff Dalton
 * Copyright: (c) 2007, AIAI, University of Edinburgh
 */

package ix.icore.plan;

import java.util.*;

import ix.icore.domain.*;
import ix.icore.*;

import ix.ip2.ActivityItem; // for EXPANSION_REFINEMENT_NAME /\/

import ix.util.*;
import ix.util.lisp.*;
import ix.util.xml.*;

/**
 * A plan represented as a domain.
 */
public class PlanAsDomain extends Domain {

    static final Symbol
	EXPANDS = Symbol.intern("expands"),
	WORLD_STATE = Symbol.intern("world-state");

    protected Map<Name,Activity> idToActivityMap;

    protected Map<Name,ItemVar> planVarIdToItemVarMap =
	new HashMap<Name,ItemVar>(); // filled in as we go

    protected Gensym.Generator nameGen = new Gensym.Generator();
    { nameGen.setSeparator("."); }

    public PlanAsDomain() {
    }

    public PlanAsDomain(Plan plan) {
	// Map ids to activities in the plan.
	idToActivityMap = makeIdToActivityMap(plan);
	// Map plan-var ids to item-vars
	planVarIdToItemVarMap = makePlanVarIdToItemVarMap(plan);
	// Top level
	addRefinement(makeTopLevelRefinement(plan));
	// Refinements
	if (plan.getPlanRefinements() != null) {
	    for (PlanRefinement pr:
		     (List<PlanRefinement>)plan.getPlanRefinements())
		addRefinement(makeRefinement(pr));
	}
    }

    Map<Name,Activity> makeIdToActivityMap(Plan plan) {
	final Map<Name,Activity> map = new HashMap<Name,Activity>();
	final List<PlanNode> lackUniqueIds = new LinkedList<PlanNode>();
	ObjectWalker walker = new PlanWalker() {
	    @Override
	    public void walk(Object o) {
		if (o instanceof PlanNode) {
		    PlanNode n = (PlanNode)o;
		    if (map.put(n.getId(), n.getActivity()) != null)
			lackUniqueIds.add(n);
		}
		else
		    super.walk(o);
	    }
	};
	walker.walk(plan);
	if (!lackUniqueIds.isEmpty())
	    throw new InvalidPlanException
		("Some nodes do not have unique ids: " + 
		 Strings.conjunction(lackUniqueIds));
	return map;
    }

    Map<Name,ItemVar> makePlanVarIdToItemVarMap(Plan plan) {
	final Map<Name,ItemVar> map = new HashMap<Name,ItemVar>();
	final List<PlanVariableDeclaration> nonUnique
	    = new LinkedList<PlanVariableDeclaration>();
	ObjectWalker walker = new PlanWalker() {
	    @Override
	    public void walk(Object o) {
		if (o instanceof PlanVariableDeclaration)
		    walkPVDcl((PlanVariableDeclaration)o);
		else
		    super.walk(o);
	    }
	    void walkPVDcl(PlanVariableDeclaration dcl) {
		if (map.put(dcl.getId(), makeItemVar(dcl)) != null)
		    nonUnique.add(dcl);
	    }
	};
	walker.walk(plan);
	if (!nonUnique.isEmpty())
	    throw new InvalidPlanException
		("Some variables do not have unique ids: " + 
		 Strings.conjunction(nonUnique));
	return map;
    }

    class PlanWalker extends ObjectWalker {
	@Override
	public void visitElement(Object elt) {
	    if (!(elt instanceof Plan)) // avoid confusion
		walk(elt);
	}
    }

    Refinement makeTopLevelRefinement(Plan plan) {
	Refinement r = new Refinement("plan-top-level",
				      Lisp.list("Top level of the plan"));
	// Variable declarations
	setVarDcls(r, plan.getPlanVariableDeclarations());
	// Issues
	// Nodes
	setNodeSpecs(r, plan.getPlanNodes());
	// Constraints and orderings
	setConstraintsAndOrderings(r, plan.getConstraints());
	// World state.
	if (plan.getWorldState() != null)
	    r.setAnnotation(WORLD_STATE,
			    PatternAssignment.assignmentsToMap
			        (plan.getWorldState()));
	// We're done.
	return r;
    }

    Refinement makeRefinement(PlanRefinement pr) {
	Activity act = idToActivityMap.get(pr.getExpands());
	Refinement r =
	    new Refinement("expand-" + pr.getExpands(),
			   patternForDomain(act.getPattern()));
	// Record which node the refinement expands.
	r.setAnnotation(EXPANDS, Symbol.intern(pr.getExpands().toString()));
	// Record the name of the refinement that was used
	// to expand the activity, if it's known.
	if (act.getAnnotation(ActivityItem.EXPANSION_REFINEMENT_NAME) != null)
	    r.setAnnotation
		(ActivityItem.EXPANSION_REFINEMENT_NAME,
		 act.getAnnotation(ActivityItem.EXPANSION_REFINEMENT_NAME));
	// Variable declarations
	setVarDcls(r, pr.getPlanVariableDeclarations());
	// Nodes
	setNodeSpecs(r, pr.getPlanNodes());
	// Constraints and orderings
	setConstraintsAndOrderings(r, pr.getConstraints());
	// We're done.
	return r;
    }

    void setVarDcls(Refinement r, ListOfPlanVariableDeclaration planDcls) {
	if (planDcls == null)
	    return;
	ListOfVariableDeclaration dcls =
	    new LinkedListOfVariableDeclaration();
	for (Iterator i = planDcls.iterator(); i.hasNext();) {
	    PlanVariableDeclaration pvdcl =
		(PlanVariableDeclaration)i.next();
	    dcls.add(new VariableDeclaration(getItemVar(pvdcl)));
	}
	r.setVariableDeclarations(dcls);
    }

    void setNodeSpecs(Refinement r, ListOfPlanNode planNodes) {
	if (planNodes == null)
	    return;
	ListOfNodeSpec specs = new LinkedListOfNodeSpec();
	for (Iterator i = planNodes.iterator(); i.hasNext();) {
	    PlanNode pn = (PlanNode)i.next();
	    LList pattern = pn.getActivity().getPattern();
	    specs.add(new NodeSpec(pn.getId(), patternForDomain(pattern)));
	}
	r.setNodes(specs);
    }

    void setConstraintsAndOrderings(Refinement r,
				    ListOfConstrainer constrainers) {
	if (constrainers == null)
	    return;
	ListOfOrdering orderings = getOrderings(constrainers);
	ListOfConstraint constraints = getNonOrderings(constrainers);
	if (!orderings.isEmpty())
	    r.setOrderings(orderings);
	if (!constraints.isEmpty())
	    r.setConstraints(constraints);
    }

    /* * * Constraint and ordering utilities * * */

    ListOfOrdering getOrderings(ListOfConstrainer constraints) {
	ListOfOrdering orderings = new LinkedListOfOrdering();
	for (Constrainer c: (List<Constrainer>)constraints) {
	    if (c instanceof Ordering)
		orderings.add((Ordering)c);
	}
	return orderings;
    }

    ListOfConstraint getNonOrderings(ListOfConstrainer constraints) {
	ListOfConstraint nonOrderings = new LinkedListOfConstraint();
	for (Constrainer c: (List<Constrainer>)constraints) {
	    if (!(c instanceof Ordering))
		nonOrderings.add((Constraint)c);
	}
	return nonOrderings;
    }

    /* * * Representing PlanVariables and PlanVariableDeclarations * * */

    LList patternForDomain(List pattern) {
	LListCollector result = new LListCollector();
	for (Object item: pattern) {
	    if (item instanceof PlanVariable)
		result.add(getItemVar((PlanVariable)item));
	    else if (item instanceof List)
		result.add(patternForDomain((List)item));
	    else
		result.add(item);
	}
	return result.contents();
    }

    ItemVar getItemVar(PlanVariable pv) {
	ItemVar v = planVarIdToItemVarMap.get(pv.getId());
	if (v == null)
	    throw new InvalidPlanException
		("Undeclared variable: " + pv);
	return v;
    }

    ItemVar getItemVar(PlanVariableDeclaration dcl) {
	ItemVar v = planVarIdToItemVarMap.get(dcl.getId());
	if (v == null)
	    throw new ConsistencyException
		("Missed declaration:", dcl);
	return v;
    }

    ItemVar makeItemVar(PlanVariableDeclaration dcl) {
	return (ItemVar)nameGen.nextSymbol(dcl.getName().toString());
    }

    /* * * Main program for testing * * */

    public static void main(String[] argv) {
	Debug.off();
	Parameters.processCommandLineArguments(argv);
	String planName = Parameters.getParameter("plan");
	String output = Parameters.getParameter("output");
	if (planName == null || output == null) {
	    System.out.println("Usage: Must specify plan= and output=");
	    System.exit(1);
	}
	Plan plan = XML.readObject(Plan.class, planName);
	Domain domain = new PlanAsDomain(plan);
	XML.writeObject(domain, output);
    }

}
