/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Feb 14 12:11:23 2007 by Jeff Dalton
 * Copyright: (c) 2000 - 2007, AIAI, University of Edinburgh
 */

package ix.icore.domain;

import java.util.*;

import ix.icore.*;
import ix.icore.domain.event.*;
import ix.iface.domain.SyntaxException; // need better package for it /\/
import ix.util.*;
import ix.util.lisp.*;

/**
 * A Domain contains descriptions of ways to refine activities
 * by expanding them into subactivities and adding constraints.
 */
public class Domain extends AbstractIXObject implements Named, Cloneable {

    // /\/: Should allow refinements to be null.

    protected String name;
    protected ListOfVariableDeclaration variableDeclarations;
    protected ListOfRefinement refinements = new LinkedListOfRefinement();
    protected ListOfObjectClass objectClasses;

    protected List listeners = new LListCollector();

    protected Map refinementNameMap = new HashMap();
    protected Map classNameMap = new HashMap();

    public Domain() {
	super();
    }

    public String getName() {
	return name;
    }

    public void setName(String name) {
	this.name = name;
    }

    public boolean isEmpty() {
	return refinements.isEmpty(); // check other fields as well? /\/
    }

    public void clear() {
	Debug.noteln(this + " cleared");
	refinements.clear();
	refinementNameMap.clear();
	if (objectClasses != null) objectClasses.clear();
	classNameMap.clear();
    }

    /*
     * Variable declarations
     */

    public ListOfVariableDeclaration getVariableDeclarations() {
        return this.variableDeclarations;
    }

    public void setVariableDeclarations
	            (ListOfVariableDeclaration variableDeclarations) {
        this.variableDeclarations = variableDeclarations;
    }

    /*
     * Refinements
     */

    public ListOfRefinement getRefinements() {
	return refinements;
    }

    public void setRefinements(ListOfRefinement refinements) {
	// N.B. List can't be null
	Debug.expect(refinements != null, "trying to set null refinements");
	this.refinements = refinements;
	// Construct name map
	refinementNameMap.clear();
	for (Iterator i = refinements.iterator(); i.hasNext();) {
	    Refinement r = (Refinement)i.next();
	    refinementNameMap.put(r.getName(), r);	    
	}
    }

    // /\/: Backwards-compatibility for I-DE.
    public void setRefinements(List refinements) {
	setRefinements(refinements == null ? null :
		       new LinkedListOfRefinement(refinements));
    }

    public Refinement getNamedRefinement(String name) {
	return (Refinement)refinementNameMap.get(name);
    }

    public void addRefinement(Refinement r) {
	Debug.expect(getNamedRefinement(r.getName()) == null,
		     "Two refinements named", r.getName());
	Debug.noteln(this + " adds " + r);
	refinementNameMap.put(r.getName(), r);
	refinements.add(r);
	fireRefinementAdded(r);
    }

    public void deleteNamedRefinement(String name) {
	deleteRefinement(getNamedRefinement(name));
    }

    public void deleteRefinement(Refinement r) {
	// /\/: Use with great caution.  Also, this metod should
	// tell the listeners but doesn't.
	if (refinements.contains(r)) {
	    Debug.noteln(this + " deletes " + r);
	    refinements.remove(r);
	    refinementNameMap.remove(r.getName());
	}
	else
	    throw new IllegalArgumentException
		("Attempt to delete " + r +
		 " when it is not in " + this);
    }

    public void replaceNamedRefinement(String name, Refinement replacement) {
	replaceRefinement(getNamedRefinement(name), replacement);
    }

    public void replaceRefinement(Refinement old, Refinement neu) {
	// /\/: Use with great caution.  Also, this metod should
	// tell the listeners but doesn't.
	// N.B. The Refinement's name might change.

	// Give the new refinement the old one's place in the list
	int i = refinements.indexOf(old);
	if (i >= 0) {
	    Debug.noteln(this + " replaces " + old + " with " + neu);
	    refinements.remove(i);
	    refinements.add(i, neu);
	    // Make the name map to the new refinement instead of the old.
	    refinementNameMap.put(old.getName(), neu);
	}
	else
	    throw new IllegalArgumentException
		("Attempt to replace " + old +
		 " when it is not in " + this);
    }

    /*
     * Object classes
     */

    public ListOfObjectClass getObjectClasses() {
	return objectClasses;
    }

    public void setObjectClasses(ListOfObjectClass classes) {
	// Debug.expect(classes != null, "trying to set null object classes");
	this.objectClasses = classes;
	// Construct the name -> class map
	classNameMap.clear();
	if (classes == null)
	    return;
	for (Iterator i = objectClasses.iterator(); i.hasNext();) {
	    ObjectClass c = (ObjectClass)i.next();
	    classNameMap.put(c.getName(), c);
	}
    }

    // /\/: Backwards-compatibility for I-DE.
    public void setObjectClasses(List classes) {
	setObjectClasses(classes == null ? null :
			 new LinkedListOfObjectClass(classes));
    }

    public ObjectClass getNamedObjectClass(String name) {
	return (ObjectClass)classNameMap.get(name);
    }

    public void addObjectClass(ObjectClass c) {
	Debug.expect(getNamedObjectClass(c.getName()) == null,
		     "Two object classes named", c.getName());
	Debug.noteln(this + " adds " + c);
	if (objectClasses == null)
	    objectClasses = new LinkedListOfObjectClass();
	classNameMap.put(c.getName(), c);
	objectClasses.add(c);
	// fireClassAdded();
    }

    public void deleteNamedObjectClass(String name) {
	deleteObjectClass(getNamedObjectClass(name));
    }

    public void deleteObjectClass(ObjectClass c) {
	// /\/: Use with great caution.  Also, this metod should
	// tell the listeners but doesn't.
	if (objectClasses.contains(c)) {
	    Debug.noteln(this + " deletes " + c);
	    objectClasses.remove(c);
	    classNameMap.remove(c.getName());
	}
	else
	    throw new IllegalArgumentException
		("Attempt to delete " + c +
		 " when it is not in " + this);
    }

    public void replaceObjectClass(ObjectClass old, ObjectClass neu) {
	// /\/: See warning commments in replaceRefinement.
	int i = objectClasses.indexOf(old);
	if (i > 0) {
	    Debug.noteln(this + " replaces " + old + " with " + neu);
	    // Give the new class the old one's place in the list.
	    objectClasses.remove(i);
	    objectClasses.add(i, neu);
	    // Make the name map to the new class instead of the old
	    classNameMap.put(old.getName(), neu);
	}
	else
	    throw new IllegalArgumentException
		("Attempt to replace " + old +
		 " when it is not in " + this);
    }

    /*
     * Listener notification
     */

    public void addDomainListener(DomainListener listener) {
	listeners.add(listener);
    }

    public void fireRefinementAdded(Refinement r) {
	RefinementEvent event = new RefinementEvent(this, r);
	for (Iterator i = listeners.iterator(); i.hasNext();) {
	    DomainListener listener = (DomainListener)i.next();
	    listener.refinementAdded(event);
	}
    }

    /*
     * Domain merging
     */

    /** Merges another domain into this one. */
    public void takeFrom(Domain other) {
	Debug.noteln(this + " takes from " + other);
	// Name
	if (other.name != null) {
	    if (name != null) {
		List parts = Strings.breakAt("+", name);
		if (!parts.contains(other.name)) {
		    name = name + "+" + other.name;
		    Debug.noteln("Extended name of", this);
		}
	    }
	    else {
		name = other.name;
		Debug.noteln("New name for", this);
	    }
	}
	// Variable declarations
	// /\/: ...
	// Refinements
	for (Iterator i = other.getRefinements().iterator(); i.hasNext();) {
	    Refinement r = (Refinement)i.next();
	    Refinement exists = getNamedRefinement(r.getName());
	    if (exists == null)
		addRefinement(r);
	    else
		replaceRefinement(exists, r);
	}
	// Object-classes
	for (Iterator i = Collect.iterator(other.getObjectClasses())
		 ; i.hasNext();) {
	    ObjectClass c = (ObjectClass)i.next();
	    ObjectClass exists = null;
	    if (exists == null)
		addObjectClass(c);
	    else
		replaceObjectClass(exists, c);
	}
	// Annotations
	for (Iterator i = Collect.ensureMap(other.getAnnotations())
		            .entrySet().iterator()
		 ; i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    Object k = e.getKey();
	    Object v = e.getValue();
	    Object ourV = getAnnotation(k);
	    if (ourV == null)
		setAnnotation(k, v);
	    else if (!ourV.equals(v))
		throw new IllegalArgumentException
		    ("Conflicting annotation values: " + this +
		     " cannot take value " + v + " for key " + k +
		     " from " + other + ", because it already has value " +
		     ourV);
	}
    }

    /*
     * Analysis
     */

    public void analyseDomain() {

	// checkRefinementReferences();

    }

    /**
     * Checks the consistency of this domain.  This method calls
     * the {@link Refinement#checkConsistency()} method of each
     * refinement, catches and remembers any {@link SyntaxException}s
     * that are thrown, and, if there were any exceptions, throws
     * a combined SyntaxException that describes them all.
     *
     * @throws SyntaxException if there are unused or undeclared
     *    variables or if any constraint refers to a nonexistent node.
     */
    public void checkConsistency() {
	Debug.noteln("Checking consistency of", this);
	List problems = new LinkedList();
	for (Iterator i = refinements.iterator(); i.hasNext();) {
	    Refinement r = (Refinement)i.next();
	    try {
		r.checkConsistency();
	    }
	    catch (SyntaxException e) {
		problems.add(e);
	    }
	}
//  	if (!problems.isEmpty()) {
//  	    ix.util.xml.XML.writeObject(this, "/tmp/buggy-domain.lsp");
//  	}
	switch (problems.size()) {
	case 0: return;
	case 1: throw (SyntaxException)problems.get(0);
	default:
	    List lines = new LinkedList();
	    lines.add("Problems:");
	    for (Iterator i = problems.iterator(); i.hasNext();) {
		SyntaxException e = (SyntaxException)i.next();
		lines.add(Debug.describeException(e));
	    }
	    throw new SyntaxException(Strings.joinLines(lines));
	}
    }

    /*
     * Utilities
     */

    public Object clone() throws CloneNotSupportedException {
	return super.clone();
    }

    public String toString() {
	return "Domain[" + name + ", "
	    + refinements.size() + " refinements]";
    }

}

// Issues:
// * Need a way to deal with a set of changes to a domain as a unit,
//   because it may be impossible to keep things consistent when changes
//   (such as refinement replacements) are done one at a time.
