/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Sep 13 02:59:04 2006 by Jeff Dalton
 * Copyright: (c) 2002, 2003, AIAI, University of Edinburgh
 */

package ix.icore.domain;

import java.lang.reflect.Method;
import java.io.Serializable;
import java.util.*;

import ix.icore.Sendable;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.MatchEnv;

/**
 * A general-purpose container for constraint information.  In most
 * cases, the type and relation will together determine (in a fairly
 * straightforward way) what the parameters should contain.
 */
public class Constraint extends Constrainer 
    implements Serializable, Cloneable, Sendable {

    // Used when registering constraint-managers.
    public static final Keyword ALL = (Keyword)Symbol.intern(":all");

    protected Symbol type;
    protected Symbol relation;
    protected List parameters;

    // For SendAble interface /\/
    protected Name senderId;	// source agent name, if source is external

    public Constraint() {
    }

    public Constraint(Symbol type, Symbol relation, List parameters) {
	super();
	this.type = type;
	this.relation = relation;
	this.parameters = parameters;
    }

    /**
     * A constructor that takes the type and relation as strings
     * and converts them to symbols.
     */
    public Constraint(String type, String relation, List parameters) {
	this(Symbol.intern(type), Symbol.intern(relation), parameters);
    }

    public Symbol getType() { return type; }
    public void setType(Symbol type) { this.type = type; }

    public Symbol getRelation() { return relation; }
    public void setRelation(Symbol relation) { this.relation = relation; }

    public List getParameters() { return parameters; }
    public void setParameters(List p) { this.parameters = p; }

    public Object getParameter(int i) { return parameters.get(i); }

    public Name getSenderId() { return senderId; }
    public void setSenderId(Name id) { this.senderId = id; }

    public PatternAssignment getPatternAssignment() {
	// /\/: This method is just for convenience.
	return
	    (PatternAssignment)Util.mustBe(PatternAssignment.class,
					   parameters.get(0));
    }

    public LList getPattern() {
	// /\/: This method is just for convenience.
	PatternAssignment pv =
	    (PatternAssignment)Util.mustBe(PatternAssignment.class,
					   parameters.get(0));
	return pv.getPattern();
    }

    public Object getValue() {
	// /\/: This method is just for convenience.
	PatternAssignment pv =
	    (PatternAssignment)Util.mustBe(PatternAssignment.class,
					   parameters.get(0));
	return pv.getValue();
    }

    public Constraint instantiate(MatchEnv env, Function1 ifUnbound) {
	return (Constraint)env.instantiate(this, ifUnbound);
    }

    public Object clone() throws CloneNotSupportedException {
	Constraint copy = (Constraint)super.clone();
	return copy;
    }

    public int hashCode() {
	int relCode = relation != null ? relation.hashCode() : 0;
	return type.hashCode()
	    +  31 * (relCode + (31 * parameters.hashCode()));
    }

    public boolean equals(Object x) {
	Constraint cx;
	return this.getClass() == x.getClass()
	    && this.getType() == (cx = (Constraint)x).getType()
	    && this.getRelation() == cx.getRelation()
	    && this.getParameters().equals(cx.getParameters());
    }

    public String toString() {
	// /\/: We used to make a LinkedList of the parameters so they
	// won't be printed by the Lisp.printToString method, because it
	// prints random objects as {type i}.  Now we have a method.
	return "Constraint[" + type + ", " + relation + ", "
	                     + listToString(parameters) + "]";
    }

    private String listToString(List list) {
        return list == null ? "[]"
            :  list instanceof LList ? ((LList)list).toJavaString()
            :  list.toString();
    }

}

// Issues:
// * Should this class be in icore rather than icore.domain?
// * Should it really be Sendable?
