/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Nov 22 17:23:56 2007 by Jeff Dalton
 * Copyright: (c) 2001 - 2007, AIAI, University of Edinburgh
 */

package ix.icore;

import java.net.UnknownHostException;

import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import java.util.*;

import ix.icore.domain.Constraint;
import ix.icore.event.AgentNameListener;
import ix.icore.event.AgentNameEvent;
import ix.icore.log.EventLogger;
import ix.icore.log.HistoryEvent;

import ix.isim.ISimTimer;

import ix.ichat.ChatMessage;
import ix.iface.util.Reporting;
import ix.ispace.*;
import ix.util.*;
import ix.util.lisp.*;

/**
 * Common class for I-X agents. <p>
 *
 * An agent is usually created by instantiating a subclass of IXAgent
 * and then calling its <code>mainStartup(String[] argv)</code> method.
 * That method will usually be inherited from IXAgent; it processes any
 * command line arguments and then calls <code>startup()</code>.
 * Thus, from the subclass's point of view, startup and initialization
 * has the following steps:
 * <ol>
 * <li>One of the subclass's constructurs, which will call superclass
 *     constructors.
 * <li>{@link #mainStartup(String[])}, which calls IXAgent methods
 *    <ol>
 *    <li>{@link #processCommandLineArguments()}
 *    <li>{@link #startup()}
 *    <li>{@link #startServer(Object agentName, String strategyName)}
 *    </ol>
 * </ol>
 * A subclass that redefines <code>processCommandLineArguments()</code>
 * or <code>startup()</code> should normally have them call the
 * corresponding <code>super</code> method to ensure that any
 * common processing is performed.
 *
 * To support IPC, a subclass of IXAgent must assign a value
 * to the {@link #ipcName} field, to be returned by the
 * {@link #getAgentIPCName()} method, and will typically want
 * to define
 * <ul>
 * <li>{@link #handleNewIssue(Issue issue)}
 * <li>{@link #handleNewActivity(Activity activity)}
 * <li>{@link #handleNewConstraint(Constraint constraint)}
 * <li>{@link #handleNewReport(Report report)}
 * <li>{@link #handleNewChatMessage(ChatMessage message)}
 * </ul>
 * or, to intervene at an earlier stage, the method that calls
 * the ones above,
 * <ul>
 * <li>{@link #handleInput(IPC.InputMessage message)}
 * </ul>
 *
 * @see ix.test.SimpleIXAgent
 * @see <a 
 *       href="../../../../java/ix/test/SimpleIXAgent.java">SimpleIXAgent.java 
 *       source code</a>
 */
public abstract class IXAgent {

    // The dreaded global variable ...
    protected static IXAgent thisAgent = null;

    protected static Map<IXAgent,Boolean> knownAgents =
	new WeakHashMap<IXAgent,Boolean>();

    /**
     * Name used for IPC purposes and returned by the
     * <code>getAgentIPCName()</code> method.
     */
    protected String ipcName = "anonymous";

    protected String displayName = "I-X Agent";
    protected String symbolName = "anonymous";

    // /\/: This is needed because of setSymbolName.  We should
    // rearrange things so that it's not needed.
    protected String initialDisplayName = "not a display name";

    protected ContactManager contactManager = new ContactManager();

    protected EventLogger eventLogger = new EventLogger(this);

    protected ISimTimer iSimTimer = null;

    protected String ipcStrategyName = null;

    protected List startupHooks = new LinkedList(); // list of Runnable
    protected List exitHooks = new LinkedList();    // list of Runnable

    protected List nameListeners = new LinkedList();

    protected Date startupDate = new Date();

    /**
     * Standard constructor.
     */
    public IXAgent() {
	this(true);
    }

    /**
     * Constructor for subclasses that need to create a second agent.
     */
    protected IXAgent(boolean setAgent) {
	Debug.noteln("Creating I-X agent, standalone = " + setAgent);
	if (setAgent) {
	    if (thisAgent != null)
		throw new Error
		    ("Attempt to create multiple IX Agents in one VM");
	    else
		thisAgent = this;
	}
	knownAgents.put(this, Boolean.TRUE);
    }

    /**
     * Method called by main(String[] argv) to perform the initialization
     * sequence common to all I-X agents.  However, all it does directly
     * is to call {@link #do_mainStartup(String[])} inside a "catch"
     * that reports any exception thrown.
     */
    public void mainStartup(String[] argv) {
	try {
	    do_mainStartup(argv);
	}
	catch (Throwable t) {
	    Debug.displayException(t);
	    if (this != thisAgent || !Parameters.isInteractive())
		throw new RethrownException(t);
	}
    }

    /**
     * The main body of {@link #mainStartup(String[])}.
     * It is called inside a "catch" that reports
     * any exception thrown.
     */
    protected void do_mainStartup(String[] argv) {

	// Parse command-line arguments
	Parameters.processCommandLineArguments(argv);

	// Have agent to look at any arguments it's interested in.
	processCommandLineArguments();

	// Finish setting things up
	startup();

	// Start IPC server, if any
	if (ipcStrategyName != null)
	    startServer(getAgentIPCName(), ipcStrategyName);

	// Start event logging, if desired
	if (Parameters.haveParameter("log-directory")
	      // Only the main agent should log.
	      && this == IXAgent.thisAgent) {
	    eventLogger.install();
	    eventLogger.startLogging();
	}

	setupISimTimer();

        Util.runHooks("startup", startupHooks);

	// All command-line arguments should now have been processed.
	Parameters.checkParameterUse();

    }

    protected void setupISimTimer() {
	if (Parameters.haveParameter("isim-agent-name")
              // Only the main agent should start a timer
	      && this == IXAgent.thisAgent) {
	    try {
		iSimTimer = ISimTimer.getISimTimer(this);
	    }
	    catch (Throwable t) {
		Debug.displayException("Can't set up I-Sim timer", t);
		if (!Parameters.isInteractive())
		    throw new RethrownException(t);
	    }
	}
    }

    /**
     * Method called by the {@link #mainStartup(String[] argv)} method
     * to perform any setup and initialization that should take place after
     * this agent's constructor has been called and command-line arguments
     * have been processed. <p>
     *
     * At present, this method does nothing, and all the work is done
     * in subclasses.
     */
    protected void startup() {
    }

    // /\/: This comment needs to document more parameters or else none?
    /**
     * Handles command-line arguments common to all I-X agents.
     * At present, this method also makes any standard I-X changes
     * to the look and feel.
     *
     * <p>The following are handled directly my this method:
     * <pre>
     *    -debug=<i>boolean</i>
     *    -ipc=<i>name</i>
     *    -ipc-name=<i>name</i>
     *    -symbol-name=<i>name</i>
     *    -display-name=<i>name</i>
     * </pre>
     * <tt>debug</tt> is used to set <code>Debug.on</code>.
     *
     * <p>The <i>name</i> in <tt>-ipc=</tt><i>name</i> argument will be
     * interpreted by the <code>IPC.makeCommunicationStrategy(String)</code>
     * method.
     *
     * <p>The <i>name</i> in the <tt>-icp-name=</tt><i>name</i>
     * argument sets the name that this agent calls itself for IPC
     * and that is returned by the <code>getAgentIPCName()</code>
     * method.
     *
     * <p><i>Needs further explanation of ipc-name, and of symbol-name
     * and display-name. ...</i>
     *
     * <p>The <code>processCommandLineArguments</code> method of
     * this agent's contact manager is called to handle arguments
     * that list relationships with other agents.
     *
     * @see ix.iface.util.IFUtil#adjustLookAndFeel()
     * @see ix.util.IPC#makeCommunicationStrategy(String methodName)
     * @see ix.ispace.ContactManager#processCommandLineArguments()
     * @see ix.util.Debug#on
     * @see ix.util.Parameters
     */
    protected void processCommandLineArguments() {

	// /\/: Theme change doesn't work if done after objects created.
	ix.iface.util.IFUtil.adjustLookAndFeel();

	Debug.on = Parameters.getBoolean("debug", Debug.on);

	if (Parameters.haveParameter("ipc-name")) {
	    ipcName = Parameters.getParameter("ipc-name");
	    Debug.noteln("Using ipc name", ipcName);
	}
	else if (Parameters.haveParameter("symbol-name")) {
	    ipcName = Parameters.getParameter("symbol-name");
	    Debug.noteln("Using symbol name " + ipcName + " as ipc name");
	}
	else {
	    try { 
		ipcName = Util.getUserName() +"@"+ Util.getHostName();
	    }
	    catch (UnknownHostException e) {
		ipcName = Util.getUserName();
		Util.displayAndWait(null, "Can't get host name; " +
				    "using ipc-name=" + ipcName);
	    }
	}

	symbolName  = Parameters.getParameter("symbol-name", ipcName);

	initialDisplayName = displayName; // /\/ for now

	displayName = Parameters.haveParameter("display-name")
	    ? Parameters.getParameter("display-name")
	    : symbolName + " " + displayName;

	if (Parameters.haveParameter("ipc")) {
	    String ipc = Parameters.getParameter("ipc");
	    if (!(ipc.equals("false")			// from "-no -ipc"
		  || ipc.equals("none"))) 		// from "-ipc=none"
		// N.B. need to do this after setting ipcName
		ipcStrategyName = ipc;
	}

	if (Parameters.getBoolean("use-long-ids", true))
	    Gensym.useUniquePrefix();

	eventLogger.processCommandLineArguments();

	contactManager.processCommandLineArguments();

    }

    public void addStartupHook(Runnable hook) {
        startupHooks.add(hook);
    }

    public void addExitHook(Runnable hook) {
        exitHooks.add(hook);
    }

    /**
     * Installs any extensions specified by the "extension-classes"
     * parameter.  The value of the paremeter must be a comma-separated
     * list of class names.  The class names are processed in the order
     * given, which allows later extensions to modify the effects
     * of earlier ones.
     *
     * <p>Each class must implement the {@link IXAgentExtension} interface
     * and must have a public 1-argument constructor that declares its
     * parameter as IXAgent or an IXAgent subclass.  One instance of
     * each class is constructed, passing this agent as the parameter,
     * and then the instance's {@link IXAgentExtension#installExtension()}
     * method is called.</p>
     *
     * <p>installAgentExtensions is not called by methods in the IXAgent class.
     * Instead, it should be called by instances of IXAgent subclasses
     * at an appropriate point for the type of agent involved.</p>
     */
    public void installAgentExtensions() {
	installAgentExtensions(Parameters.getList("extension-classes"));
    }

    protected void installAgentExtensions(List classNames) {
	for (Iterator i = classNames.iterator(); i.hasNext();) {
	    String className = i.next().toString();
	    Debug.noteln("Installing agent extension", className);
	    try {
		Class c = Class.forName(className);
		IXAgentExtension e =
		    (IXAgentExtension)Util.makeInstance(c, this);
		e.installExtension();
	    }
	    catch (Exception e) {
		throw new RethrownException(e);
	    }
	}
    }

    /**
     * Returns an object that represents the agent.
     * At present, there can be only one "main" IXAgent per VM.
     */
    public static IXAgent getAgent() {
	return thisAgent;
    }

    /**
     * Returns a set containing all IXAgents that have not been
     * garbage-collected.
     */
    public static Set<IXAgent> getKnownAgents() {
	return knownAgents.keySet(); // snapshot instead? /\/
    }

    /**
     * Returns the object used to represent the agent as an IPC "destination".
     * This object is usually, but not necessarily, a string containing the
     * agent's name.  The details will depend on the set of communication
     * strategies that might be used.
     *
     * @see ix.util.IPC
     * @see ix.util.IPC.CommunicationStrategy
     */
    public Object getAgentIPCName() {
	return ipcName;
    }

    /**
     * Returns this agent's symbol name.
     */
    public String getAgentSymbolName() {
	// /\/: Explain "symbol name"
	// /\/: Why not abstract?
	// /\/: Will symbol name and ipc name merge?
	return symbolName;
    }

    /**
     * Changes the agent's symbol name
     */
    public void setAgentSymbolName(String name) {
	String oldName = symbolName;
	ipcName = name;
	symbolName = name;
	displayName = Parameters.haveParameter("display-name")
	    ? Parameters.getParameter("display-name")
	    : symbolName + " " + initialDisplayName;
	fireSymbolNameChanged(oldName, symbolName);
    }

    public void addAgentNameListener(AgentNameListener listener) {
	nameListeners.add(listener);
    }

    public void fireSymbolNameChanged(String oldName, String newName) {
	AgentNameEvent e = new AgentNameEvent(this, oldName, newName);
	for (Iterator i = nameListeners.iterator(); i.hasNext();) {
	    ((AgentNameListener)i.next()).symbolNameChanged(e);
	}
    }

    /**
     * Returns this agent's display name.
     */
    public String getAgentDisplayName() {
	return displayName;
    }

    public Date getAgentStartupDate() {
	return startupDate;
    }

    /**
     * Returns this agent's contact manager.
     */
    public ContactManager getContactManager() {
	return contactManager;
    }

    /**
     * Returns this agents event-logger.
     *
     * @see #log(HistoryEvent)
     */
    public EventLogger getEventLogger() {
	return eventLogger;
    }

    public void log(HistoryEvent event) {
	eventLogger.log(event);
    }

    /**
     * Returns this agent's ISimTimer;
     */
    public ISimTimer getISimTimer() {
	return iSimTimer;
    }

    /**
     * Adds a tool, usually by adding an entry to a "Tools" menu in the GUI.
     * This method throws an exception that says tool additions are not
     * supported; it should be overridden in subclasses that do allow
     * tools.
     *
     * @throws UnsupportedOperationException if called.
     */
    public void addTool(ix.iface.util.ToolController tc) {
	Debug.noteln("Attempt to add tool " + tc.getToolName());
	throw new UnsupportedOperationException
	    ("This agent does not allow tools to be added.");
    }

    /**
     * Returns the tool of the specified name, causing it to be created
     * if it does not already exist.  Note that it returns the tool,
     * not its tool-controller, and that it does not change the tool's
     * visibility.
     *
     * <p>The method supplied by the IXAgent class throws an
     * exception; it should be overridden in subclasses that
     * do allow tools.
     *
     * @throws UnsupportedOperationException if called.
     */
    public Object ensureTool(String toolName) {
	Debug.noteln("Attempt to ensure tool", toolName);
	throw new UnsupportedOperationException
	    ("This agent does not support GUI tools.");
    }


    /*
     * IPC server setup and input handling.
     */

    /**
     * Set the main (global) IPC communication strategy and set up
     * to receive messages by calling the strategy's setupServer
     * method.  If the <code>"enqueue-incoming-messages"</code>
     * parameter is true, it uses a BufferedMessageListener so
     * that the thread that supplies a message need't wait for the
     * message to be processed by the panel; otherwise, a plain
     * MessageListener is used.  In both cases, the listener delivers
     * the message by calling {@link #pre_handleInput(IPC.InputMessage)},
     * but a buffered listener has its own thread that waits for that
     * call to return before looking for the next message.
     *
     * @see ix.util.IPC
     * @see ix.util.IPC#setupServer(Object, IPC.MessageListener)
     * @see ix.util.IPC.BufferedMessageListener
     */
    protected void startServer(Object agentName, String strategyName) {
	// Create the communication strategy.
	IPC.setCommunicationStrategy(strategyName);
	// Set up to receive messages.
	if (Parameters.getBoolean("enqueue-incoming-messages", false)) {
	    IPC.setupServer(agentName, new IPC.BufferedMessageListener() {
		public void handleMessage(IPC.InputMessage message) {
		    pre_handleInput(message);
		}
	    });
	}
	else {
	    // Assume that the thread that calls the listener can
	    // afford to wait for the call to return, and that its
	    // waiting will not stop any messages from being sent.
	    IPC.setupServer(agentName, new IPC.MessageListener() {
		public synchronized void 
		       messageReceived(IPC.InputMessage message) {
		    pre_handleInput(message);
		}
	    });
	}
    }

    /**
     * Gets the message before the <code>handleInput</code> method
     * and ensures that <code>handleInput</code> is called in the
     * AWT event dispatching thread.  This is perhaps a temporary
     * measure.  It also marks the message as external and (once
     * in the event dispatching thread) catches any Errors or Exceptions
     * thrown out of input handling and reports them to the user.
     */
    protected synchronized 
              void pre_handleInput(final IPC.InputMessage message) {
	// Mark the message as external.
	// /\/: Probably there should be some sender information instead.
	message.setAnnotation("is-external", Boolean.TRUE);
	// Switch to the AWT event thread.
	// /\/: We have to do this somewhere, and it's awkward
	// to do it anywhere else for reports.  So for now ...
	Util.swingAndWait(new Runnable() {
	    public void run() {
		try {
		    Debug.expect(SwingUtilities.isEventDispatchThread(),
				 "Message handling begins in wrong thread");
		    notePossibleNewContact(message); // s.b. elsewhere?  /\/
		    IPC.fireMessageReceived(message); // s.b. elsewhere?  /\/
		    handleInput(preprocessInput(message));
		}
		catch (Throwable t) {
		    Debug.noteException(t);
		    reportInputException(message, t);
		}
	    }
	});

    }

    /**
     * A chance to modify or replace a message on its way in.  This method
     * is called by {@link #pre_handleInput(IPC.InputMessage)}
     * before it calls {@link #handleInput(IPC.InputMessage)}.
     *
     * <p>The method in the IXAgent class currently renames
     * {@link ix.util.lisp.ItemVar}s in issue and activity patterns.
     *
     * @see ix.util.lisp.ItemVar#renameItemVars(Object)
     */
    protected IPC.InputMessage preprocessInput(IPC.InputMessage message) {
	if (!Parameters.getBoolean("rename-variables-in-input", true))
	    return message;
	if (message.getContents() instanceof TaskItem) {
	    TaskItem item = (TaskItem)message.getContents();
	    LList pattern = item.getPattern();
	    if (pattern != null) {
		LList newPat = (LList)ItemVar.renameItemVars(pattern);
		item.setPattern(newPat);
	    }
	}
	return message;
    }

    /**
     * Called by the pre_handleInput method to put up a dialog
     * describing an Error or Exception and the message that caused it
     * to be thrown.
     */
    protected void reportInputException(IPC.InputMessage message,
					Throwable t) {
	Object contents = message.getContents();
	String sender = "unknown sender"; // unless we find something better
	if (contents instanceof Sendable) {
	    Sendable s = (Sendable)contents;
	    if (s.getSenderId() != null)
		sender = s.getSenderId().toString();
	}
	JOptionPane.showMessageDialog(null,
	    new Object[] {
	        "Problem handling message from " + sender,
		"Contents class " + contents.getClass().getName(),
		Debug.describeException(t)},
	    "Error during message handling",
	    JOptionPane.ERROR_MESSAGE);
    }

    /**
     * Tells this agent's contact manager about the sender-id of the
     * message (if the id can be determined) in case it represents
     * a new contact.
     */
    public void notePossibleNewContact(IPC.InputMessage message) {
	Object contents = message.getContents();
	if (contents instanceof Sendable) {
	    Name senderId = ((Sendable)contents).getSenderId();
	    if (senderId != null)
		// /\/: Contact manager should work with Names, not Strings.
		contactManager.noteAgent(senderId.toString());
	}
    }

    /**
     * Handles external input in the form of an IPC.InputMessage
     * that contains an object such as an Issue, Activity, or
     * Report.
     *
     * It calls one of <code>handleNewIssue</code>, 
     * <code>handleNewReport</code>, etc as appropriate.
     */
    public void handleInput(IPC.InputMessage message) {
	Object contents = message.getContents();
        if (contents instanceof Issue) {
            handleNewIssue((Issue)contents);
        }
	else if (contents instanceof Activity) {
	    handleNewActivity((Activity)contents);
	}
	else if (contents instanceof Constraint) {
	    handleNewConstraint((Constraint)contents);
	}
        else if (contents instanceof Report) {
            handleNewReport((Report)contents);
        }
	else if (contents instanceof ChatMessage) {
	    handleNewChatMessage((ChatMessage)contents);
	}
        else {
            displayMessage("Unexpected message contents " + contents);
        }
    }

    /**
     * Handles new issues from external sources.  Subclasses will
     * usually redefine this method.
     */
    public void handleNewIssue(Issue issue) {
        displayMessage(Reporting.issueDescription(issue));
    }

    /**
     * Handles new activities from external sources.  Subclasses will
     * usually redefine this method.
     */
    public void handleNewActivity(Activity activity) {
        displayMessage(Reporting.activityDescription(activity));
    }

    /**
     * Handles new constraints from external sources.  Subclasses will
     * usually redefine this method.
     */
    public void handleNewConstraint(Constraint constraint) {
        displayMessage(Reporting.constraintDescription(constraint));
    }

    /**
     * Handles new reports from external sources.  Subclasses will
     * usually redefine this method.
     */
    public void handleNewReport(Report report) {
        displayMessage(Reporting.reportDescription(report));
    }

    /**
     * Utility for sending a "Received" report about an issue
     * or activity.
     */
    protected void handleReceivedReport(TaskItem item) {
	if (!Parameters.getBoolean("send-received-reports", true))
	    return;
	if (item.getReportBack() == YesNo.YES) {
	    Name ourName = Name.valueOf(getAgentIPCName());
	    Report rep = new Report();
	    rep.setReportType(ReportType.PROGRESS);
	    rep.setText("Received by " + ourName +
			", " + Reporting.dateString());
	    rep.setRef(item.getRef());
	    rep.setSenderId(ourName);
	    IPC.sendObject(item.getSenderId() // whoever sent this item to us
			       .toString(),   // /\/ 'cause it's a Name
			   rep);
	}
    }

    /**
     * Handles new chat messages from external sources.  Subclasses will
     * usually redefine this method.
     */
    public void handleNewChatMessage(ChatMessage message) {
        displayMessage(Reporting.chatMessageDescription(message));
    }

    /*
     * Default message display
     */

    /**
     * A text area in a separate frame used to display information about
     * incoming messages.
     */
    protected TextAreaFrame textFrame = null;

    /**
     * Adds the specified string to the default message display.
     */
    protected void displayMessage(String message) {
	if (textFrame == null)
	    textFrame = new TextAreaFrame("Messages to " + getAgentIPCName());
	textFrame.appendLine(message);
	textFrame.setVisible(true);
    }

}

// Issues:
// * Should keep a record of how IXAgent.getAgent() is used.
// * If a startup() method makes the GUI visible, subclass startup()
//   methods that call super.startup() will need to know this so they can
//   ensure that subsequent GUI stuff happens in the event-handling thread.
//   It might be better to have a separate makeVisible() that happens
//   after startup(); and maybe startup() should be renamed to something
//   like completeSetup().
