/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Aug 31 20:05:24 2004 by Jeff Dalton
 * Copyright: (c) 2003, AIAI, University of Edinburgh
 */

package ix.util.rdf;

import java.io.*;
import java.net.URL;
import java.util.*;

import com.hp.hpl.jena.rdf.model.*;
// import com.hp.hpl.jena.vocabulary.*;

import ix.util.*;
import ix.util.xml.*;
import ix.util.reflect.*;
import ix.util.lisp.*;

public class RdfTranslator implements Loader, Saver {

    protected ClassSyntax classSyntax;

    public RdfTranslator() {
	this.classSyntax = Rdf.classSyntax();
    }

    /* ************************ *
     *      Objects to RDF      *
     * ************************ */

    public String objectToString(Object obj) {
	return modelToString(objectToModel(obj));
    }

    public String modelToString(Model model) {
	OutputStream out = new ByteArrayOutputStream();
	model.write(out, "RDF/XML-ABBREV");
	try { out.flush(); }
	catch (IOException e) { throw new RethrownException(e); }
	return out.toString();
    }

    public Model objectToModel(Object obj) {
	ObjectModel om = new ObjectModel(this, obj);
	Model model = om.getModel();
	return model;
    }

    public void writeObject(Object obj, File file) throws IOException {
	// For the Saver interface.
	Model model = objectToModel(obj);
	OutputStream out = new FileOutputStream(file);
	model.write(new BufferedOutputStream(out),
			"RDF/XML-ABBREV");
	out.flush();
	out.close();
    }

    /* ************************** *
     *      Objects from RDF      *
     * ************************** */

    public Object objectFromString(String rdf) {
	return objectFromModel(modelFromString(rdf));
    }

    public Model modelFromString(String rdf) {
	// /\/: Unfortunately, it wants a base URI for converting
	// and relative URIs in the string.
	Model model = ModelFactory.createDefaultModel();
	return model.read(new StringReader(rdf), null);
    }

    public Object objectFromModel(Model model) {
	ModelParser mp = new ModelParser(this, model);
	return mp.getRootObject();
    }

    public Object readObject(URL url) {
	// For the Loader interface.
	Model model = ModelFactory.createDefaultModel();
	return objectFromModel(model.read(url.toString()));
    }

    /**
     * For testing.
     */
    public static void main(String[] argv) throws IOException {
	Parameters.processCommandLineArguments(argv);

	RdfTranslator rdft = new RdfTranslator();

	String fileName = Parameters.getParameter("file");
	if (fileName != null) {
	    Object obj = XML.readObject(fileName);
	    rdft.test(obj);
	    return;
	}

	// Empty model
	String s = rdft.modelToString(ModelFactory.createDefaultModel());
	System.out.println(s);
	Model model = rdft.modelFromString(s);
	System.out.println(rdft.modelToString(model));

	// Tests
	rdft.test(null);
	rdft.test("Applie pie is nice to eat");
	rdft.test("Apple <i>pie</i> is nice to eat.");
	rdft.test(ix.icore.Priority.HIGH);
	rdft.test(new ix.icore.Issue());
	rdft.test(new Long(12));
	rdft.test(Lisp.NIL);
	rdft.test(Lisp.list("apple", "pie"));
	rdft.test(Lisp.readFromString("(2 or 3 pies)"));
	rdft.test(Lisp.list(new ix.icore.Issue(), new ix.icore.Activity()));
	{
	    Map m = new TreeMap();
	    rdft.test(m);	// empty Map
	    m.put("apple", "red");
	    m.put("grass", "green");
	    m.put("snow", "white");
	    rdft.test(m);	// 3-entry Map
	    {
		Map m2 = new TreeMap();
		m2.put("tasty", "apple");
		rdft.test(Lisp.list(m, m2)); // list of 2 Maps
	    }
	}
	
    }

    void test(Object obj) {
	System.out.println("- - - - - - - - - - - - - - - - - - - -");
	StructuralEquality structEq = new StructEq(classSyntax);
	try {
	    Model m = objectToModel(obj);
	    System.out.println(modelToString(m));
	    // m.write(System.out, "N3"); // write the same thing in N3 syntax
	    // m.write(System.out, "N3-TRIPLE");
	    // m.write(System.out, "N3-PLAIN");
	    Object back = objectFromModel(m);
	    System.out.println(back);
	    Debug.expect(structEq.equal(obj, back), "Not structurally equal");
	}
	catch (Throwable t) {
	    Debug.displayException(t);
	}
    }

    static class StructEq extends StructuralEquality {
	StructEq(ClassSyntax s) { super(s); }
	protected boolean haveEquivalentClasses(Object a, Object b) {
	    Class c_a = a.getClass();
	    Class c_b = b.getClass();
	    return c_a == c_b
		|| (a instanceof List && b instanceof List)
		|| (a instanceof Set && b instanceof Set)
		|| (a instanceof Map && b instanceof Map);
	}
    }

}
