/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat May 22 01:30:27 2004 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.util.owls;

import org.mindswap.owl.*;
import org.mindswap.owls.*;
import org.mindswap.owls.io.*;
import org.mindswap.owls.service.*;
import org.mindswap.owls.process.Process; // vs java.lang.Process /\/
import org.mindswap.owls.process.*;

import java.net.URI;
import java.net.URISyntaxException;
import java.io.*;
import java.util.*;

import ix.icore.*;
import ix.icore.domain.*;
import ix.iplan.ServiceSymbols;
import ix.iface.domain.LTF_Writer;
import ix.util.*;
import ix.util.xml.*;
import ix.util.lisp.*;

/**
 * Translates OWL-S service descriptions to I-X Refinements.
 */
public class ServiceTranslator implements ServiceSymbols {

    StableHashMap idToService = new StableHashMap();

    NameMapper nameMap = new LCNameMapper();

    List translatedServices = new LinkedList();

    Symbol S_TO = Symbol.intern("to");
    Symbol S_TYPE = Symbol.intern("type");
    Symbol S_NAME_MAP = Symbol.intern("name-map");

    int VAR = 0, TYPE = 1;

    public ServiceTranslator() {
    }

    public NameMapper getNameMapper() {
	return nameMap;
    }

    public Collection getTranslatedServices() {
	return idToService.values();
    }

    public Service getServiceForRefinement(Refinement r) {
	String id = nameMap.longName(r.getName().toString());
	Service s = (Service)idToService.get(id);
	if (s != null)
	    return s;
	else
	    throw new IllegalArgumentException
		("Can't find a service for " + r);
    }

    public Domain readServiceDomain(String servicesURI) {
	return serviceListToDomain(readServiceList(servicesURI));
    }

    public Domain serviceListToDomain(List services) {
	Domain dom = new Domain();
	for (Iterator i = services.iterator(); i.hasNext();) {
	    Service s = (Service)i.next();
	    dom.addRefinement(serviceToRefinement(s));
	}
	// Add the short-to-long map as an annotation.
	dom.setAnnotation(S_NAME_MAP, nameMap.getShortToLongMap());
	return dom;
    }

    public List readServiceList(String servicesURI) {
	URI uri = makeURI(servicesURI);
	// OWLSReader reader = OWLSFactory.createOWLSReader();
	OWLSReader reader = new ServiceReader();
	List services = reader.readAll(uri);
	return services;
    }

    public URI makeURI(String uri) {
	try { return new URI(uri); }
	catch (URISyntaxException e) {
	    throw new RethrownException(e);
	}
    }

    public Refinement serviceToRefinement(Service s) {
	Debug.noteln("- - - Service to translate - - -");
	Debug.noteln("Service", s);
	Debug.noteln("File URI", s.getFileURI());
	Debug.noteln("OWL-S version", s.getOWLSVersion());
	Debug.noteln("Service URI", s.getURI());

	Refinement r = new Refinement();

	String rname = nameMap.mapLongName(s.getURI());

	r.setName(rname);
	r.setPattern(Lisp.list(rname));	// for now

	Process proc = s.getProcess();
	InputList inputs = proc.getInputs();
	OutputList outputs = proc.getOutputs();

	LList inPairs = makeParameterPairs(inputs);
	LList outPairs = makeParameterPairs(outputs);

	LList pattern =
	    new Cons(Symbol.intern(rname),
		     inPairs.mapNth(VAR)
		         .append(new Cons(S_TO,
					  outPairs.mapNth(VAR))));
	r.setPattern(pattern);

	r.setVariableDeclarations
	    (makeVariableDeclarations(inPairs)
	       .append(makeVariableDeclarations(outPairs)));

	r.setAnnotation(S_INPUT_OBJECTS, inPairs);
	r.setAnnotation(S_OUTPUT_OBJECTS, outPairs);

	r.setAnnotation(S_OWL_S_PROCESS,
			s.getProcess().getURI().toString());

	r.setConstraints
	    (makeConstraints("condition", inPairs)
	         .append(makeConstraints("effect", outPairs)));

	return r;
    }

    // Returns a ((var type) ...) list.
    LList makeParameterPairs(ParameterList parameters) {
	LListCollector result = new LListCollector();
	for (Iterator i = parameters.iterator(); i.hasNext();) {
	    Parameter p = (Parameter)i.next();
	    OWLResource type = p.getType();
	    String pName = nameMap.mapLongName(p.getURI());
	    String typeName = nameMap.mapLongName(type.getURI());
	    result.add
		(Lisp.list(Symbol.intern("?" + pName),
			   Symbol.intern(typeName)));
	}
	return result.contents();
    }

    LList makeVariableDeclarations(LList parameterPairs) {
	return parameterPairs.mapcar(new Function1() {
	    public Object funcall(Object arg) {
		LList pair = (LList)arg;
		ItemVar var = (ItemVar)pair.get(VAR);
		return new VariableDeclaration(var);
	    }
	});
    }

    LList makeConstraints(final String relation, LList parameterPairs) {
	return parameterPairs.mapcar(new Function1() {
	    public Object funcall(Object arg) {
		LList pair = (LList)arg;
		ItemVar var = (ItemVar)pair.get(VAR);
		Symbol type = (Symbol)pair.get(TYPE);
		return makeConstraint(relation, var, type);
	    }
	});
    }

    Constraint makeConstraint(String relation, ItemVar var, Symbol type) {
	return new Constraint
	    ("world-state",
	     relation,		// "condition" or "effect"
	     Lisp.list
	         (new PatternAssignment(Lisp.list(S_TYPE, var), type)));
    }

    public static void main(String[] argv) throws Exception {
	Parameters.processCommandLineArguments(argv);
	ServiceTranslator t = new ServiceTranslator();
	if (Parameters.haveParameter("service")) {
	    String serviceURI = Parameters.getParameter("service");
	    URI uri = new URI(serviceURI);
	    // OWLSReader reader = OWLSFactory.createOWLSReader();
	    OWLSReader reader = new ServiceReader();
	    Service s = reader.read(uri);
	    Refinement r = t.serviceToRefinement(s);
	    LTF_Writer ltf = new LTF_Writer();
	    ltf.writeRefinement(r, new OutputStreamWriter(System.out));
	}
	else if (Parameters.haveParameter("service-list")) {
	    String servicesURI = Parameters.getParameter("service-list");
	    Domain d = t.readServiceDomain(servicesURI);
	    LTF_Writer ltf = new LTF_Writer();
	    ltf.writeDomain(d, new OutputStreamWriter(System.out));
	    // System.out.println(XML.objectToXMLString(d));
	}
	else
	    throw new UnsupportedOperationException
		("No recognizable command");
    }

}
