/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat May 22 19:07:38 2004 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.util.owls;

import org.mindswap.owls.*;
// import org.mindswap.owls.io.*;
import org.mindswap.owls.service.*;
import org.mindswap.owls.process.Process; // because of java.lang.Process
import org.mindswap.owls.process.*;

import org.mindswap.owl.OWLResource;

import com.hp.hpl.jena.rdf.model.*;

import java.net.*;
// import java.io.*;
import java.util.*;

import ix.ip2.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.icore.plan.*;
import ix.icore.process.*;
import ix.icore.process.*;
import ix.iplan.ServiceSymbols;
import ix.util.*;
import ix.util.xml.*;
import ix.util.lisp.*;

/**
 * Translates an I-X Plan to an OWL-S service description
 */
public class PlanToOWLS implements ServiceSymbols {

    Ip2 ip2;			// holds the plan and domain

    NameMapper nameMap;		// lets us get back long URIs

    CompositeProcess topLevelProcess; // /\/ convenient global

    Map activityToProcessMap = new HashMap();

    public PlanToOWLS(Ip2 ip2) {
	this.ip2 = ip2;
	Map shortToLong = (Map)ip2.getDomain().getAnnotation(S_NAME_MAP);
	this.nameMap = shortToLong != null
	    ? new NameMapper(shortToLong)
	    : null;
    }

    public Service planToService() {
	Resource planResource =
	    ResourceFactory.createResource("#composed-service");

	// Create and connect Service down to Process
	Service service = OWLSFactory.createService(planResource);
	ProcessModel processModel = OWLSFactory.createProcessModel();
	CompositeProcess process = 
	    OWLSFactory.createCompositeProcess
	       (ResourceFactory.createResource("#main-process"));
	service.setProcessModel(processModel);
	// not needed: processModel.setService(service);
	processModel.setProcess(process);
	process.setService(service);

	// /\/: Make the process easily accessible to all methods
	this.topLevelProcess = process;

	// For now, we always use a Sequence.  /\/
	Sequence seq = OWLSFactory.createSequence();
	process.setComposedOf(seq);

	// Get a list of the atomic steps.
	// /\/: Some might be compositite, really, though
	// we don't yet model that.
	List atomicSteps = listAtomicSteps();

	// We can add to the component list of the Sequence.
	ProcessComponentList components = seq.getComponents();
	for (Iterator i = atomicSteps.iterator(); i.hasNext();) {
	    PNode node = (PNode)i.next();
	    Activity act = (Activity)((AgendaItem)node).getAbout();
	    String uri = getProcessURI(node);
	    Resource res = ResourceFactory.createResource(uri);
	    Process proc = OWLSFactory.createAtomicProcess(res);
	    activityToProcessMap.put(act, proc);
	    components.add(proc);
	}

	// Add dataflow...
	addDataFlow(process, atomicSteps);

	return service;
    }

    boolean isServiceNode(PNode node) {
	// /\/: This test still isn't quite right.
	String shortName = getServiceRef(node);
	boolean isMapped = nameMap == null
	    || nameMap.hasLongName(shortName);
	return isMapped
	    && ip2.getDomain().getNamedRefinement(shortName) != null;
    }

    String getProcessURI(PNode node) {
	String shortName = getServiceRef(node);
	Refinement ref = ip2.getDomain().getNamedRefinement(shortName);
	if (ref != null) {
	    String uri = (String)ref.getAnnotation(S_OWL_S_PROCESS);
	    if (uri != null)
		return uri;
	}
	return shortName + "#process"; // invent something /\/
    }

    String getServiceRef(PNode node) {
	return node.getPattern().car().toString();
    }

    // /\/: This is a over-simple way to determine the steps in
    // the composite process.  In general, we won't alway want
    // to have a plain sequence, and we will want to be able to
    // handle services that are already composite.
    List listAtomicSteps() {
	List nodeEnds = getNodeEnds();
	List sorted = new TopologicalSorter() {
	    protected Collection getChildren(Object nodeEnd) {
		return ((PNodeEnd)nodeEnd).getSuccessors();
	    }
	}.sort(nodeEnds);
	List result = new LinkedList();
	for (Iterator i = sorted.iterator(); i.hasNext();) {
	    PNodeEnd ne = (PNodeEnd)i.next();
	    if (ne.getEnd() == End.END)	// Look at only BEGIN ends.
		continue;
	    PNode node = ne.getNode();
	    // /\/: What if a service node isn't childless?
	    if (node.getChildren().isEmpty() && isServiceNode(node))
		result.add(node);
	}
	return result;
    }

    List getNodeEnds() {
	return ((Ip2ModelManager)ip2.getModelManager()).getNodeEnds();
    }

    /*
     * Data-Flow
     */

    void addDataFlow(CompositeProcess process,
		     List atomicSteps) {
	ProcessDataFlow pdflow = new ProcessDataFlow(atomicSteps);
	List sameValueLists = pdflow.getSameValueLists();
	List initialInputs = pdflow.getInitialInputs();
	List finalOutputs = pdflow.getFinalOutputs();

	Debug.noteln("Data-flow", XML.objectToXMLString(sameValueLists));
	Debug.noteln("Initial inputs", XML.objectToXMLString(initialInputs));
	Debug.noteln("Final outputs", XML.objectToXMLString(finalOutputs));

	addParamters(process.getInputs(), initialInputs);
	addParamters(process.getOutputs(), finalOutputs);

	List fullSameList = pdflow.makeFullSameList
	                      (initialInputs, finalOutputs);
	DataFlow flow = process.getDataFlow();
	for (Iterator i = fullSameList.iterator(); i.hasNext();) {
	    List sameList = (List)i.next();
	    if (sameList.size() > 1)
		flow.add(makeDataFlowElement(sameList));
	}
    }

    void addParamters(ParameterList params, List ixParams) {
	for (Iterator i = ixParams.iterator(); i.hasNext();) {
	    ProcessParameter p = (ProcessParameter)i.next();
	    Parameter par = makeOWLSParameter(p);
	    par.setType(getOWLSType(p));
	    params.add(par);
	}
    }

    DataFlowElement makeDataFlowElement(List sameList) {
	DataFlowElement dfe = OWLSFactory.createDataFlowElement();
	for (Iterator i = sameList.iterator(); i.hasNext();) {
	    ProcessParameter p = (ProcessParameter)i.next();
	    dfe.add(makeParameter(p));
	}
	return dfe;
    }

    Parameter makeParameter(ProcessParameter p) {
	Parameter par = makeOWLSParameter(p);
	Activity act = p.getActivity();
	if (act == null) {
	    // Assume we have an initial input or final output.
	    par.setProcess(topLevelProcess);
	}
	else {
	    Process proc = (Process)activityToProcessMap.get(act);
	    Debug.expect(proc != null, "no process for", act);
	    par.setProcess(proc);
	}
	return par;
    }

    Parameter makeOWLSParameter(ProcessParameter p) {
	ItemVar var = (ItemVar)p.getName();
	String name = var.toString().substring(1);
	String fullName = nameMap != null && nameMap.hasLongName(name)
	    ? nameMap.longName(name)
	    : "#" + name;
	Resource nameResource = ResourceFactory.createResource(fullName);
	return p.isInput()
	    ? (Parameter)OWLSFactory.createInput(nameResource)
	    : (Parameter)OWLSFactory.createOutput(nameResource);
    }

    OWLResource getOWLSType(ProcessParameter p) {
	String type = p.getType().toString();
	String fullName = nameMap != null && nameMap.hasLongName(type)
	    ? nameMap.longName(type)
	    : type;
	return OWLSFactory.createOWLResource
	          (ResourceFactory.createResource(fullName));
    }

    /*
     * Main program for testing
     */

    public static void main(String[] argv) throws Exception {

	Debug.off();

	Ip2 ip2 = new ix.test.PlainIp2();
	ip2.mainStartup(argv);	// should load a plan

	Debug.on = Parameters.getBoolean("debug", true);

	PlanToOWLS owls = new PlanToOWLS(ip2);
	Service s = owls.planToService();

	// OWLSWriter writer = OWLSFactory.createOWLSWriter("1.0");
	ProcessWriter writer = new ProcessWriter();

	writer.write(s, System.out);

    }

}
