/****************************************************************************
 * A graph with layout and view information.
 *
 * @author Jussi Stader
 * @version 2.4
 * Updated: Mon Mar 26 13:33:35 2007
 * Copyright: (c) 2002, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iview.igraph;


import java.util.*;
import java.awt.Point;
import javax.swing.*;
import java.io.*;
import lt.monarch.graph.*;
//import lt.monarch.graph.plugins.*;
import lt.monarch.graph.view.*;

import ix.*;
import ix.util.*;
import ix.util.lisp.*;
//import ix.test.xml.NodePosition;
import ix.iface.ui.*;
import ix.iface.ui.util.*;
import ix.icore.domain.*;
import ix.iview.domain.*;

/**
 * A graph with layout and view information.
 */
public class IGraph extends Object implements Named {
  public Class nodeViewType = NodeView.class;
  public Class nodeType = Object.class;
  //public Class baseObjectClass;
  public String name;
  //public Object baseObject;
  public LListCollector nodes = new LListCollector();

  private HashMap canvasPoints;
  private List autoPoints;

  protected GraphView view;

  /**
   *Creates a new graph object with the given name and information from the 
   *given graph view. Notes each model node with its location.
   *@param theName the graph's name
   *@param graphView the GraphView object that holds the MGraph information
   */
  public IGraph(String theName, GraphView graphView) {
    super();
    name = theName;
    view = graphView;
    refreshGraph(graphView);
  }

  /**
   *Creates a new graph object with the given name and information from the 
   *given graph view. Notes each model node with its location.
   *@param theName the graph's name
   */
  public IGraph(String theName) {
    super();
    name = theName;
  }

  /**
   *Creates a new graph object with the given name and position information
   * from the given map. Notes each model node with its location.
   *@param theName the graph's name
   *@param cpGraph the graph's canvas points in a hash map
   */
  public IGraph(String theName, HashMap cpGraph) {
    super();
    name = theName;
    setCanvasPoints(cpGraph);
  }

  /**
   *Creates a new graph object with the given name and position information
   * from the given map. Notes each model node with its location.
   *@param theName the graph's name
   *@param cpGraph the graph's canvas points in a list of NodePoint objects
   */
  public IGraph(String theName, List cpGraph) {
    super();
    HashMap map = pointsListToMap(cpGraph);
    name = theName;
    setCanvasPoints(map);
  }



  /**
   *Refreshes the IGraph from the information in the graphView.
   *@param graph the GraphView object that holds the MGraph information
   */
  public void refreshGraph(GraphView graphView) {
    view = graphView;
    //Debug.noteln("IGraph: refreshing graph");
    Iterator nodeViews = graphView.nodeLayer.getViews();
    nodes.clear();
    NodeView nodeView = null;
    NodeSpec node = null;
    Point position;
    while (nodeViews.hasNext()) {
      nodeView = (NodeView)nodeViews.next();
      position = nodeView.getPosition();
      node = (NodeSpec)nodeView.getModelNode();
      //Debug.noteln(" put node/pos " + node.toString() + position.toString());
      nodes.add(new GraphNodeComponent(position, node));
    }
    if (nodeView != null) nodeViewType = nodeView.getClass();
    if (node != null) nodeType = node.getClass();
  }

  public Point getPosition(Object node) {
    GraphNodeComponent gnc;
    try {
      gnc = (GraphNodeComponent)node;
      return gnc.position;
    }
    catch (ClassCastException cce) {
      gnc = findNode(node);
      if (gnc == null) {
	//new graph may have canvas points stored
	try {
	  if (canvasPoints != null) {
	    NodeSpec spec = (NodeSpec)node;
	    java.util.List pointList =  (List)canvasPoints.get(spec.getId());
	    if (pointList != null)
	      return listToPoint(pointList);
	  }
	  return null;
	}
	catch (Exception e) {
	  //(new Throwable()).printStackTrace();   
	  Debug.noteln("IG: getPosition cannot do node " + node.toString());
	  return null;
	}
      }
      else return gnc.position;
    }
  }

  public void setCanvasPoints(HashMap canvasPoints) {
    this.canvasPoints = canvasPoints;
  }

  public List updateCanvasPoints() {
    canvasPoints = readCanvasPoints();
    return pointsMapToList(canvasPoints);
  }

  public List getCanvasPoints() {
    //if (canvasPoints == null) readCanvasPoints();
    return pointsMapToList(canvasPoints);
  }
  public List getGraphCanvasPoints() {
    //if (canvasPoints == null) readCanvasPoints();
    return pointsMapToList(readCanvasPoints());
  }

  private java.util.List pointToList(Point point) {
    ArrayList list = new ArrayList();
    list.add(new Double(point.getX()));
    list.add(new Double(point.getY()));
    return list;
  }
  private Point listToPoint(java.util.List list) {
    if (list == null) return null;
    Double x = (Double)list.get(0);
    Double y = (Double)list.get(1);
    Point point = new Point(x.intValue(), y.intValue());
    return point;
  }

  private HashMap pointsListToMap(List pointsList) {
    if (pointsList == null) return null;
    HashMap map = new HashMap();
    for (Iterator i=pointsList.iterator(); i.hasNext();) {
      Object point = i.next();
      try {
	NodePosition np = (NodePosition)point;
	ArrayList list = new ArrayList();
	list.add(new Double(np.getX()));
	list.add(new Double(np.getY()));
	map.put(np.getName(), list);
      }
      catch (Exception e) {}
    }
    return map;
  }

  private List pointsMapToList(HashMap pointsMap) {
    if (pointsMap == null) return null;
    List pointsList = new ArrayList();
    for (Iterator i = pointsMap.keySet().iterator(); i.hasNext(); ) {
      Name id = (Name)i.next();
      List list = (List)pointsMap.get(id);
      Double x = (Double)list.get(0);
      Double y = (Double)list.get(1);
      pointsList.add(new NodePosition(id, x.doubleValue(), y.doubleValue()));
    }
    return pointsList;
  }


  /**
   * Getting and noting canvas point information from the graph
   */
  public HashMap readCanvasPoints() {
    //Debug.noteln("IG: getting points from graph");
    HashMap points = new HashMap();
    Iterator mNodes = nodes.iterator();
    while (mNodes.hasNext()) {
      try {
	GraphNodeComponent mNode = (GraphNodeComponent)mNodes.next();
	Point position = getPosition(mNode);
	Name id = mNode.baseNode.getId();
	if ((id != null) && (position != null))
	  points.put(id, pointToList(position));
      }
      catch (Exception e) {}
    }
    return points;
  }

  public boolean isSameLayout(Object positions) {
    if (positions == null) {
      List points = getGraphCanvasPoints();
      return ((points == null) || points.isEmpty());
    }
    if (positions instanceof List) return isSameLayoutList((List)positions);
    if (positions instanceof HashMap) 
      return isSameLayoutList(pointsMapToList((HashMap)positions));
    Debug.noteln("IGraph: cannot understand positions of class", 
		 positions.getClass());
    Debug.noteln(" assuming that they are the same as this.");
    return true;
  }
  private boolean isSameLayoutList(List layoutPoints) {
    //no view yet, so nothing can have changed!
    if (view == null) return true;
    List lPoints;
    if (layoutPoints == null) lPoints = new ArrayList();
    else lPoints = layoutPoints;
    try {
      //Debug.noteln("IG: comparing layout", UIUtil.listToDisplay(lPoints));
      //Debug.noteln(" to grph", UIUtil.listToDisplay(getGraphCanvasPoints()));
      return Collect.equalAsSets(getGraphCanvasPoints(), lPoints);
    }
    catch (Exception t) {
      Debug.noteException(t);
      Debug.noteln(" assuming that the two graphs are the same.");
      return true;
    }
  }

  /**
   * Lays out the graph from the current canvas points.
   */
  public void layoutFromCanvasPoints(GraphView graphView) {
    view = graphView;
    //Debug.noteln("IG: laying out from canvas points");
    if ((canvasPoints == null) || (canvasPoints.isEmpty())) return;
    //make sure graphView has nodes in it (in case it is new)
    //graphView.layoutGraph();
    //Debug.noteln(" got canvas points");
    Iterator nodeViews = graphView.nodeLayer.getViews();
    if (!nodeViews.hasNext()) {
      //new graph so make new nodes
      Debug.noteln("IG: graph view has no nodes");
      (new Throwable()).printStackTrace();
    }
    while (nodeViews.hasNext()) {
      try {
	NodeView nodeView = (NodeView)nodeViews.next();
	NodeSpec node = (NodeSpec)nodeView.getModelNode();
	java.util.List pointList = 
	  (java.util.List)canvasPoints.get(node.getId());
	Point position = listToPoint(pointList);
	if (position != null) nodeView.setPosition(position);
	else Debug.noteln("got no position for", node.getId());
      }
      catch (Exception e) {
	Debug.noteln("layout from canvas problem:", e.getMessage());
      }
    }
    //Debug.noteln("...done canvas point layout");
    //make sure this IGraph has everything it needs!
    //refreshGraph(graphView);
  }

  protected GraphNodeComponent findNode(Object node) {
    Enumeration e = nodes.elements();
    while (e.hasMoreElements()) {
      GraphNodeComponent n = (GraphNodeComponent)e.nextElement();
      //Debug.noteln("IGraph: findNode checking node " + n.toString());
      if (UIRefinement.isRef(n.baseNode, node)) return n;
    }
    return null;
  }

  public String getName() {
    return name;
  }
  public Object clone() throws CloneNotSupportedException {
    return super.clone();
  }


  private class GraphNodeComponent extends Object {
    public Point position;
    public double width;
    public double height;
    public NodeSpec baseNode;

    public GraphNodeComponent(Point p, NodeSpec base) {
      super();
      position = p;
      baseNode = base;
    }
    public GraphNodeComponent(Point p, double w, double h, NodeSpec base) {
      this(p, base);
      width = w;
      height = h;
    }
  }

}
