/****************************************************************************
 * An editor for specifying generic constraints in refinements.
 *
 * @author Jussi Stader
 * @version 3.1
 * Updated: Mon Feb  5 09:18:25 2007
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iview;

import javax.swing.*;       
import java.util.*;
import java.awt.Color;
import java.awt.Frame;
import java.awt.Component;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseListener;
import java.awt.event.MouseEvent;
import ix.*;
import ix.util.*;
import ix.util.Debug;
import ix.util.lisp.*;
import ix.icore.domain.*;
import ix.iview.util.*;
import ix.iview.domain.*;
import ix.iface.ui.*;
import ix.iface.ui.util.*;
import ix.iface.domain.*;
import ix.ip2.Ip2ModelManager;

/****************************************************************************
 * An editor for specifying generic constraints in refinements.
 *
 * Constraints are attached to self (the refinement) or to a child. <p>
 *
 * The editor is implemented as a frame using JList objects, a JCheckBox
 * and JButtons.  <p>
 *
 * The editor can be created with no arguments or with a given refinement
 * which will be set as the current refinement whose conditions and effects
 * are to be specified. Subsequently, the refinement in the editor can be set
 * using the setRefinement method.
 *****************************************************************************
 */
public class DefaultConstraintEditor extends ConstraintEditor
{
  //relation items
  

  protected JTextField typeField; 
  protected JTextField relField; 
  protected JRadioButton condBut;
  protected JRadioButton effBut;
  private ButtonGroup group;
  protected JTextField statement;
  
  /**
   * Creates an editor for specifying time point relations.
   * The editor has components for selecting two time points of two
   * sub-activities, and two buttons (Ok, Cancel)
   * "- enter pattern-value statements,",
   * "- select whether you are specifying a condition or an effect,",
   * "- select the node to which the condition/effect applies."});
   */
  public DefaultConstraintEditor(Frame owner){
    super(owner, "Constraint Editor", "Please specify a constraint");
    }
    
  /**
   * Creates an editor for specifying time point relations within the given
   * refinement.
   * As OrderingEditor() but with a given refinement.
   * @param refinement the refinement whose sub-activities are to be related
   */
  public DefaultConstraintEditor(Frame owner, UIRefinement refinement) {
    this(owner);
    setObject(refinement);
  }
  
  
  /**
   * Makes all items on the relations panel.
   * These are a type field, a relations (sub-type) field, and a statement
   * field.
   */
  protected Component makeRels() {
    //make a panel that can take all the bits
    JPanel panel = new JPanel(new GridBagLayout());
    
    //make the type and relations field
    JLabel tLabel = new JLabel("Constraint type");
    JLabel rLabel = new JLabel("Constraint relation (sub-type)");
    typeField = new JTextField();
    relField = new JTextField();
    
    //make the statement text
    JLabel sLabel = new JLabel("Parameters (pattern and value)");
    statement = new JTextField();
    IVUtil.enableVars(this, statement);

    //add it all to the panel
    GridBagConstraints c;
    c = new GridBagConstraints(0, 0, //x,y
			       1, 1, //width height
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.NORTHWEST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,2,0,12),0,0); //insets and padding
    panel.add(tLabel, c);
    c = new GridBagConstraints(1, 0 , //x,y
			       1, 1, //width height
			       1.0,0.0,  //weight x,y
			       GridBagConstraints.NORTHWEST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    panel.add(rLabel, c);
    c = new GridBagConstraints(0, 1, //x,y
			       1, 1, //width height
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.WEST, //anchor
			       GridBagConstraints.HORIZONTAL,  //fill
			       new Insets(0,2,0,12),0,0); //insets and padding
    panel.add(typeField, c);
    c = new GridBagConstraints(1, 1, //x,y
			       1, 1, //width height
			       1.0,0.0,  //weight x,y
			       GridBagConstraints.EAST, //anchor
			       GridBagConstraints.HORIZONTAL,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    panel.add(relField, c);
    c = new GridBagConstraints(0, 2 , //x,y
			       2, 1, //width height
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.WEST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    panel.add(sLabel, c);
    c = new GridBagConstraints(0, 3, //x,y
			       2, 1, //width height
			       1.0,0.0,  //weight x,y
			       GridBagConstraints.SOUTHWEST, //anchor
			       GridBagConstraints.HORIZONTAL,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    panel.add(statement, c);

    return panel;
  }

  protected JComponent getFocusComponent() {
    return typeField;
  }



  /**
   * Sets the refinement whose components (children) are to be realted.
   * Fills the activity lists of the panel with the children of the refinement.
   * @param refinement the refinement whose children are to be related 
   *
   * @throws NullPointerException if the refinement has no children.
   */
  protected void initFromObject() throws NullPointerException {
    typeField.setText("");
    relField.setText("");
    statement.setText("");
    //typeField.requestFocus();
  }

  protected void setFromObject(Object original) {
    typeField.setText("");
    relField.setText("");
    statement.setText("");
    if (original == null) return;
    if (!(original instanceof Constraint)) {
      String message = "Condition/Effect editor can only deal with constraints"
	+ " not objects of class " + original.getClass().getName();
      JOptionPane.showMessageDialog(this,message);
      return;
    }

    Constraint constraint = (Constraint)original;
    typeField.setText(constraint.getType().toString());
    if (constraint.getRelation() == null)
      relField.setText("");
    else relField.setText(constraint.getRelation().toString());
    
    statement.setText(IVUtil.printConstraintParameters(constraint));
    //statement.requestFocus();
  }

  /**
   * Reads the selected relation and let interested parties know.
   * Checks that selections have been made.
   */
  protected Object collectConstraint() {
    String typeStr = typeField.getText().trim();
    String relStr = relField.getText().trim();
    Symbol type = Symbol.valueOf(typeStr);
    Symbol rel = Symbol.valueOf(relStr);
    
    String text = statement.getText().trim();
    //Debug.noteln("CEE: Got statement", text);
    Constraint newConstraint = null;
    try { newConstraint = IVUtil.readCondition(type, rel, text); }
    catch (Exception e1) {
	Debug.noteln("DefCEd: constraint not a condition");
      try {
	List parameters = readParameters(text);
	newConstraint = new Constraint();
	newConstraint.setType(type);
	if (rel != null) newConstraint.setRelation(rel);
	newConstraint.setParameters(parameters);
      }
      catch (Exception e2) {
	Debug.noteException(e2);
	JOptionPane.showMessageDialog(this, "Constraint Editor: " + 
				      Debug.foldException(e2));
      }
    }
    Debug.noteln("DefCEd: new constraint is", newConstraint);
    if (isEditing()) {
      try {
	Constraint constraint = (Constraint)((Constraint)original).clone();
	constraint.setType(type);
	constraint.setRelation(rel);
	if (newConstraint != null)
	    constraint.setParameters(newConstraint.getParameters());
	Debug.noteln("DConstrEd: got constraint", constraint);
	return constraint;
      }
      catch (CloneNotSupportedException cnse) {
	Debug.describeException(cnse);
	JOptionPane.showMessageDialog(this, "Constraint Editor: " + 
				      Debug.foldException(cnse));
	return null;
      }	
    }
    else return newConstraint;
  }

  

  protected List readParameters(String text) {
    String[] parts = Strings.breakAtFirst("=", text);
    String pat = UIUtil.ensureParenthesized(parts[0].trim());
    String val = parts[1].trim();
    if (val.equals("")) val = "true";

    String parameters = "(" + pat + " = " + val + ")";
    LList spec = (LList)Lisp.readFromString(parameters);
    //Debug.noteln("Spec list is", spec);
    //Debug.noteln(" class", spec.getClass());
    
    LListCollector list = new LListCollector();
    list.add(new PatternAssignment((LList)spec.car(), val));
    return list.contents();
  }
}

//This would probably be neater with JComboBox objects for type and rel?

