/****************************************************************************
 * An abstract class with useful things for building constraint editors.
 *
 * @author Jussi Stader
 * @version 4.0+
 * Updated: Thu May  4 15:42:24 2006
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iface.ui;

import javax.swing.*;       
import java.util.*;
import java.awt.BorderLayout;
//import java.awt.GridBagLayout;
//import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Frame;
import java.awt.Component;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseListener;
import java.awt.event.MouseEvent;
import ix.*;
import ix.util.Debug;
import ix.util.lisp.LList;
import ix.util.lisp.Lisp;
//import ix.icore.domain.*;
//import ix.icore.*;

/****************************************************************************
 * An abstract class with useful things for building constraint editors.
 *
 * The editor is implemented as a frame using Boxes and JButtons.  <p>
 *
 * The editor can be created with a message and optional title and n optional
 * object which will be set as the current object to which the constraint
 * applies.
 * Subsequently, the object in the editor can be set using the setObject
 * method. <p>
 *
 * Concrete editors that extend this class should implement the following 
 * methods:
 * <ul> makeBits() - makes all components needed to specify the constraint.
 * <li> initFromObject() - initialises components with information from the
 *      current object, e.g. lists of children are filled with those of the
 *      current refinement.
 * <li> collectConstraint() - collects user input from the components and 
 *      creates (and returns) a Constraint object from that information.
 *      Also deals with edit vs add behaviour.
 * <li> If you allow list-fields to be edited, also define protected List 
 *      getPreviousList(Object constraint) and protected String 
 *      getField(Object constraint) to obtain previous values to be updated.
 * <li> If there is a component that should have the keyboard focus when the 
 *      editor comes up, also define protected JComponent getFocusComponent()
 * </ul>
 * The rest can be handled by the generic methods defined in this class.
 * For an example of an implementing class see TPRelationEditor.
 *
 * Implements the VarSpecifier interface for UIRefinements. Use
 * IVUtil.enableVars(VarSpecifier editor, JTextComponent textBit);
 * to activate variable input help. Collects variable declarations and
 * updates the construct when the constraint is processed (in case of cancel)
 *
 *****************************************************************************
 */
public abstract class JSelectorDialog extends JDialog
  implements MouseListener
{
  protected JPanel mainPanel = new JPanel();
  protected Component bitsBox;
  protected Component butBox;
  protected JComponent focusComponent;

  //box items -- to be defined by the concrete class
  //control items
  protected JButton okBut = new JButton("Ok");
  protected JButton cancelBut = new JButton("Cancel");

  protected JLabel label;

  protected ArrayList selectionListeners = new ArrayList();
  public Object currentObject;

  
  /**
   * Creates a dialog with the title for selecting from a list of items.
   * Creates space for components for specifying the constraints
   * and creates two buttons (Ok, Cancel)
   */
  public JSelectorDialog(Frame parent, String title, String message) {
    super(parent, title, true); //modal
    //Debug.noteln("CE: making selector");
    this.setSize(405,175);
    this.mainPanel.setLayout(new BorderLayout());
    this.getContentPane().add(this.mainPanel);
    this.mainPanel.setBounds(0,0,500,300);
    bitsBox = makeBits();
    butBox = makeButs();

    if (!message.equals("")) {
      label = new JLabel(message);
      mainPanel.add(label, BorderLayout.NORTH);
    }
    this.mainPanel.add(bitsBox, BorderLayout.CENTER);
    mainPanel.add(new JSeparator());
    mainPanel.add(butBox, BorderLayout.SOUTH);  
    
    focusComponent = getFocusComponent();
    //if (focusComponent != null) focusComponent.requestFocusInWindow();
    this.invalidate();
    pack();
    setVisible(false);
  }
    
  /**
   * As above but with a default title.
   */
  public JSelectorDialog(Frame parent, String message) {
    this(parent, "Selector", message);
  }
    
  /**
   * Creates a a selector with the context of the given object.
   * As JSelectorDialog() but with a given object.
   * @param object the object for whom the selection is to be made
   */
  public JSelectorDialog(Frame parent, String title, String message, 
			 Object object){
    this(parent, title, message);

    try {
      this.setObject(object);
    } 
    catch (NullPointerException e) {
      Debug.noteException(e);
      JOptionPane.showMessageDialog(this, "Selector: " + 
				    "Cannot set object");
      return;
    }
    pack();
    setVisible(false);
  }
  
  
  /**
   * Makes and adds ok and cancel buttons.
   */
  private Component makeButs(){
    okBut.addMouseListener(this);
    cancelBut.addMouseListener(this);
    Box butBox = new Box(BoxLayout.X_AXIS);
    butBox.add(Box.createGlue());
    butBox.add(okBut);
    butBox.add(cancelBut);
    butBox.add(Box.createGlue());
    return butBox;
  }

  public void setMessage(String message) {
    if (label != null) label.setText(message);
  }

  /**
   * Sets the object whose components (children) are to be realted.
   * Fills the activity lists of the panel with the children of the object.
   * @param object the object whose children are to be related 
   */
  public void setObject(Object object) throws NullPointerException {
    //Debug.noteln("Object in Selector is", object);
    if (object == null) return;
    this.currentObject = object;
    this.initFromObject();
    //assignFocus();
  }

  /**
   * By default, there is no focus component
   */
  protected JComponent getFocusComponent() {
    return null;
  }

  public void start(Point point) {
    setLocation(point);
    start();
  }
  public void start() {
    setVisible(true);
  }
  /** Starts an editor for for selecting something for the given object.
   */
  public void start(Object object, Point point) {
    setLocation(point);
    start(object);
  }
  /**
   * As above, default location.
   */
  public boolean start(Object object) {
    try {
      setObject(object); 
      start();
      return true;
    }
    catch (NullPointerException e) {
      Debug.noteException(e);
      JOptionPane.showMessageDialog(this, "Selection Dialogue: " + 
				    "Cannot start (problem in set object)");
      return false;
    }
  }

  public void addSelectionListener(JSelectorListener listener){
    this.selectionListeners.add(listener);
  }
  public void removeSelectionListener(JSelectorListener listener){
    while (this.selectionListeners.contains(listener)){
      this.selectionListeners.remove(listener);
    }
  }

  /**
   * Reads the selected relation and let interested parties know.
   * Checks that selections have been made.
   * If selections have not been made, a message will have been shown and 
   * listeners are not notified.
   * If an original constraint has been edited (rather than new constraints
   * added), the listeners are notified with a null constraint object (they
   * already hold the original that has now been updated).
   */
  protected Object processSelection() {
    Object selection = collectSelection();
    if (selection == null) return null; // selection not specified
    Object newValue = checkSelection(selection, currentObject);
    return newValue;
  }

  public void mouseClicked(MouseEvent e) {
    if (e.getSource() == this.cancelBut) {
      fireGotSelection(null);
      this.closeEditor();
      return;
    }
    if (e.getSource() == this.okBut) {
      Object selection = processSelection();
      if (selection != null) {
	fireGotSelection(selection);
	this.closeEditor();
      }
      return;
    }
  }
  public void mouseEntered(MouseEvent e){}
  public void mouseExited(MouseEvent e){}
  public void mousePressed(MouseEvent e){}
  public void mouseReleased(MouseEvent e){}

  private void fireGotSelection(Object selection) {
    if (this.selectionListeners == null) return;
    for (int i=0; i<selectionListeners.size(); i++) {
      JSelectorListener l = 
	(JSelectorListener) selectionListeners.get(i);
      l.gotSelection(this, selection, currentObject);
    }
  }

  public void closeEditor() {
    setVisible(false);
    currentObject = null;
  }

  /**
   * Makes all items on the panel used to specify parts of the constraint.
   */
  abstract protected Component makeBits();
  /**
   * Fills the controls with relevant object-sensitive information.
   * To be implemented by the sub-class. E.g. fill ordering editor with the
   * sub-activities of the refinement.
   */
  abstract protected void initFromObject();
  /**
   * Collects input from the panel components and builds the right kind of
   * constraint.
   */
  abstract protected Object collectSelection();
  /**
   * Checks the selection against the given object. Returns the
   * selection (possibly modified) if all is well, null if the
   * selection is not suitable in this context.
   */
  abstract protected Object checkSelection(Object selection, Object object);
  //also define protected JComponent getFocusComponent()




  public interface JSelectorListener {
    public void gotSelection(JSelectorDialog selector, 
			     Object selection, Object object);
  }


}

/** Changes
 * 
 * 16/02/04: Fixed bug on variable declarations.
 *
 */
