/****************************************************************************
 * An abstract class with useful things for building constraint editors.
 *
 * @author Jussi Stader
 * @version 4.0+
 * Updated: Thu Jan 25 10:15:00 2007
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iface.ui;

import javax.swing.*;       
import java.util.*;
//import java.awt.BorderLayout;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Frame;
import java.awt.Component;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseListener;
import java.awt.event.MouseEvent;
import ix.*;
import ix.util.Debug;
//import ix.util.lisp.LList;
//import ix.util.lisp.Lisp;
//import ix.icore.domain.*;
//import ix.icore.*;

/****************************************************************************
 * An abstract class with useful things for building constraint editors.
 *
 * The editor is implemented as a frame using Boxes and JButtons.  <p>
 *
 * The editor can be created with a message and optional title and n optional
 * object which will be set as the current object to which the constraint
 * applies.
 * Subsequently, the object in the editor can be set using the setObject
 * method. <p>
 *
 * Concrete editors that extend this class should implement the following 
 * methods:
 * <ul> makeBits() - makes all components needed to specify the constraint.
 * <li> initFromObject() - initialises components with information from the
 *      current object, e.g. lists of children are filled with those of the
 *      current refinement.
 * <li> setFromObject(Object original) - set the constraints using the given
 *      object (used when editing constraints).
 * <li> collectConstraint() - collects user input from the components and 
 *      creates (and returns) a Constraint object from that information.
 *      Also deals with edit vs add behaviour. Called when editor buttons
 *      "Ok" or "More" are clicked.
 * <li> If you allow list-fields to be edited, also define protected List 
 *      getPreviousList(Object constraint) and protected String 
 *      getField(Object constraint) to obtain previous values to be updated.
 * <li> If there is a component that should have the keyboard focus when the 
 *      editor comes up, also define protected JComponent getFocusComponent()
 * </ul>
 * The rest can be handled by the generic methods defined in this class.
 * For an example of an implementing class see TPRelationEditor.
 *
 * Implements the VarSpecifier interface for UIRefinements. Use
 * IVUtil.enableVars(VarSpecifier editor, JTextComponent textBit); to
 * activate variable input help. Collects variable declarations and
 * updates the construct when the constraint is processed (in case of
 * cancel)
 *
 * NOTE: this is modal, so as soon as it is visible (start()), it blocks!
 *
 *****************************************************************************
 */
public abstract class JConstraintEditor extends JDialog //JFrame
  implements MouseListener
{
  protected JPanel mainPanel = new JPanel();
  protected Component bitsBox;
  protected Component butBox;
  protected JComponent focusComponent;

  //box items -- to be defined by the concrete class
  //control items
  protected JButton okBut = new JButton("Ok");
  protected JButton cancelBut = new JButton("Cancel");
  protected JButton moreBut = new JButton("Add More");

  protected JLabel label;

  protected ArrayList constraintListeners = new ArrayList();
  public Object currentObject;
  protected Object original; //used for "edit" rather than "add" behaviour

  
  /**
   * Creates an editor with the title for specifying constraints.
   * Creates space for components for specifying the constraints
   * and creates two buttons (Ok, Cancel)
   */
  public JConstraintEditor(Frame owner, String title, String message) {
    super(owner, title, true);
    //Debug.noteln("CE: making constraint editor");
    this.setSize(405,175);
    this.mainPanel.setLayout(new GridBagLayout());
    this.getContentPane().add(this.mainPanel);
    this.mainPanel.setBounds(0,0,500,300);
    bitsBox = makeBits();
    butBox = makeButs();

    GridBagConstraints gbc;
    if (!message.equals("")) {
      label = new JLabel(message);
      gbc = 
	new GridBagConstraints(0,0,  GridBagConstraints.REMAINDER,1, //x,y, w,h
			       1.0,0.0,  //weight x,y
			       GridBagConstraints.NORTHWEST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,4,0,0),0,0); //insets and padding
      mainPanel.add(label, gbc);
    }
    mainPanel.add(new JSeparator());
    gbc = new GridBagConstraints(0,1, GridBagConstraints.REMAINDER,1, //x,y,w,h
				 1.0,1.0,  //weight x,y
				 GridBagConstraints.NORTHWEST, //anchor
				 GridBagConstraints.BOTH,  //fill
				 new Insets(0,4,0,0),0,0); //insets and padding
    this.mainPanel.add(bitsBox, gbc);
    mainPanel.add(new JSeparator());
    gbc = new GridBagConstraints(0,2, GridBagConstraints.REMAINDER,1, //x,y,w,h
				 0.0,0.0,  //weight x,y
				 GridBagConstraints.SOUTHWEST, //anchor
				 GridBagConstraints.HORIZONTAL,  //fill
				 new Insets(0,4,0,0),0,0); //insets and padding
    mainPanel.add(butBox, gbc);  
    
    focusComponent = getFocusComponent();
    //if (focusComponent != null) focusComponent.requestFocusInWindow();
    this.invalidate();
    pack();
    this.setVisible(false);
  }
    
  /**
   * As above but with a default title.
   */
  public JConstraintEditor(Frame owner, String message) {
    this(owner, "Constraint Editor", message);
  }
    
  /**
   * Creates an editor for specifying time point relations of the given object.
   * As JConstraintEditor() but with a given object.
   * @param object the object whose constraints are to be edited.
   */
  public JConstraintEditor(Frame owner, String title, String message, 
			   Object object){
    this(owner, title, message);

    try {
      this.setObject(object);
    } 
    catch (NullPointerException e) {
      Debug.noteException(e);
      JOptionPane.showMessageDialog(this, "Constraint Editor: " + 
				    "Cannot set object");
      return;
    }
    pack();
    setVisible(true);
  }
  
  
  /**
   * Makes and adds ok and cancel buttons.
   */
  private Component makeButs(){
    okBut.addMouseListener(this);
    cancelBut.addMouseListener(this);
    moreBut.addMouseListener(this);
    Box butBox = new Box(BoxLayout.X_AXIS);
    butBox.add(Box.createGlue());
    butBox.add(okBut);
    butBox.add(cancelBut);
    butBox.add(moreBut);
    butBox.add(Box.createGlue());
    return butBox;

  }




  public boolean isEditing() {
    return (original != null);
  }

  public void setMessage(String message) {
    if (label != null) label.setText(message);
  }

  /**
   * Sets the object whose components (children) are to be realted.
   * Fills the activity lists of the panel with the children of the object.
   * @param object the object whose children are to be related 
   */
  public void setObject(Object object) throws NullPointerException {
    //Debug.noteln("Object in ConstraintEditor is", object);
    if (object == null) return;
    this.currentObject = object;
    this.initFromObject();
    //assignFocus();
  }

  /**
   * Gets the object that is to be constrainted (UIRefinement)
   */
  public Object getObject() {
    return this.currentObject;
  }

  /**
   * By default, there is no focus component
   */
  protected JComponent getFocusComponent() {
    return null;
  }

  protected void focus() {
    if (getFocusComponent() != null) getFocusComponent().requestFocus();
  }

  /**
   * Gets the previous list of values for the given constraint.
   * By default, there is no previous list
   */
  protected List getPreviousList(Object constraint) {
    Debug.noteln("Dummy getPreviousList(Constraint c) in JCE");
    return null;
  }
  /**
   * Gets the field that the constraint is kept in. 
   * By default, we do not look up the value
   */
  protected String getField(Object constraint) {
    Debug.noteln("Dummy getField(Constraint c) in JCE");
    return "";
  }

  /**
   * Set this to the original value (non-null) to get edit-bahaviour rather
   * than add-behaviour.
   */
  public void setOriginal(Object original) {
    this.original = original;
  }

  public void start(Point point) {
    //Debug.noteln("JCEd: starting with point");
    setLocation(point);
    start();
  }
  public void start() {
    setVisible(true);
  }
  /** Starts an editor for for adding specifications of one of the fields
   * of the given object.
   */
  public void start(Object object, Point point) {
    setLocation(point);
    start(object);
  }
  /**
   * As above, default location.
   */
  public void start(Object object) {
    try {
      original = null;
      setObject(object); 
      start();
    }
    catch (NullPointerException e) {
      Debug.noteException(e);
      JOptionPane.showMessageDialog(this, "Constraint Editor: " + 
				    "Cannot start (" +
				    e.getMessage() + ")");
    }
  }
  /** Starts an editor for for editing the given specification of
   * one of the fields of the given object.
   */
  public void start(Object object, Object original, Point point) {
    setLocation(point);
    start(object, original);
  }
  /**
   * As above, default location.
   */
  public void start(Object object, Object original) {
    //Debug.noteln(" and original", original);

    try {
      this.original = null;
      setObject(object); 
      setOriginal(original);
      setFromObject(original);
      moreBut.setEnabled(false);
      start();
    }
    catch (Exception e) {
      Debug.noteException(e);
      JOptionPane.showMessageDialog(this, "Constraint Editor: " + 
				    "Cannot start (" +
				    e.getMessage() + ")");
    }
  }


  public void addConstraintListener(JConstraintListener listener){
    this.constraintListeners.add(listener);
  }
  public void removeConstraintListener(JConstraintListener listener){
    while (this.constraintListeners.contains(listener)){
      this.constraintListeners.remove(listener);
    }
  }

  /**
   * Reads the selected relation and let interested parties know.
   * Checks that selections have been made.
   * If selections have not been made, a message will have been shown and 
   * listeners are not notified.
   * If an original constraint has been edited (rather than new constraints
   * added), the listeners are notified with a null constraint object (they
   * already hold the original that has now been updated).
   */
  protected boolean processConstraint() {
    Object constraint = collectConstraint();
    if (constraint == null) return false; // constraint not specified
    Debug.noteln("JCE: made constraint", constraint);
    //Debug.noteln(" original is", original);
    Object newValue = noteNewValue(constraint);
    if (this.constraintListeners == null) return true;
    for (int i=0; i<this.constraintListeners.size(); i++) {
      JConstraintListener l = 
	(JConstraintListener) this.constraintListeners.get(i);
      l.gotConstraint(this, this.currentObject, newValue);
    }
    return true;
  }

  /**
   * make UIO update properly so that the event is noted elsewhere (panels..)
   */
  protected Object noteNewValue(Object constraint) {
    //Debug.noteln("JCE: noting new constraint", constraint);
    List previousList = getPreviousList(constraint);

    if (isEditing()) {     //remove original, add new
      //Debug.noteln("JCE: editing - noting value");
      if (previousList != null) {
	int position = previousList.indexOf(original);
	if (position >= 0) {
	  previousList.remove(position);
	  previousList.add(position, constraint);
	  ((EditableObject)currentObject).setValue(getField(constraint), 
						   previousList);
	} 
      }
      return null;
    }
    else {     //not editing, so just add new
      //Debug.noteln("JCE: new constraint - noting value into", previousList);
      if (previousList != null) {
	previousList.add(previousList.size(),constraint);
	//Debug.noteln(" noted", previousList);
	((EditableObject)currentObject).setValue(getField(constraint), 
						 previousList);
      } 
      return constraint;
    }
  }


  public void mouseClicked(MouseEvent e){
    //Debug.noteln("JCEd: processing mouse click from", e.getSource());
    if (e.getSource() == this.cancelBut) {
      this.closeEditor();
      return;
    }
    if (e.getSource() == this.okBut) {
      if (this.processConstraint()) this.closeEditor();
      //else JOptionPane.showMessageDialog(this, "Constraint Editor: " + 
      //				 "Cannot process constraint");
      return;
    }
    if (e.getSource() == this.moreBut) {
      if (this.processConstraint())
	moreInitFromObject();
      //else JOptionPane.showMessageDialog(this, "Constraint Editor: " + 
      //				 "Cannot process constraint");
      return;
    }
  }
  public void mouseEntered(MouseEvent e){}
  public void mouseExited(MouseEvent e){}
  public void mousePressed(MouseEvent e){}
  public void mouseReleased(MouseEvent e){}

  public void closeEditor() {
    setVisible(false);
    original = null;
    currentObject = null;
    moreBut.setEnabled(true);
  }

  /**
   * Called instead of initFromObject when addMore is clicked rather than add.
   */
  public void moreInitFromObject() { initFromObject(); }

  /**
   * Makes all items on the panel used to specify parts of the constraint.
   */
  abstract protected Component makeBits();
  /**
   * Fills the controls with relevant object-sensitive information.
   * To be implemented by the sub-class. E.g. fill ordering editor with the
   * sub-activities of the refinement.
   */
  abstract protected void initFromObject();
  /**
   * Sets the editor up to edit (rather than add) an original constraint
   * object.
   */
  abstract protected void setFromObject(Object original);
  /**
   * Collects input from the panel components and builds the right kind of
   * constraint.
   */
  abstract protected Object collectConstraint();
  //also define protected JComponent getFocusComponent()




}

/** Changes
 * 
 * 16/02/04: Fixed bug on variable declarations.
 *
 */
