/****************************************************************************
 * A toolbar button with text and icon that is not necessarily shown
 *
 * @author Jussi Stader
 * @version 4.0+
 * Updated: Thu Aug 17 12:00:05 2006
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iface.ui;

import ix.*;
import ix.util.*;
import ix.iface.ui.util.*;
import ix.iface.util.*;

import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
//import java.net.*;

/**
 * A toolbar button that is not necessarily shown
 *
 */
//NOTE: ideally, buttons and items in these objects would persist and be 
// adapted, but that results in multiple events when they are clicked.
// Therefore buttons and menu items are junked (removed) when they are not
// visible.
public class IXToolItem extends Object
{

  String command;
  String label;
  String buttonLabel;
  Icon icon;
  boolean itemShown;
  boolean buttonShown;
  boolean showButtonText = false;
  char mnemo = KeyEvent.VK_UNDEFINED;
  String toolTip;
  static final char noMnemo = KeyEvent.VK_UNDEFINED;

  JMenu menu;
  JMenuItem item;
  Class itemClass = JMenuItem.class;
  int menuPosition = -1;

  JToolBar bar;
  AbstractButton button;

  ActionListener al;


  /**
   * Creates an object that holds information for a menu item only.
   * The information held is parentMenu, command, listener, label, toolTip.
   * Does not create the menu item.
   */
  //item
  public IXToolItem(JMenu parentMenu, ActionListener al, 
		    String command, String label, String toolTip){
    super();
    this.menu = parentMenu;
    setActionListener(al);
    this.command = command;
    this.label = label;
    this.toolTip = toolTip;
  }
  /**
   * As above with mnemo.
   */
  //item, mnemo
  public IXToolItem(JMenu parentMenu, ActionListener al, 
		    String command, String label, char mnemo, String toolTip){
    this(parentMenu, al, command, label, toolTip);
    this.mnemo = mnemo;
  }
  /**
   * Creates an object that holds information for a menu item and a toolbar
   * button. 
   * The information held is as above and toolbar, icon.
   * Does not create menu item or button.
   */
  //item, button
  public IXToolItem(JMenu parentMenu, JToolBar bar, ActionListener al, 
		    String command, String label, Icon icon, String toolTip){
    this(parentMenu, al, command, label, toolTip);
    this.buttonLabel = label;
    this.bar = bar;
    this.icon = icon;
  }
  /**
   * As above with mnemo.
   */
  //item, button, mnemo
  public IXToolItem(JMenu parentMenu, JToolBar bar, ActionListener al, 
		    String command, String label, Icon icon, char mnemo, 
		    String toolTip){
    this(parentMenu, bar, al, command, label, icon, toolTip);
    this.mnemo = mnemo;
  }
  /**
   * Creates an object that holds information for a menu item and a toolbar
   * button with a seperate button label. 
   * The information held is as above and button label.
   * Does not create menu item or button.
   */
  //item, button, buttonLabel
  public IXToolItem(JMenu parentMenu, JToolBar bar, ActionListener al, 
		    String command, String label, String buttonLabel,
		    Icon icon, String toolTip){
    this(parentMenu, al, command, label, toolTip);
    this.buttonLabel = buttonLabel;
    //showButtonText = true;
    this.bar = bar;
    this.icon = icon;
  }
  /**
   * As above with mnemo.
   */
  //item, button, buttonLabel, mnemo
  public IXToolItem(JMenu parentMenu, JToolBar bar, ActionListener al, 
		    String command, String label, String buttonLabel, 
		    Icon icon, char mnemo, String toolTip){
    this(parentMenu, bar, al, command, label, buttonLabel, icon, toolTip);
    this.mnemo = mnemo;
  }
  /**
   * Creates an object that holds information for a toolbar button only.
   * Does not create the button.
   */
  //button
  public IXToolItem(JToolBar bar, ActionListener al, 
		    String command, String label,
		    Icon icon, String toolTip){
    super();
    this.bar = bar;
    setActionListener(al);
    this.command = command;
    this.buttonLabel = label;
    this.icon = icon;
    this.toolTip = toolTip;
  }


  public void setActionListener(ActionListener al) {
    if (al instanceof CatchingActionListener)
      this.al = al;
    else this.al = new CatchingActionListener(al);
  }



  public JMenuItem makeItem() {
    itemClass = JMenuItem.class;
    item = new JMenuItem();
    return item;
  }
  public JRadioButtonMenuItem makeRadioItem() {
    itemClass = JRadioButtonMenuItem.class;
    item = new JRadioButtonMenuItem();
    return (JRadioButtonMenuItem)item;
  }
  public JCheckBoxMenuItem makeCheckItem() {
    itemClass = JCheckBoxMenuItem.class;
    item = new JCheckBoxMenuItem();
    return (JCheckBoxMenuItem)item;
  }
  public JToggleButton makeRadioTool() {
    itemClass = JRadioButton.class;
    button = new JRadioButton();
    return (JToggleButton)button;
  }

  public JCheckBoxMenuItem ensureCheckItem() {
    if (item == null) return makeCheckItem();
    else return (JCheckBoxMenuItem)item;
  }

  /**
   * Creates a button (if necessary) and shows it on its toolbar.
   */
  public JMenuItem show() {
    if (bar != null) showButton();
    if (menu != null) return showItem();
    else return null;
  }
   /**
   * Creates a button (if necessary) and shows it on its toolbar.
   */
  public JMenuItem showItem() {
    if (menu == null) return null;

    //if (item == null) item = new JMenuItem();
    if (item == null) {
      try { item = (JMenuItem)itemClass.newInstance(); }
      catch (Exception e) { item = new JMenuItem();}
    }
    item.setHorizontalTextPosition(SwingConstants.LEFT);
    if (label != null) item.setText(label);
    if (mnemo != noMnemo) item.setMnemonic(mnemo);
    if (toolTip != null) item.setToolTipText(toolTip);
    if (icon != null) item.setIcon(icon);
    if (command != null) item.setActionCommand(command);
    if (al != null) {
      java.util.List lrs = Arrays.asList(item.getActionListeners());
      if ((lrs != null) && !lrs.contains(al))
	item.addActionListener(al);
    } 
    if (!menu.isMenuComponent(item)) {
      if (menuPosition == -1) menuPosition = menu.getItemCount();
      try {menu.insert(item, menuPosition);}
      catch (Exception e) { menu.add(item);}
    }
    item.setVisible(true);
    itemShown = true;
    return item;
  }

  /**
   * Creates a button (if necessary) and shows it on its toolbar.
   */
  public AbstractButton showButton() {
    if (bar == null) return null;

    if (button == null) {
      button = new JButton();
      //Debug.noteln("IXTI: making button", buttonLabel);
      button.setVerticalTextPosition(AbstractButton.BOTTOM);
      button.setHorizontalTextPosition(AbstractButton.CENTER);
    }
    if (icon != null) button.setIcon(icon);
    if ((buttonLabel != null) && showButtonText) button.setText(buttonLabel);
    else button.setText("");
    if (mnemo != noMnemo) button.setMnemonic(mnemo);
    if (toolTip != null) button.setToolTipText(toolTip);
    if (command != null) button.setActionCommand(command);
    if (al != null) {
      java.util.List lrs = Arrays.asList(button.getActionListeners());
      if ((lrs != null) && !lrs.contains(al))
	button.addActionListener(al);
    }
    if (!bar.isAncestorOf(button)) bar.add(button);
    button.setVisible(true);
    buttonShown = true;
    return button;
  }

  public void hide() {
    if ((bar != null) && (button != null)) hideButton();
    if ((menu != null) && (item != null)) hideItem();
  }

  /** hides the button from its toolbar. */
  public void hideButton() {
    if ((bar != null) && (button != null) && bar.isAncestorOf(button)) {
      //try { bar.remove(button); }
      try { 
	bar.remove(button); 
	//button.setVisible(false); 
	button = null;
      }
      catch (NullPointerException e) {}
    }
    buttonShown = false;
  }
  /** hides the button from its toolbar. */
  public void hideItem() {
    if ((menu != null) && (item != null) && menu.isMenuComponent(item)) {
      //Debug.noteln("IXTI: hiding item");
      try { 
	menu.remove(this.item); 
	item = null;
      }
      //try { item.setVisible(false); }
      catch (NullPointerException e) {
	Debug.noteln("IXTI: Got null pointer in hideItem");
      }
    }
    itemShown = false;
  }
  public void showButtonText(boolean show) {
    showButtonText = show;
    if (button != null) {
      if (show) button.setText(buttonLabel);
      else button.setText("");
    }
  }

  public void setIcon(Icon icon) {
    this.icon = icon;
    if (item != null) item.setIcon(icon);
    if (button != null) button.setIcon(icon);
  }
  public void setLabel(String label) {
    this.label = label;
    if (buttonLabel == null) buttonLabel = label;
    if (item != null) item.setText(label);
    if (button != null) button.setText(label);
  }
  public void setItemLabel(String label) {
    this.label = label;
    if (item != null) {
      item.setText(label);
    }
  }
  public void setButtonLabel(String label) {
    buttonLabel = label;
    if (button != null) {
      button.setText(label);
    }
    //showButtonText = true;
  }
  public void setToolTip(String text) {
    toolTip = text;
    if (item != null) item.setToolTipText(text);
    if (button != null) button.setToolTipText(text);
  }
  public void setEnabled(boolean enabled) {
    if (button != null) button.setEnabled(enabled);
    if (item != null) item.setEnabled(enabled);
  }

  /** Sets the item's menu ONLY if the current menu is null. */
  public void setMenu(JMenu menu) {
    if (this.menu == null) this.menu = menu;
  }

  private int findPosition(JMenu menu, JMenuItem item) {
    if ((menu == null) || (item == null)) return -1;
    int all = menu.getItemCount();
    JMenuItem mItem;
    for (int i=0; i<all; i++) {
      mItem = menu.getItem(i);
      if (item.equals(mItem)) return i;
    }
    return -1;
  }

      
  
}
