/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Dec  6 15:49:05 2007 by Jeff Dalton
 * Copyright: (c) 2005, 2006, 2007, AIAI, University of Edinburgh
 */

package ix.util.http;

import org.mortbay.jetty.Server;
import org.mortbay.jetty.Connector;
import org.mortbay.jetty.Handler;
import org.mortbay.jetty.NCSARequestLog;

// import org.mortbay.jetty.handler.ContextHandler;
import org.mortbay.jetty.handler.ContextHandlerCollection;
import org.mortbay.jetty.handler.DefaultHandler;
import org.mortbay.jetty.handler.HandlerCollection;
import org.mortbay.jetty.handler.RequestLogHandler;
// import org.mortbay.jetty.handler.ResourceHandler;

import org.mortbay.jetty.servlet.Context;
import org.mortbay.jetty.servlet.DefaultServlet;
import org.mortbay.jetty.servlet.ServletHandler;
import org.mortbay.jetty.servlet.ServletHolder;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import java.net.*;
import java.io.*;
import java.util.*;

import ix.util.*;

/**
 * An HTTP 1.1 server that can serve static content (files) and
 * also supports servlets.
 *
 * <p>Example:
 * <pre>
 *   import javax.servlet.ServletException;
 *   import javax.servlet.http.HttpServlet;
 *   import javax.servlet.http.HttpServletRequest;
 *   import javax.servlet.http.HttpServletResponse;
 *
 *   import java.io.IOException;
 *
 *   import ix.util.http.HttpServer;
 *
 *   public class Example {
 *       public static void main(String[] argv) {
 *           HttpServer server = new HttpServer(8080);
 *           server.setDocRoot("myfiles", "/files");
 *           server.setLogPath("logs", "myapp");
 *           server.addServlet(new HelloServlet(), "/hello/*");
 *           server.start();
 *       }
 *   }
 *
 *   public class HelloServlet extends HttpServlet {
 *
 *       protected void doGet(HttpServletRequest req,
 *                            HttpServletResponse resp)
 *                 throws ServletException, IOException {
 *           resp.setContentType("text/plain");
 *           resp.setStatus(HttpServletResponse.SC_OK);
 *           resp.getWriter().println("Hello!");
 *       }
 *
 *   }
 * </pre>
 *
 * @see HttpObjectServlet
 * @see HttpStringServlet
 * @see HelloServlet
 */
public class HttpServer {

    private int listenPort = 0;

    private String docRoot = null;
    private String docPath = null;

    private String logDir = null;
    private String logPrefix = null;

    private Server server;
    private Connector socketListener;
    private ContextHandlerCollection contexts;
    private ServletHandler servlets;

    private HttpUtilities util = new HttpUtilities();

    static {
	System.setProperty("java.net.preferIPv4Stack", "true");
    }

    /**
     * Construct a server that listens on the specified port.
     * If the port is 0, the operating system will be asked for
     * a free port.  After the server has been started, the port number
     * can be obtained by calling {@link #getServerPort()}.
     */
    public HttpServer(int port) {
	this.listenPort = port;
	initialSetup();
    }

    /**
     * Main program for testing.  The effect is similar to running I-Serve,
     * but there's no GUI or log file.  However, the "http-root-directory"
     * and "http-server-port" parameters can be used.
     */
    public static void main(String[] argv) {
	Parameters.processCommandLineArguments(argv);

	String docRoot = Parameters.getParameter("http-root-directory");
	int port = Parameters.getInt("http-server-port", 0);

	HttpServer server = new HttpServer(port);
	if (docRoot != null)
	    server.setDocRoot(docRoot, "/files");

	// Add some test servlets
	server.addServlet(new HelloServlet("Hello!"), "/hello/*");
	server.addServlet(new HelloServlet("Bye!"), "/bye/*");

	server.start();
	server.join();
    }

    /**
     * Returns the port this server is listening on.
     */
    public synchronized int getServerPort() {
	return socketListener.getLocalPort();
    }

    /**
     * Returns the name of the directory, if any, that contains the
     * files being served as static content.
     *
     * @see #setDocRoot(String docRoot, String docPath)
     */
    public synchronized String getDocRoot() {
	return docRoot;
    }

    /**
     * Returns the base path of the URLs that correspond to the files
     * being served as static content.
     *
     * @see #setDocRoot(String docRoot, String docPath)
     */
    public synchronized String getDocPath() {
	return docPath;
    }

    /**
     * Tells this server to serve files from the tree below the
     * specified root directory and to use the specified path
     * in the corresponding URLs.
     *
     * <p>If this method has not been called, no files will be served.
     * Otherwise, file docRoot/... will be available as URL
     * http://serverHost:serverPort/docPath/..., except that
     * docPath must contain the initial "/".
     */
    public synchronized void setDocRoot(String docRoot, String docPath) {
	this.docRoot = docRoot;
	this.docPath = docPath;
    }

    /**
     * Tells this server to write an NCSA-format request log in the
     * specified directory.  The log file's name will have the form
     * <tt>prefix_yyyy_mm_dd.log</tt>.
     */
    public synchronized void setLogPath(String dir, String prefix) {
	this.logDir = dir;
	this.logPrefix = prefix;
    }

    /**
     * Setup that's done for the constructor when the server is created.
     */
    private void initialSetup() {

	Debug.noteln("Initial setup for I-X HttpServer");

	// Server
	server = new Server();

	// Port listener
	socketListener =
	    new org.mortbay.jetty.bio.SocketConnector();
//  	    new org.mortbay.jetty.nio.SelectChannelConnector();
	// Set the port.  0 means to ask the OS for a free port.
	socketListener.setPort(listenPort);

	server.setConnectors(new Connector[] {socketListener});

	// Contexts
	contexts = new ContextHandlerCollection();

	// Context for servlets
 	Context context = new Context(contexts, "/");
	servlets = context.getServletHandler();

    }

    /**
     * Setup that's done for the start() method just before the
     * server is started.
     */
    private void finalSetup() {

	// Handler for static content
	if (docRoot != null) {
	    //\/: This works and it's not clear what else would.
	    Context c = new Context(contexts, docPath);
	    c.setResourceBase(docRoot);
	    c.addServlet(new ServletHolder(new DefaultServlet()), "/*");
	}

	// Request log handler
	if (logDir != null) {
	    // Make the log-handler.
	    RequestLogHandler logHandler = makeRequestLogHandler();
	    // Set handlers with the log-handler last.
	    HandlerCollection handlers = new HandlerCollection();
	    handlers.setHandlers(new Handler[] {contexts, logHandler});
	    server.setHandler(handlers);
	}
	else {
	    server.setHandler(contexts);
	}

    }

    /**
     * Makes a RequestLogHandler that will write an NCSA-format log
     * in the logDir directory.
     */
    private RequestLogHandler makeRequestLogHandler() {
	RequestLogHandler logHandler = new RequestLogHandler();
	String slash = System.getProperty("file.separator");
	String path = logDir + slash + logPrefix + "_yyyy_mm_dd.log";
	NCSARequestLog requestLog = new NCSARequestLog(path);
        requestLog.setRetainDays(90);
        requestLog.setAppend(true);
        requestLog.setExtended(false);
        requestLog.setLogTimeZone("GMT");
        logHandler.setRequestLog(requestLog);
	return logHandler;
    }

    /**
     * Add a servlet serving URLs that match the specified path.
     * For example:
     * <pre>
     *   server.addServlet(new HelloServlet("Hello!"), "/hello/*");
     * </pre>
     */
    public synchronized void addServlet(HttpServlet servlet, String pathSpec) {
	servlets.addServletWithMapping(new ServletHolder(servlet), pathSpec);
    }

    public void addServlet(String pathSpec, HttpServlet servlet) {
	addServlet(servlet, pathSpec);
    }

    /**
     * Starts the server running.
     */
    public synchronized void start() {
	// Start the server
	try {
	    finalSetup();
	    server.start();
	    Debug.noteln("Server port:", getServerPort());
	}
	catch (Throwable t) {
	    throw new RethrownException
		("Couldn't start the HTTP server. ", t);
	}
    }

    public synchronized void join() {
	try {
	    server.join();
	}
	catch (InterruptedException e) {
	}
    }

}
