/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri Dec  7 16:45:47 2007 by Jeff Dalton
 * Copyright: (c) 2005 - 2007, AIAI, University of Edinburgh
 */

package ix.util.http;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import java.net.*;
import java.io.*;
import java.util.*;

import ix.util.*;

import ix.util.lisp.SafeLispInterpreter; // for testing

/**
 * Handles HTTP requests that can be regarded as sending an object
 * and receiving one in reply.  The usual I-X XML encoding is used;
 * however, it's possible to change that in a subclass.
 *
 * <p>An HttpObjectServelt is like an {@link HttpObjectClient} but on
 * the server side: where a client sends an object as XML, and receives
 * an object as XML in reply, the servlet receives an object as XML,
 * and sends one back.
 *
 * <p>At present, only POST requests are supported.
 *
 * @see HttpServer
 * @see ix.util.xml.XMLTranslator
 */
public abstract class HttpObjectServlet extends HttpServlet {

    protected HttpUtilities util = new HttpUtilities();

    protected String responseContentType = "application/xml";
    protected String responseCharsetName = "UTF-8";

    public HttpObjectServlet() {
    }

    /**
     * Set the Content-Type used when sending a response.
     */
    public void setResponseContentType(String type) {
	this.responseContentType = type;
    }

    /**
     * Set the character set used when sending a response.
     */
    public void setResponseCharsetName(String name) {
	this.responseCharsetName = name;
    }

    /**
     * Handle an HTTP POST request.
     */
    @Override
    protected void doPost(HttpServletRequest req,
			  HttpServletResponse resp)
	      throws ServletException, IOException {

	describeRequest(req);

	String content;
	try {
	    content = readRequest(req);
	}
	catch (HttpRequestException e) {
	    Debug.displayException("Problem reading request", e);
	    resp.sendError(e.getStatus(), e.getReason());
	    return;
	}
	
	Debug.noteln("\nRecieved:", content);

	Object reply;
	try {
	    Object received = decodeReceived(content);
	    reply = handleRequest(req, received);
	}
	catch (HttpRequestException e) {
	    Debug.displayException("Problem handling request", e);
	    resp.sendError(e.getStatus(), e.getReason());
	    return;
	}
	catch (Throwable t) {
	    Debug.displayException("Problem handling request", t);
	    resp.sendError
		(HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
		 "Internal server error: " + Debug.describeException(t));
	    return;
	}

	byte[] bytes = encodeForSend(reply);
	sendResponse(bytes, resp);

    }

    /**
     * Utility method that produces debugging output that describes
     * the request.
     */
    protected void describeRequest(HttpServletRequest req) {
	Debug.noteln("HTTP request URI:", req.getRequestURI());
	Debug.noteln("\nRequest headers:");
	for (Enumeration e = req.getHeaderNames(); e.hasMoreElements();) {
	    String name = (String)e.nextElement();
	    Debug.noteln(name + ": " + req.getHeader(name));
	}
    }

    /**
     * Utility method that reads the request's contents and returns
     * it as a string.
     */
    protected String readRequest(HttpServletRequest req) 
	throws HttpRequestException,
	       IOException {

	// Read the request data.
	InputStream in = req.getInputStream();
	String contentType = req.getContentType();
	String charset = req.getCharacterEncoding();
	int len = req.getContentLength();

	Debug.noteln("Request Content-Type: " + contentType +
		     "; charset=" + charset +
		     "; length=" + len);

	try {
	    return util.readContent(in, len, charset);
	}
	catch (EOFException eof) {
	    throw new HttpRequestException
		(HttpServletResponse.SC_BAD_REQUEST,
		 "Data ended prematurely.");
	}
	finally {
	    in.close();
	}

    }

    /**
     * Turns request contents into an object.  For example, the
     * content might be an XML representation of an object.
     *
     * @see #encodeForSend(Object contents)
     */
    protected Object decodeReceived(String contents) {
	return util.decodeReceived(contents);
    }

    /**
     * The main part of handling the request.  The contents have
     * already been read and turned into an object by calling
     * {@link #decodeReceived(String)}.  The HttpServletRequest
     * is passed primarily to allow this method to look at the
     * request headers.
     *
     * @return the object to encode and send as a response.
     *
     * @throws HttpRequestException to indicate that an error
     *    response should be sent.  The status value and message
     *    will be taken from the exception.
     */
    protected abstract Object handleRequest(HttpServletRequest req,
					    Object received)
	throws HttpRequestException;

    /**
     * Turns an object into response contents.
     *
     * @see #decodeReceived(String contents)
     */
    protected byte[] encodeForSend(Object contents)
	throws UnsupportedEncodingException {
	return util.encodeForSend(contents, responseCharsetName);
    }

    /**
     * Utility method that writes the response contents.
     */
    protected void sendResponse(byte[] bytes, HttpServletResponse resp)
	throws IOException {
	// Send a response
	resp.setContentType(responseContentType);
	resp.setCharacterEncoding(responseCharsetName);
	resp.setContentLength(bytes.length);
	resp.setStatus(resp.SC_OK);

	ServletOutputStream stream = resp.getOutputStream();
	stream.write(bytes);
	stream.flush();
	stream.close();
    }

    /**
     * Main program for testing.  It creates an {@link HttpServer}
     * and adds an HttpObjectServlet (subclass) for path "/take";
     * the servlet acts as a {@link SafeLispInterpreter} for the objects
     * it's sent.  Each object that's received is treated as an
     * expression and evaluated; the result is sent back as the reply.
     * The <tt>http-server-port</tt> parameter can be used to say
     * what port the server should use.
     */
    public static void main(String[] argv) {

	Parameters.processCommandLineArguments(argv);

	final SafeLispInterpreter lin = new SafeLispInterpreter();

	class TestServlet extends HttpObjectServlet {
	    protected Object handleRequest(HttpServletRequest req,
					   Object content) {
		return lin.topLevelEval(content);
	    }
	}

	int port = Parameters.getInt("http-server-port", 0);
	HttpServer server = new HttpServer(port);
	server.addServlet(new TestServlet(), "/take");
	server.start();

    }

}
