/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Dec 12 03:39:33 2007 by Jeff Dalton
 * Copyright: (c) 2003, 2007, AIAI, University of Edinburgh
 */

package ix.iserve.ipc;

import java.net.*;
import java.io.*;
import java.util.*;
import javax.swing.SwingUtilities;

import ix.icore.IXAgent;
import ix.ip2.Ip2;
import ix.iface.util.Reporting;
import ix.util.*;
import ix.util.ipc.ServiceAddress;
import ix.util.xml.*;
import ix.util.http.*;
import ix.util.lisp.*;

/**
 * A communication strategy that sends all messages via a server.
 * An agent that uses this strategy must specify an "ipc-server"
 * parameter.
 *
 * @see Parameters
 */
public class IServeCommunicationStrategy
       implements IPC.CommunicationStrategy {

    IServeCommunicationTool tool; // a user interface.

    boolean ableToSend = false;

    // One higher than the highest we've received so far.
    // Volatile because registerAs refers to it and is called
    // in the GUI event thread.  /\/
    volatile int expectedSeqNo = 0;

    // We need to know which messages we've received, but all
    // we really need to keep is a record of their sequence numbers --
    // the keys in the messageMap. /\/ For now, we are keeping the
    // values as well.
    MessageMemory messageMap = new MessageMemory();

    HttpObjectClient http = new HttpObjectClient();

    HttpUtilities httpUtil = new HttpUtilities();

    Ip2 agent;
    ServiceAddress serverAddr;
    URL messageUrl;

    IPC.MessageListener messageListener;

    public IServeCommunicationStrategy() {
    }

    void setTool(IServeCommunicationTool tool) {
	// This method exists because the tool doesn't exist until
	// the user asks for it.
	this.tool = tool;
    }

    String getAgentName() {
	return agent.getAgentSymbolName();
    }

    synchronized boolean isAbleToSend() {
	return ableToSend;
    }
    synchronized void setIsAbleToSend(boolean v) {
	ableToSend = v;
    }

    synchronized URL getMessageUrl() {
	return messageUrl;
    }

    public synchronized void setupServer(Object destination,
					 IPC.MessageListener listener) {

	Debug.noteln("Setting up IServeCommunicationStrategy server.");
	messageListener = listener;
	agent = (Ip2)IXAgent.getAgent();
	String ipcServer = Parameters.getParameter("ipc-server");
	if (ipcServer != null) {
	    serverAddr = new ServiceAddress(ipcServer);
	    messageUrl = httpUtil.makeMessageURL(serverAddr, "/iscs-message");
	}
	else {
	    Util.displayAndWait(null,
	      "No ipc-server was given for the I-Serve communication strategy");
	    return;
	}

	agent.addTool(new IServeCommunicationTool.Controller(this));

    }

    void registerAs(String name) {
	// Called only from the communication tool.
	Debug.expect(SwingUtilities.isEventDispatchThread(), "not in Swing");
	Debug.expect(!isAbleToSend(), "Registering when already registered");
	Debug.noteln("Trying to register as", name);

	MessageWrapper request =
	    new MessageWrapper(getAgentName(), "register-as", name);

	// Let the message-server know the highest-numbered message
	// we've received (in case the server exited and was restarted).
	// N.B. If we haven't yet received anything, the value will be -1.
	request.setSeqNo(expectedSeqNo - 1);

	Object reply = requestObject(request);
	if (!reply.equals("ok"))
	    throw new IPC.IPCException(reply.toString());

	transcript("Registered as " + name);
	agent.setAgentSymbolName(name);
	setIsAbleToSend(true);
	new ReceiveThread().start();
    }

    void serverUnavailable() {
	Util.swingAndWait(new Runnable() {
	    public void run() {
		tool.needToReregister();
	    }
	});
    }

    class ReceiveThread extends Thread {

	String ourName = getAgentName();

	public void run() {
	    MessageWrapper message = null;
	messageLoop:
	    while(true) {
		// Get a message
		try {
		    message = nextMessage(message);
		}
		catch (IPC.BrokenConnectionException b) {
		    transcript("Connection problem: " +
			       Debug.describeException(b));
		    Debug.displayException
			("Exception while waiting for input.  " +
			 "Probably means the message server has exited.  " +
			 "Another exception may follow.  " +
			 "This exception was",
			 b);
		    // No message to handle.
		    continue messageLoop;
		}
		catch (ConnectFailure f) {
		    setIsAbleToSend(false);
		    transcript("Connection problem: " +
			       Debug.describeException(f));
		    Debug.displayException(f);
		    serverUnavailable();
		    return;	// exit this thread
		}
		catch (Throwable t) {
		    transcript("Receive failure: " +
			       Debug.describeException(t));
		    Debug.displayException
			("Exception while waiting for input.  " +
			 "Probably a timeout that can be ignored.  " +
			 "The exception was",
			 t);
		    // No message to handle.
		    continue messageLoop;
		}
		// Handle the message
		try {
		    handleMessage(message);
		}
		catch (Throwable t) {
		    transcript(Debug.describeException(t));
		    Debug.displayException(t);
		}
	    }
	}

	MessageWrapper nextMessage(MessageWrapper previous) {
	    transcript("Waiting for next message ...");

	    MessageWrapper request = 
		new MessageWrapper(getAgentName(), "get-message");

	    // Let the message-server know we received the previous message.
	    if (previous != null)
		request.setSeqNo(previous.getSeqNo());

	    Object reply = requestObject(request);
	    if (!(reply instanceof MessageWrapper))
		throw new ClassCastException
		    ("Received an instance of " + reply.getClass().getName() +
		     " instead of a MessageWrapper.");

	    // The message should have the form
	    //   MessageWrapper(fromName, "send-to", [ourName, message])
	    MessageWrapper next = (MessageWrapper)reply;
	    Debug.expect(next.getCommand().equals("send-to"));
	    Debug.expect(next.getArg(0).equals(ourName));
	    return next;
	}

	void handleMessage(MessageWrapper message) {
	    // Perhaps we've already received this message, but
	    // the server didn't know we had and so sent it again.
	    // Or we may have missed a message while the server
	    // thought it had been sent successfully.
	    Integer mSequence = message.getSequenceNumber();
	    int mSeq = message.getSeqNo();
	    Debug.expect(mSeq == mSequence.intValue());
	    if (mSeq == expectedSeqNo) {
		Debug.expect(!messageMap.containsKey(mSequence));
		expectedSeqNo++;
	    }
	    else if (mSeq > expectedSeqNo) {
		Debug.noteln("Missing messages between " + expectedSeqNo +
			     " and " + mSeq);
		expectedSeqNo = mSeq + 1;
	    }
	    else {
		Debug.expect(mSeq < expectedSeqNo);
		if (messageMap.containsKey(mSequence)) {
		    // We've already received this message.
		    transcript("Received " + mSeq + " again");
		    // So ignore it.
		    return;
		}
		else {
		    // This message fills a gap in the previous messages.
		    Debug.noteln("Received gap message", mSeq);
		}
	    }
	    // We have a new message.
	    Object contents = message.getArg(1);
	    transcript("Received " + mSeq + ": " +
		       Reporting.description(contents));
	    messageMap.remember(message); 	//\/ won't GC message
	    messageListener.messageReceived
		(new IPC.BasicInputMessage(contents));
	}

    }

    protected void transcript(final String line) {
	Debug.noteln("Transcript:", line);
	Util.swingAndWait(new Runnable() {
	    public void run() {
		do_transcript(Reporting.dateString() + " " + line);
	    }
	});
    }

    protected void do_transcript(String line) {
	tool.transcript(line);
    }

    public synchronized void sendObject(Object destination, Object contents) {
	if (!isAbleToSend())
	    throw new UnsupportedOperationException
		("Cannot send to " + destination +
		 " until you have registered with the message server.");
	transcript("Sending to " + destination + ": " +
		   Reporting.description(contents));
	MessageWrapper command =
	    new MessageWrapper
	        (getAgentName(), "send-to", destination, contents);
	command.setSendDate(new Date());
	try {
	    Object reply = requestObject(command);
	    Debug.noteln("Received", reply);
	    if (!reply.equals("ok"))
		throw new IPC.IPCException
		    ("Unexpected reply from message-server: " + reply);
	}
	catch (RuntimeException e) {
	    Debug.noteln("sendObject rethrowing", Debug.describeException(e));
	    throw e;
	}
	catch (Exception e) {
	    Debug.noteException(e);
	    throw new RethrownException(e);
	}
    }

    /**
     * Sends a request to the message server and returns the server's reply.
     * A string reply that reports an exception is turned into an appropriate
     * exception and then thrown, rather than being returned.
     */
    protected Object requestObject(MessageWrapper m) {
	try {
	    return http.sendRequest(messageUrl, m);
	}
	catch (RethrownIOException e) {
	    // Includes MalformedURLException
	    Debug.noteException(e);
	    throw interpretRequestException(e);
	}
    }

    RuntimeException interpretRequestException(RethrownIOException e) {
	return e;
    }

    static class ConnectFailure extends IPC.IPCException {
	ConnectFailure(String message) {
	    super(message);
	}
    }

}
