/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Dec 12 04:01:00 2007 by Jeff Dalton
 * Copyright: (c) 2003, 2007, AIAI, University of Edinburgh
 */

package ix.iserve.ipc;

import java.util.*;
import java.io.*;
import java.net.*;

import ix.iface.util.Reporting;
import ix.util.*;
import ix.util.ipc.*;
import ix.util.xml.*;
import ix.util.http.*;
import ix.util.lisp.*;

/**
 * Handles messages for an {@link IServeCommunicationStrategy}.
 */
public class IServeCommunicationServer {

    ServiceAddress addr =
	new ServiceAddress(Parameters.getParameter("ipc-server"));

    Strategy strategy = new Strategy(addr);

    Date startupDate = new Date();

    public IServeCommunicationServer() {
    }

    public static void main(String[] argv) {
	Util.printGreeting("Applet Message Server");

	Parameters.setIsInteractive(false);
	Parameters.processCommandLineArguments(argv);
	Debug.on = Parameters.getBoolean("debug", Debug.on);
	Debug.noteThreads = true;

	//\/ When exiting, could try starting a new server using 
	// Runtime.getRuntime().exec(...)

	try {
	    new IServeCommunicationServer().start();
	}
	catch (Throwable t) {
	    Debug.noteException(t);
	    System.exit(1);
	}
	Debug.noteln("Exiting main method for no good reason");
	System.exit(2);
    }

    /**
     * A communication strategy used by both the message-server
     * and its clients.  It gives "message-server" an address
     * with a host and port specified by a {@link ServiceAddress}.
     */
    static class Strategy extends IPC.SimpleIXCommunicationStrategy {

	ServiceAddress serverAddr;

	Strategy(ServiceAddress serverAddr) {
	    super();
	    this.serverAddr = serverAddr;
	    // setDestinationData("message-server", serverAddr);
	    setDestinationAddress("message-server", serverAddr);
	}

    }

    /*
     * The server
     */

    void start() throws Exception {
	Debug.noteln("Started " + startupDate + " at " + addr);

	ServerSocket servSock = new ServerSocket(addr.getPort());
	
	while (true) {
	    Socket s = servSock.accept();
	    Debug.noteln(new Date() + " Client connection", s);
	    serveClientOn(new ObjectStreamConnection(s));
	}

    }

    void serveClientOn(final ObjectStreamConnection connection) {
	new CatchingThread() {
	    public void innerRun() {
		clientService(connection);
	    }
	    protected void handleException(Throwable t) {
		Debug.noteln("Exception not handled", t);
		Debug.noteException(t);
		try { connection.close(); }
		catch (Throwable t2) {
		    Debug.noteln("Exception when closeing connection.");
		    Debug.noteException(t2);
		}
	    }
	}.start();
    }

    void clientService(ObjectStreamConnection connection) {
	try {
	    Object contents = strategy.postDecode(connection.receive());
	    handleMessage(connection, contents);
	}
	catch (IPC.BrokenConnectionException e) {
	    // Happens when the other agent exits, so more or less
	    // "normal" termination.  /\/: Remove registration?
	    Debug.noteln("Message-server lost connection to",
			 connection.getDestination());
	    connection.close();
	}
    }

    void handleMessage(ObjectStreamConnection connection,
		       Object contents) {
	Debug.noteln("Message server received", contents);
	Object reply = null;
	try {
	    MessageWrapper message = unpackRequest(contents);
	    // learnHostName(message);
	    reply = evalMessage(message, connection);
	}
	catch (Throwable t) {
	    Debug.noteException(t);
	    reply = "Server exception: " + Debug.describeException(t);
	}
	if (reply != null) {
	    // N.B. evalGetMessage returns null
	    sendReply(connection, reply);
	    connection.close();
	}
    }

    MessageWrapper unpackRequest(Object contents) {
	// CGIRelay sends us an MessageWrapper that contains the
	// host name and addr from the CGI request and, as its
	// command, the original MessageWrapper from the applet
	// represented as a string of XML.
	MessageWrapper outer = (MessageWrapper)contents;
	if (outer.getCommand().equals("server-status")) {
	    // No inner MessageWrapper in this case;
	    return outer;
	}
	MessageWrapper inner = 
	    (MessageWrapper)XML.objectFromXML(outer.getCommand());
	inner.setRemoteHost(outer.getRemoteHost());
	inner.setRemoteAddr(outer.getRemoteAddr());
	return inner;
    }

    Object sendReply(ObjectStreamConnection connection,
		     Object reply) {
	Debug.noteln(new Date() + " Trying to send reply:", reply);
	// CGIRelay is expecting the reply as an XML string
	String asXML = XML.objectToXMLString(reply);
	Debug.noteln("The reply as XML:", asXML);
	connection.send(strategy.preEncode(asXML));
	Debug.noteln("Sent:", reply);
	Object status = strategy.postDecode(connection.receive());
	Debug.noteln("Status from sending reply " + reply + ":", status);
	return status;
    }

    Object evalMessage(MessageWrapper m, ObjectStreamConnection conn) {
	String asXML = XML.objectToXMLString(m);
	Debug.noteln(new Date() + " Evaluating", asXML);
	String command = m.getCommand();
	if (command.equals("register-as")) {
	    return evalRegisterAs(m);
	}
	else if (command.equals("send-to")) {
	    return evalSendTo(m);
	}
	else if (command.equals("get-message")) {
	    return evalGetMessage(m, conn);
	}
	else if (command.equals("server-status")) {
	    return evalServerStatus(m);
	}
	else
	    throw new UnsupportedOperationException
		("Unknown message command " + command);
    }

    String evalRegisterAs(MessageWrapper m) {
	// Syntax: MessageWrapper(sender, "register-as", [name])
	//   where sender is not really meaningful.
	String name = (String)m.getArg(0);
	User already = getUser(name);
	if (already == null) {
	    // New user
	    Debug.expect(m.getRemoteAddr() != null,
			 "No host addr for user " + Strings.quote(name));
	    User user = new User(name, m); 
	    recordUser(user);
	    user.registered(m, true);
	    return "ok";
	}
	// If a user of that name is already registered, allow
	// it only if from the same host.
	else if (m.getRemoteAddr().equals(already.getHostAddr())) {
	    // The user agent
	    already.getConnectionQueue().clear();
	    already.registered(m, false);
	    return "ok";
	}
	else
	    throw new IllegalArgumentException
		("Another user is already registered as " +
		 Strings.quote(name));
    }

    String evalSendTo(MessageWrapper m) {
	// Syntax: MessageWrapper(sender, "send-to", [toName, contents]) 
	checkSender(m);
	String toName = (String)m.getArg(0);
	User to = getUser(toName);
	if (to == null)
	    throw new IllegalArgumentException
		("There is no user named " + Strings.quote(toName));
	// Give the message to the user's message thread.
	to.addMessage(m);
	return "ok";
    }

    MessageWrapper evalGetMessage(MessageWrapper m,
				 ObjectStreamConnection conn) {
	// Syntax: MessageWrapper(sender, "get-message")
	checkSender(m);
	User user = getUser(m.getFrom());
	// The sequence number tells us what the agent that sent
	// the get-message most recently received.  So now we know
	// the agent really received it.
	if (m.getSequenceNumber() != null)
	    user.acknowledgedReceipt(m.getSequenceNumber());
	// If there's an unacknowledged message, try sending it
	// again.
	MessageWrapper u = user.getFirstUnacknowledgedMessage();
	if (u != null)
	    user.pushMessage(u);
	// Give the connection to the user's message thread so that
	// a reply can be sent once one is available.
	user.takeConnection(conn);
	return null;		// no reply yet.
    }

    void checkSender(MessageWrapper m) {
	// Ensure user is known and from correct host.
	String fromName = m.getFrom();
	User from = getUser(fromName);
	if (from == null)
	    throw new IllegalArgumentException
		("Message from unknown user " + Strings.quote(fromName));
	if (!m.getRemoteAddr().equals(from.getHostAddr()))
	    throw new IllegalArgumentException
		("Message from wrong user " + Strings.quote(fromName));
	from.sent(m);
    }

    String evalServerStatus(MessageWrapper m) {
	// /\/: We don't call checkSender(m) in this case,
	// because MessageWrapper doesn't have one.  (Indeed,
	// the request might have come from a random browser.)
	List lines = new LinkedList();
	Date now = new Date();
	lines.add("Status as of " + now);
	// Startup date
	lines.add("Started running " + agoTime(startupDate, now));
	// Log file name
	lines.add("Log file: " +
		  Parameters.getParameter("server-log-file", "none"));
	// User status
	List users = getUsers();
	if (users.isEmpty())
	    lines.add("No users");
	else {
	    lines.add("");
	    lines.add("Users:");
	    for (Iterator i = getUsers().iterator(); i.hasNext();) {
		User u = (User)i.next();
		lines.add(u.status(now));
		lines.add("");
	    }
	}
	return Strings.joinLines(lines);
    }


    /*
     * Utilities
     */

    protected String agoTime(Date then, Date now) {
	Duration delta = new Duration(then, now).roundToMinutes();
	return then +
	    (delta.asMilliseconds() > 0 	// means minutes > 0
	         ? ", " + delta + " ago"
	         : "");
    }


    /*
     * User profiles
     */

    private Map userNameToUserMap = new TreeMap();

    protected User getUser(String name) {
	synchronized(userNameToUserMap) {
	    return (User)userNameToUserMap.get(name);
	}
    }

    protected List getUsers() {
	synchronized(userNameToUserMap) {
	    List result = new LinkedList();
	    for (Iterator i = userNameToUserMap.entrySet().iterator();
		 i.hasNext();) {
		Map.Entry e = (Map.Entry)i.next();
		result.add(e.getValue());
	    }
	    return result;
	}
    }

    protected void recordUser(User user) {
	user.noteln("Recording user");
	synchronized(userNameToUserMap) {
	    String name = user.getName();
	    Debug.expect(getUser(name) == null,
			 "User " + name + " already exists");
	    userNameToUserMap.put(name, user);
	    user.startMessageThread();
	}
    }

    protected void removeUser(User user) {
	user.noteln("Removing user");
	synchronized(userNameToUserMap) {
	    user.killMessageThread();
	    userNameToUserMap.remove(user);
	}
    }

    protected class User {

	String name;
	String hostName;
	String hostAddr;
	MessageQueue messageQueue = new MessageQueue(true);
	MessageQueue connectionQueue = new MessageQueue(true);
	MessageMemory unacknowledged = new MessageMemory();
	UserMessageThread messageThread;
	List registrationDates = new LinkedList();
	Date lastContactDate = null;    // last contact with this server
	Date lastSendDate = null; 	// last sent to another agent
	Date lastAckDate = null; 	// last new ack of other-agent message

	int seqNo = 0;		// for messages from other agents to this user

	User(String name, MessageWrapper registration) {
	    this.name = name;
	    this.hostName = registration.getRemoteHost();
	    this.hostAddr = registration.getRemoteAddr();
	    // The thread uses the user name in its name, and so
	    // it can't be created until after the user name is available.
	    this.messageThread = new UserMessageThread(this);
	}

	synchronized String getName() { return name; }
	synchronized String getHostAddr() { return hostAddr; }
	synchronized MessageQueue getMessageQueue() { return messageQueue; }
	synchronized MessageQueue getConnectionQueue() {
	    return connectionQueue;
	}
	synchronized void sent(MessageWrapper m) {
	    Date now = new Date();
	    lastContactDate = now;
	    if (m.getCommand().equals("send-to"))
		lastSendDate = now;
	}
	synchronized List getRegistrationDates() {
	    return new ArrayList(registrationDates); // snapshot
	}

	synchronized void registered(MessageWrapper m, boolean isNew) {
	    Date rdate = new Date();
	    lastContactDate = rdate;
	    noteln(rdate + " " +
		   (isNew ? "new registration" : "re-registration"));
	    Debug.expect(registrationDates.isEmpty() == isNew);
	    registrationDates.add(rdate);
	    // If the user is new to this server, but has received
	    // some messages [seq > -1], that probably means it
	    // talked with an earlier server that then exited.
	    // If the user's new to us, we can't have sent it anything,
	    // so we have to make any messages we have (or get)
	    // look new to the user's agent by giving them numbers
	    // above those the agent's already received.
	    int seq = m.getSeqNo();
	    noteln("Seq numbers: reg says " + seq + ", we have " + seqNo);
	    if (isNew && seq > -1) {
		noteln("Renumbering messages");
		final int delta = seq + 1;
		// Renumber any pending messages.  /\/ Since we don't
		// yet allow mail to unregistered users, there won't
		// be any messages.
		messageQueue.callOnContents(new Proc() {
		    public void call(Object arg) {
			List contents = (List)arg;
			for (Iterator i = contents.iterator(); i.hasNext();) {
			    MessageWrapper m = (MessageWrapper)i.next();
			    int mSeq = m.getSeqNo();
			    int toSeq = mSeq + delta;
			    noteln("renumbering " + mSeq + " to " + toSeq);
			    m.setSeqNo(toSeq);
			}
		    }
		});
		seqNo += delta;
		noteln("Revised next-message number " + seqNo);
	    }
	}

	synchronized void addMessage(MessageWrapper m) {
	    noteln("Adding message " + seqNo);
	    m.setSeqNo(seqNo++);
	    remember(m);
	    messageQueue.send(m);
	}

	synchronized void pushMessage(MessageWrapper m) {
	    // This is a message already sent, but the send may have
	    // failed without an exception to say it did.
	    Debug.expect(m.getSequenceNumber() != null);
	    // Debug.expect(unacknowledged.containsKey(m.getSequenceNumber()));
	    noteln("Pushing message with key", m.getSequenceNumber());
	    messageQueue.push(m);
	}

	private void remember(MessageWrapper m) {
	    Integer key = m.getSequenceNumber();
	    if (!unacknowledged.containsKey(key)) {
		noteln("Remembering message with key", key);
		unacknowledged.remember(m);
	    }
	}

	synchronized boolean isUnacknowledged(MessageWrapper m) {
	    return unacknowledged.containsKey(m.getSequenceNumber());
	}

	synchronized MessageWrapper getFirstUnacknowledgedMessage() {
	    MessageWrapper m = unacknowledged.getFirstRemainingMessage();
	    if (m != null)
		noteln("Getting 1st message; it has key",
		       m.getSequenceNumber());
	    return m;
	}

	synchronized void acknowledgedReceipt(Integer key) {
	    Date now = new Date();
	    lastContactDate = now;
	    if (unacknowledged.containsKey(key)) {
		noteln("Acknowledged receipt of message", key);
		lastAckDate = now;
		unacknowledged.forgetKey(key);
	    }
	}

	synchronized void takeConnection(ObjectStreamConnection c) {
	    connectionQueue.send(c);
	}

	synchronized void startMessageThread() {
	    messageThread.start();
	}

	synchronized void killMessageThread() {
	    messageThread.exit = true;
	    messageThread.interrupt();
	}

	void noteln(String line) { 			// convenience method
	    Debug.noteln(this + ": " + line);
	}

	void noteln(String text, Object about) { 	// convenience method
	    Debug.noteln(this + ": " + text, about);
	}

	public synchronized String toString() {
	    return "User[" + name + "]";
	}

	public synchronized String status(Date now) {
	    final List lines = new LinkedList();
	    // Name and address
	    lines.add(name + " at " + hostName + " " + hostAddr);
	    // Times since ...
	    lines.add("  Time of last active contact: " +
		      agoTime(lastContactDate, now));
	    if (lastSendDate != null)
		lines.add("  Last send to another agent:  " +
			  agoTime(lastSendDate, now));
	    if (lastAckDate != null)
		lines.add("  Last receipt acknowledgment: " +
			  agoTime(lastAckDate, now));
	    // Registration dates
	    List regs = registrationDates;
	    if (!regs.isEmpty()) {
		// Probably an error if it was empty. /\/
		lines.add("  Registration times:");
		for (Iterator i = regs.iterator(); i.hasNext();) {
		    Date d = (Date)i.next();
		    lines.add("    " + agoTime(d, now));
		}
	    }
	    // Waiting messages
	    if (unacknowledged.size() > 0) {
		lines.add("  Waiting messages:");
		unacknowledged.walkContents(new Proc() {
		    public void call(Object key, Object val) {
			MessageWrapper m = (MessageWrapper)val;
			lines.add("    " + describeMessage(m));
		    }
		});
	    }
	    return Strings.joinLines(lines);
	}

	protected String describeMessage(MessageWrapper m) {
	    Object contents = m.getArg(1); 		// /\/
	    Class c = contents.getClass();
	    String className = XML.nameForClass(c);
	    return m.getSendDate() + " " + className + " from " + m.getFrom();
	}

    }

    protected class UserMessageThread extends Thread {

	User user;

        volatile boolean exit = false;  // gets the thread to exit.

	UserMessageThread(User user) {
	    super(user + " messages");
	    this.user = user;
	}

	public void run() {
	    try {
		transferMessages();
	    }
	    catch (Throwable t) {
		user.noteln("Uncaught exception in message thread: " +
			    Debug.describeException(t));
		Debug.noteException(t);
	    }
	    user.noteln("Message thread exits");
	}

	void transferMessages() {
	    // /\/: Now that we have the MessageMemory, consider rewriting
	    // this to make the outer loop get the next connection.
	    MessageQueue mQ = user.getMessageQueue();
	    MessageQueue cQ = user.getConnectionQueue();
	messageLoop:
	    while (!exit) {
		// First get a message
		Object m = mQ.nextMessage();
		if (m instanceof InterruptedException) {
		    user.noteln("Message thread interrupted");
		    continue messageLoop;
		}
		MessageWrapper am = (MessageWrapper)m;
		user.noteln("Have " + am + ", key " + am.getSequenceNumber());
		// Now try to send the message to our user's agent.
		// Keep getting new connections until one works.
	    connectionLoop:
		while (!exit && m != null) {
		    ObjectStreamConnection conn =
			(ObjectStreamConnection)cQ.nextMessage();
		    user.noteln("Taking connection", conn);
		    user.noteln("Trying to send", m);
		    if (!user.isUnacknowledged(am)) {
			Debug.noteln("Oops, user already has", m);
			cQ.push(conn);
			continue messageLoop;
		    }
		    try {
			Object status = sendReply(conn, m);
			if (status.equals("ok")) {
			    conn.close();
			    user.noteln("Successfully sent", m);
			    m = null; 		// success!
			}
			else
			    user.noteln("Send had status", status);
		    }
		    catch (Throwable t) {
			Debug.noteln("Exception during send:",
				     Debug.describeException(t));
		    }
		}
	    }
	}

    }

}
