/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Dec  6 17:37:08 2007 by Jeff Dalton
 * Copyright: (c) 2005 - 2007, AIAI, University of Edinburgh
 */

package ix.http;

import org.jdom.Document;
import org.jdom.output.XMLOutputter;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import java.net.*;
import java.io.*;
import java.util.*;

import ix.util.ipc.ServiceAddress;

import ix.util.*;
import ix.util.xml.*;
import ix.util.http.*;

/**
 * A communication strategy that uses HTTP to transport messages.
 */
public class HttpCommunicationStrategy implements IPC.CommunicationStrategy {

    private Map<String,URL> nameToAddrMap =
	new Hashtable<String,URL>();	        // synchronized

    private URL nameServerURL;
    private HttpNameServer.Syntax syntax = new HttpNameServer.Syntax();

    private String thisAgentsName;

    private IPC.MessageListener messageListener;

    private HttpObjectClient http = new HttpObjectClient();
    private HttpUtilities util = new HttpUtilities();

    private HttpServer server;
    private int listenPort;

    public HttpCommunicationStrategy() {
	this(Parameters.getInt("port", 0));
    }

    public HttpCommunicationStrategy(int port) {
	this.listenPort = port;
    }

    /*
     * Setting up the communication strategy.
     */

    public void setupServer(Object destination, IPC.MessageListener listener) {

	thisAgentsName = (String)destination;
	messageListener = listener;

	// Get the name-server's host and port; make them into a URL.
	ServiceAddress addr =
	    new ServiceAddress
	        (Parameters.getParameter("name-server",
					 HttpNameServer.DEFAULT_ADDRESS));
	nameServerURL = makeMessageURL(addr);

	// Server side
	server = new HttpServer(listenPort);

	// Servlet
 	server.addServlet(new CommServlet(), "/http-cs-message");

	// Start the server and see what port it's actually using.
	server.start();
	listenPort = server.getServerPort();

	registerWithNameServer();

    }

    private URL makeMessageURL(ServiceAddress addr) {
	return util.makeMessageURL(addr, "/http-cs-message");
    }

    protected void registerWithNameServer() {
	ServiceAddress addr =
	    new ServiceAddress(util.getHostName(), listenPort);
	Object message =
	    syntax.makeRegisterMessage(thisAgentsName, addr);
	Object reply =
	    requestObject(nameServerURL, message);
	Debug.expectEquals
	    ("OK", reply,
	     "Problem registering with name-server.");
    }

    /*
     * Sending
     */

    public void sendObject(Object destination, Object contents) {
	try {
	    Object result = requestObject((String)destination, contents);
	    Debug.expectEquals
		("OK", result,
		 "Invalid response from " + destination + ". ");
	}
	catch (RethrownIOException e) {
	    Debug.displayException(e);
	    // Perhaps the destination exited and will restart.
	    // Since it may not restart at the same address:
	    nameToAddrMap.remove(destination);
	}
    }

    private Object requestObject(String destination, Object contents) {
	return requestObject(getAddress(destination), contents);
    }

    private Object requestObject(URL url, Object contentsToSend) {
	return http.sendRequest(url, contentsToSend);
    }

    private URL getAddress(String destination) {
	URL addr = nameToAddrMap.get(destination);
	if (addr == null) {
	    addr = askNameServer(destination);
	    nameToAddrMap.put(destination, addr);
	}
	return addr;
    }

    private URL askNameServer(String destination) {
	String agentName = destination;
	Object message = syntax.makeLookupMessage(agentName);
	ServiceAddress addr =
	    (ServiceAddress)
	        Util.mustBe(ServiceAddress.class,
			    requestObject(nameServerURL, message));
	return makeMessageURL(addr);
    }

    /*
     * Receiving
     */

    protected Object handleInput(Object contents) {
	messageListener.messageReceived(new IPC.BasicInputMessage(contents));
	return "OK";
    }

    private class CommServlet extends HttpObjectServlet {

	private CommServlet() {
	}

	@Override
	    protected Object handleRequest(HttpServletRequest req,
					   Object contents) {
	    return handleInput(contents);
	}

    }

}
