/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Feb 23 19:52:42 2004 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.util.context;

import junit.framework.*;

import java.util.*;

import ix.util.*;
import ix.util.lisp.*;

/**
 * Simple context-related test cases for ContextHashMaps.
 */
public class ContextHashMapContextTest extends TestCase {

    public ContextHashMapContextTest(String name) {
	super(name);
    }

    protected void setUp() {
	Context.clearContexts();
    }

    protected void tearDown() {
	Context.clearContexts();
    }

    public void testInThreeContexts() {
	ContextHashMap colour = new ContextHashMap();
	// Base context: a child of the root context.
	Context baseContext = Context.pushContext();
	// In base, apple -> green, sky -> blue
	colour.put("apple", "green");
	colour.put("sky", "blue");
	assertEquals("green", colour.get("apple"));
	assertEquals("blue", colour.get("sky"));
	// c1, a child of the base context
	Context c1 = Context.pushContext();
	// Check inherited values
	assertEquals("green", colour.get("apple"));
	assertEquals("blue", colour.get("sky"));
	// In c1, apple -> red; check values
	colour.put("apple", "red");
	assertEquals("red", colour.get("apple"));
	assertEquals("blue", colour.get("sky"));
	// Back to base context; check
	Context.popContext();
	assertSame(baseContext, Context.getContext());
	assertEquals("green", colour.get("apple"));
	assertEquals("blue", colour.get("sky"));
	// c2, a different child of the base
	Context c2 = Context.pushContext();
	// Check inherited values are unaffected by what happened in c1
	assertEquals("green", colour.get("apple"));
	assertEquals("blue", colour.get("sky"));
	// In c2, sky -> gray; check
	colour.put("sky", "grey");
	assertEquals("grey", colour.get("sky"));
	// Back to base
	Context.popContext();
	// Check values
	assertSame(baseContext, Context.getContext());
	assertEquals("green", colour.get("apple"));
	assertEquals("blue", colour.get("sky"));
    }

    public void testEntryVisibility () {
	ContextHashMap colour = new ContextHashMap();
	// In base, apple -> green
	Context baseContext = Context.pushContext();
	colour.put("apple", "green");
	// In c1, sky -> blue
	Context c1 = Context.pushContext();
	colour.put("sky", "blue");
	Context.popContext();
	// In c2, snow -> white
	Context c2 = Context.pushContext();
	colour.put("snow", "white");
	Context.popContext();
	// Check base
	// With a ContextHashMap, we can rely on the order of keys
	// and values.
	assertTrue(colour.containsKey("apple"));
	assertFalse(colour.containsKey("sky"));
	assertFalse(colour.containsKey("snow"));
	assertEquals(Lisp.list("apple"), LList.newLList(colour.keySet()));
	assertEquals(Lisp.list("green"), LList.newLList(colour.values()));
	// Check c1
	Context.setContext(c1);
	assertTrue(colour.containsKey("apple"));
	assertTrue(colour.containsKey("sky"));
	assertFalse(colour.containsKey("snow"));
	assertEquals(Lisp.list("apple", "sky"),
		     LList.newLList(colour.keySet()));
	assertEquals(Lisp.list("green", "blue"),
		     LList.newLList(colour.values()));
	// Check c2
	Context.setContext(c2);
	assertTrue(colour.containsKey("apple"));
	assertFalse(colour.containsKey("sky"));
	assertTrue(colour.containsKey("snow"));
	assertEquals(Lisp.list("apple", "snow"),
		     LList.newLList(colour.keySet()));
	assertEquals(Lisp.list("green", "white"),
		     LList.newLList(colour.values()));
	// Try a child of c2 with sky -> grey
	Context c2_1 = Context.pushContext();
	colour.put("sky", "grey");
	assertTrue(colour.containsKey("apple"));
	assertTrue(colour.containsKey("sky"));
	assertTrue(colour.containsKey("snow"));
	// Even though sky was added last here, it was added earlier
	// in the total history of the ContextHashMap and so will
	// appear 2nd.
	assertEquals(Lisp.list("apple", "sky", "snow"),
		     LList.newLList(colour.keySet()));
	assertEquals(Lisp.list("green", "grey", "white"),
		     LList.newLList(colour.values()));
	// A child of c2_1, with no changes
	Context c2_1_1 = Context.pushContext();
	assertEquals(Lisp.list("apple", "sky", "snow"),
		     LList.newLList(colour.keySet()));
	assertEquals(Lisp.list("green", "grey", "white"),
		     LList.newLList(colour.values()));
	// Check iterator
	Iterator i = colour.entrySet().iterator();
	assertTrue(i.hasNext());
	assertTrue(i.hasNext());	// do hasNext() twice in a row
	Map.Entry e = (Map.Entry)i.next();
	assertEquals("apple", e.getKey());
	assertEquals("green", e.getValue());
	assertTrue(i.hasNext());
	e = (Map.Entry)i.next();
	e = (Map.Entry)i.next();	// do next() twice in a row
	assertEquals("snow", e.getKey());
	assertEquals("white", e.getValue());
	// A new child of the base, adding rose -> red
	Context.setContext(baseContext);
	Context c3 = Context.pushContext();
	colour.put("rose", "red");
	assertEquals(Lisp.list("apple", "rose"),
		     LList.newLList(colour.keySet()));
	assertEquals(Lisp.list("green", "red"),
		     LList.newLList(colour.values()));
    }

    public void testWhenCreatedInSibling() {
	// A context, c1
	Context c1 = Context.pushContext();
	ContextHashMap colour = new ContextHashMap();
	assertTrue(colour.isEmpty());
	colour.put("sky", "blue");
	// c2, a sibling of c1
	Context.popContext();
	Context c2 = Context.pushContext();
	assertSame(c1.getParent(), c2.getParent());
	// Map should be empty here.
	assertTrue(colour.isEmpty());
    }

}
