/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Nov  9 05:50:14 2004 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.util;

import java.util.*;

import junit.framework.*;

import ix.util.lisp.*;

/**
 * Test cases for graph utilities.
 */
public class GraphsTest extends AbstractGraphsTest {

    public GraphsTest(String name) {
	super(name);
    }

    public void testMapToDirectedGraphAndBack() {
	// We need this test because we want to know that we can
	// control the order in which vertices are visited,
	// e.g. when testing transitive closures.
	LList alist = readGraphLList("((a (b)) (b (d c)) (c (d)) (d ()))");
	Map m = alist.alistToMap(new StableHashMap());
	DirectedGraph g = Graphs.makeDirectedGraph(m);
	Map m2 = Graphs.toMap(new StableHashMap(), g);
	assertEquals(m, m2);
	// Since we've been using StableHashMaps, we should
	// also be able to get back the orginal alist.
	LList alist2 = LList.mapToAlist(m2);
	assertEquals(alist, alist2);
    }

    public void testMinimalElements() {
	LList alist = readGraphLList("((a (b)) (b (d c)) (c (d)) (d ()))");
	Map m = alist.alistToMap(new StableHashMap());
	DirectedGraph g = Graphs.makeDirectedGraph(m);
	Set minimals = Graphs.minimalElements(g);
	assertEquals(readSet("a"), minimals);
    }

    public void testMaximalElements() {
	LList alist = readGraphLList("((a (b)) (b (d c)) (c (d)) (d ()))");
	Map m = alist.alistToMap(new StableHashMap());
	DirectedGraph g = Graphs.makeDirectedGraph(m);
	Set maximals = Graphs.maximalElements(g);
	assertEquals(readSet("d"), maximals);
    }

    public void testTranspose() {
	LList alist = readGraphLList("((a (b)) (b (d c)) (c (d)) (d ()))");
	Map m = alist.alistToMap(new StableHashMap());
	DirectedGraph g = Graphs.makeDirectedGraph(m);
	DirectedGraph t = Graphs.transpose(g);
	checkGraph
	    (readGraphLList
	       ("((a ())" +
		" (b (a))" + 
		" (c (b))" +
		" (d (b c)))"),
	     t);
    }

    public void testFanComposition() {
	// Country -> cities
	LList countryToCities =
	    readMapLList
	      ("((scotland (edinburgh glasgow))" +
	       " (england (london york)))");
	// City -> attractions
	LList cityToAttractions =
	    readMapLList
	      ("((edinburgh (castle parliament))" +
	       " (glasgow (football iron-bru))" +
	       " (london (eye tate foyles parliament))" +
	       " (york (minster)))");
	// Compose to get a country -> attractions map.
	LList countries = Lisp.elementsFromString("scotland england");
	FanComposer fc = new FanComposer(countries) {
	    protected Collection makeValueCollection() {
		return new LinkedList();
	    }
	};
	fc.apply(countryToCities.alistToMap(new StableHashMap()));
	fc.apply(Graphs.makeDirectedGraph
		   (cityToAttractions.alistToMap(new StableHashMap())));
	Map countryToAttractions = fc.getResultMap();
	checkMapStrictly
	    (readMapLList
	       ("((scotland (castle parliament football iron-bru))" +
		" (england (eye tate foyles parliament minster)))"),
	     countryToAttractions);
	// Attraction -> types
	LList attractionToTypes =
	    readMapLList
	      ("((castle (building old))" +
	       " (parliament (building political))" +
	       " (football (sport))" +
	       " (iron-bru (drink))" +
	       " (eye (wheel view))" +
	       " (tate (building art))" +
	       " (foyles (building bookshop))" +
	       " (minster (building old church)))");
	fc.apply(attractionToTypes.alistToMap(new StableHashMap()));
	Map countryToTypes = fc.getResultMap();
	// Note that the composed uses lists for the value collections
	// and so there can be duplicate elements.
	checkMapStrictly
	    (readMapLList
	       ("((scotland" +
		"  (building old building political sport drink))" +
		" (england" +
		"   (wheel view building art" +
		"    building bookshop building political" +
		"    building old church)))"),
	     countryToTypes);
    }

}
