/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Aug 19 00:47:27 2004 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.util;

import java.util.*;

import junit.framework.*;

import ix.util.lisp.*;

/**
 * Shared test cases for transitive-closure algorithms.
 */
public abstract class AbstractTransitiveClosureTest
       extends AbstractGraphsTest {

    public AbstractTransitiveClosureTest(String name) {
	super(name);
    }

    protected abstract TransitiveClosure
	               makeTransitiveClosure(DirectedGraph g);

    // The tests in this file all compute the transitive closure of
    // a directed acyclic graph (DAG), so that the more limited
    // DAG algorithm can be used.  However, the same tests are
    // also run with other algorithms, which is why this abstract
    // class exists.

    public void testDAGTransitiveClosure1() {
	LList alist = readGraphLList("((a (b)) (b (d c)) (c (d)) (d ()))");
	Map m = alist.alistToMap(new StableHashMap());
	DirectedGraph g = Graphs.makeDirectedGraph(m);
	TransitiveClosure tc = makeTransitiveClosure(g);
	checkGraph
	    (readGraphLList("((a (b d c)) (b (d c)) (c (d)) (d ()))"),
	     tc);
    }

    public void testDAGTransitiveClosure2() {
	// This test was copied from O-Plan.  It drove a bug in an
        // early version of tclosure when run in Lucid 4.1, but that
	// was a different algorithm.  It also had (d (e d)), which
	// the ix.util.DAGTransitiveClosure algorithm won't accept
	// because it's a cycle.
	LList alist = readGraphLList
	    ("((e (i)) (a (b c)) (b (d e f)) (d (e f)) (i (j)) (k ()))");
	Map m = alist.alistToMap(new StableHashMap());
	DirectedGraph g = Graphs.makeDirectedGraph(m);
	TransitiveClosure tc = makeTransitiveClosure(g);
	checkGraph
	    (readGraphLList
	       ("((a (b c f e d i j))" +
                " (b (f e d i j))" +
		" (c ())" +
                " (d (e f i j))" +
                " (e (i j))" +
		" (f ())" +	// no g or h for some reason
                " (i (j))" +
		" (j ())" +
		" (k ()))"),
	     tc);
    }

    public void testDAGTransitiveClosure3() {
	// This test was also copied from O-Plan.  it's the same as
	// the previous test but has the keys in the order that
	// drove a bug when the test was run in Lucid 4.1.  To get
	// exactly the same test as in Lucid 4.1, we have to control
        // the order of the keys by putting them in the desired
	// order in the initial alist and by using a StableHashMap
	// to make the DirectedGraph.  The order we want is:
	// (i k b d a e).
	LList alist = readGraphLList
	    ("((i (j)) (k ()) (b (d e f)) (d (e f)) (a (b c)) (e (i)))");
	Map m = alist.alistToMap(new StableHashMap());
	DirectedGraph g = Graphs.makeDirectedGraph(m);
	TransitiveClosure tc = makeTransitiveClosure(g);
	checkGraph
	    (readGraphLList
	       ("((a (b c f e d i j))" +
                " (b (f e d i j))" +
		" (c ())" +
                " (d (e f i j))" +
                " (e (i j))" +
		" (f ())" +	// no g or h for some reason
                " (i (j))" +
		" (j ())" +
		" (k ()))"),
	     tc);
    }

}
