/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Jan 19 14:53:29 2004 by Jeff Dalton
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 */

package ix.util;

import java.util.*;

import junit.framework.*;

import ix.util.lisp.*;

/**
 * ListIterator test cases.
 */
public abstract class AbstractListIteratorTest extends AbstractIteratorTest {

    public AbstractListIteratorTest(String name) {
	super(name);
    }

    protected List makeEmptyList() {
	return (List)makeEmptyCollection();
    }

    protected List makeList(Collection source) {
	return (List)makeCollection(source);
    }

    protected Iterator getIterator(Collection c) {
	return ((List)c).listIterator(); // we're testing the ListIterator.
    }

    public void testEmptyIterator() {
	super.testEmptyIterator();
	ListIterator li = (ListIterator)getIterator(makeEmptyList());
	assertTrue(!li.hasPrevious());
	assertEquals(0, li.nextIndex());
	assertEquals(-1, li.previousIndex());
	try {
	    li.previous();
	    fail("Should have thrown NoSuchElementException");
	}
	catch (NoSuchElementException e) {
	}
    }

    public void testElementRetrieval() {
	List abc = makeList(Lisp.list(STR_A, STR_B, STR_C));
	ListIterator abc_li = abc.listIterator();
	assertTrue(abc_li.hasNext());
	assertSame(STR_A, abc_li.next());
	assertTrue(abc_li.hasNext());
	assertSame(STR_B, abc_li.next());
	assertTrue(abc_li.hasNext());
	assertSame(STR_C, abc_li.next());
	assertTrue(!abc_li.hasNext());
    }

    public void testRepeatNextAndPrevious() {
	List abc = makeList(Lisp.list(STR_A, STR_B, STR_C));
	ListIterator abc_li = abc.listIterator();
	assertSame(STR_A, abc_li.next());
	assertSame(STR_A, abc_li.previous());
	assertSame(STR_A, abc_li.next());
	assertSame(STR_A, abc_li.previous());
	assertSame(STR_A, abc_li.next());
	assertSame(STR_A, abc_li.previous());

	assertSame(STR_A, abc_li.next());
	assertSame(STR_B, abc_li.next());
	assertSame(STR_B, abc_li.previous());
	assertSame(STR_A, abc_li.previous());
    }

    public void testIndexValue() {
	List abc = makeList(Lisp.list(STR_A, STR_B, STR_C));
	ListIterator abc_li = abc.listIterator();

	assertTrue(!abc_li.hasPrevious());
	assertTrue(abc_li.hasNext());

	assertTrue(abc_li.previousIndex() == -1);
	assertTrue(abc_li.nextIndex() == 0);

	// Go forward

	assertSame(STR_A, abc_li.next());
	assertTrue(abc_li.previousIndex() == 0);
	assertTrue(abc_li.nextIndex() == 1);

	assertSame(STR_B, abc_li.next());
	assertTrue(abc_li.previousIndex() == 1);
	assertTrue(abc_li.nextIndex() == 2);

	assertSame(STR_C, abc_li.next());
	assertTrue(abc_li.previousIndex() == 2);
	assertTrue(abc_li.nextIndex() == 3);

	assertTrue(abc_li.hasPrevious());
	assertTrue(!abc_li.hasNext());

	// Go back

	assertSame(STR_C, abc_li.previous());
	assertTrue(abc_li.previousIndex() == 1);
	assertTrue(abc_li.nextIndex() == 2);

	assertSame(STR_B, abc_li.previous());
	assertTrue(abc_li.previousIndex() == 0);
	assertTrue(abc_li.nextIndex() == 1);

	assertSame(STR_A, abc_li.previous());
	assertTrue(abc_li.previousIndex() == -1);
	assertTrue(abc_li.nextIndex() == 0);

	assertTrue(!abc_li.hasPrevious());
	assertTrue(abc_li.hasNext());
    }

    public void testFirstElementRemoval() {
	List abc = makeList(Lisp.list(STR_A, STR_B, STR_C));
	ListIterator abc_li = abc.listIterator();

	// Remove after a next().
	assertSame(STR_A, abc_li.next());
	abc_li.remove();
	assertTrue(haveSameElements(Lisp.list(STR_B, STR_C), abc));

	// Then after a next() followed by a previous().
	assertSame(STR_B, abc_li.next());
	assertSame(STR_B, abc_li.previous()); // has failed
	abc_li.remove();
	assertTrue(haveSameElements(Lisp.list(STR_C), abc));

	// Remove after two next()s followed by a previous().
	// And this time start with a new List.
	abc = makeList(Lisp.list(STR_A, STR_B, STR_C));
	abc_li = abc.listIterator();
	assertSame(STR_A, abc_li.next());
	assertSame(STR_B, abc_li.next());
	assertSame(STR_B, abc_li.previous());
	abc_li.remove();
	assertSame(STR_A, abc_li.previous());
	assertTrue(haveSameElements(Lisp.list(STR_A, STR_C), abc));

	// Again, this time with a (new) 2-element list.
	abc = makeList(Lisp.list(STR_A, STR_B));
	abc_li = abc.listIterator();
	assertSame(STR_A, abc_li.next());
	assertSame(STR_B, abc_li.next());
	assertSame(STR_B, abc_li.previous());
	abc_li.remove();
	assertSame(STR_A, abc_li.previous());
	assertTrue(haveSameElements(Lisp.list(STR_A), abc));

    }

    // /\/: Not clear where this method should be.
    // /\/: It now appears in > 1 class.
    public static boolean haveSameElements(List a, List b) {
	Iterator ai = a.iterator(), bi = b.iterator();
	while (ai.hasNext() && bi.hasNext()) {
	    if (!ai.next().equals(bi.next()))
		return false;
	}
	if (ai.hasNext() || bi.hasNext())
	    return false;
	else
	    return true;
    }

}
