/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri Mar 25 07:21:15 2005 by Jeff Dalton
 * Copyright: (c) 2005, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.io.*;
import java.util.*;

import junit.framework.*;

import ix.icore.process.TimePoint;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;
import ix.util.xml.*;

/**
 * {@link TimePointNet} test cases.
 *
 * <p>A typical test has the form
 * <ul>
 * <li>Start with an enpty net.
 * <li>Add some points.
 * <li>Add some contraints, checking whether an invalidity
 *     is detected.
 * <li>Check the final min and max values of the points.
 * </ul></p>
 *
 * <p>Points are affected only by adding constraints, not by
 * changing a min or max directly; and there are no tests for
 * deleting points or constraints, because the TPN algorithm
 * doesn't support deletions or indeed anything that loosens
 * contraints.</p>
 */

public class TimePointNetTest extends TestCase {

    public TimePointNetTest(String name) {
	super(name);
    }

    /*
     * O-Plan's TPN tests
     */

    static final Object testGroupSyntax =
	Lisp.readFromString("(define-test-group ?name &rest ?clauses)");

    static final Object testSyntax =
	Lisp.readFromString
	    ("((test-net ?points ?constraints &rest ?options)" +
	     " ==> ?expected-result)");

    static final Object pointSyntax =
	Lisp.readFromString("(?name ?min ?max)");

    static final Object constraintSyntax =
	Lisp.readFromString("(?from ?to ?min ?max)");

    static final LList RESET_NIL =
	(LList)Lisp.readFromString("(:reset nil)");

    static final Symbol INFINITY = Symbol.intern(":infinity");

    static Object valueOf(String name, MatchEnv e) {
	Object value = e.get(Symbol.intern(name));
	if (value != null)
	    return value;
	else
	    throw new IllegalArgumentException("No value for " + name);
    }

    public void testOPlanTpnTests() throws IOException {
	LispReader lin = Lisp.openForInput("ix/iplan/oplan-tpn-tests.lsp");
	while (true) {
	    Object in = lin.readObject();
	    if (in == Lisp.EOF)
		break;
	    runOPlanTestGroup((LList)in);
	}
    }

    void runOPlanTestGroup(LList testGroup) {
	MatchEnv e = SimpleMatcher.mustMatch(testGroupSyntax, testGroup);
	LList clauses = (LList)valueOf("?clauses", e);
	for (Iterator i = clauses.iterator(); i.hasNext();) {
	    LList test = (LList)i.next();
	    runOPlanTest(test);
	}
    }

    void runOPlanTest(LList test) {
	// Debug.noteln("O-Plan test", test);
	Debug.note("Tpn");
	MatchEnv e = SimpleMatcher.mustMatch(testSyntax, test);
	LList points = (LList)valueOf("?points", e);
	LList constraints = (LList)valueOf("?constraints", e);
	LList options = (LList)valueOf("?options", e);
	LList expectedResult = (LList)valueOf("?expected-result", e);
	boolean reset = !options.equals(RESET_NIL);
	if (reset) {
	    new TestNet();
	}
	TestNet.net.extendNet(points);
	List failedConstraints = new LinkedList();
	for (Iterator i = constraints.iterator(); i.hasNext();) {
	    LList constraint = (LList)i.next();
	    if (!TestNet.net.tryConstraint(constraint))
		failedConstraints.add(constraint);
	}
	if (failedConstraints.isEmpty())
	    assertEquals("point descriptions",
			 expectedResult,
			 TestNet.net.listPointValues());
	else
	    assertEquals("failed constraints",
			 expectedResult,
			 failedConstraints);
    }

    static class TestNet {

	static TestNet net = null; // there's never > 1 TestNet at a time

	TimePointNet tpn = new TimePointNet();
	Map pointMap = new StableHashMap();
	List correctPointValues = new LinkedList();

	TestNet() {
	    net = this;
	}

	TimePoint getPoint(Symbol name) {
	    TimePoint result = (TimePoint)pointMap.get(name);
	    Debug.expect(result != null, "Can't find time-point", name);
	    return result;
	}

	void extendNet(LList pointDescriptions) {
	    for (Iterator i = pointDescriptions.iterator(); i.hasNext();) {
		LList descr = (LList)i.next();
		MatchEnv e = SimpleMatcher.mustMatch(pointSyntax, descr);
		Symbol name = (Symbol)valueOf("?name", e);
		long min = theLong(valueOf("?min", e));
		long max = theLong(valueOf("?max", e));
		pointMap.put(name, tpn.addTimePoint(min, max));
	    }
	    correctPointValues.addAll(pointDescriptions);
	    checkPointValues("initial-setup");
	}

	boolean tryConstraint(LList constraint) {
	    MatchEnv e = SimpleMatcher.mustMatch(constraintSyntax, constraint);
	    Symbol from = (Symbol)valueOf("?from", e);
	    Symbol to = (Symbol)valueOf("?to", e);
	    long min = theLong(valueOf("?min", e));
	    long max = theLong(valueOf("?max", e));
	    TimePoint fromTP = getPoint(from);
	    TimePoint toTP = getPoint(to);
	    if (tpn.addTimeConstraint(fromTP, toTP, min, max)) {
		// The constraint has been added.
		// Remember the point values as correct.
		// If we cannot add the next constraint,
		// these are the values that should be restored.
		correctPointValues = listPointValues();
		return true;
	    }
	    else {
		// The constraint couldn't be added.
		// Check that the TPN restored the values to
		// what they were before we tried adding the constraint.
		checkPointValues(constraint);
		return false;
	    }
	}

	long theLong(Object obj) {
	    return (obj == INFINITY) ? Inf.INFINITY : ((Long)obj).longValue();
	}

	void checkPointValues(Object descrOfSuspect) {
	    assertEquals("point values, suspect " + descrOfSuspect,
			 correctPointValues,
			 listPointValues());
	}

	List listPointValues() {
	    List result = new LinkedList();
	    for (Iterator i = pointMap.entrySet().iterator(); i.hasNext();) {
		Map.Entry e = (Map.Entry)i.next();
		Symbol name = (Symbol)e.getKey();
		TimePoint tp = (TimePoint)e.getValue();
		Long min = new Long(tp.getMinTime());
		Object max = tp.hasUnboundedMax()
		    ? (Object)INFINITY
		    : (Object)new Long(tp.getMaxTime());
		result.add(Lisp.list(name, min, max));
	    }
	    return result;
	}

    }

}
