/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Sep  1 02:09:35 2005 by Jeff Dalton
 * Copyright: (c) 2002 - 2005, AIAI, University of Edinburgh
 */

package ix.util.xml;

import java.lang.reflect.*;
import java.util.*;

import java.net.URL;
import java.io.*;

// Imports for using JDOM
// import java.io.IOException;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.Attribute;
import org.jdom.Namespace;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.XMLOutputter;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.reflect.*;

/**
 * A class for translating between objects and XML.  Note that a
 * single XMLTranslator maps in both directions to make it easier
 * to create mappings that are consistent. <p>
 */

public class XMLTranslator implements Loader, Saver {

    protected ClassSyntax classSyntax;

    protected XMLOutputter outputter = XML.makePrettyXMLOutputter();

    protected Function1 prefilter = null;
    protected Function1 postfilter = null;

    protected String implAttributeName;
    protected String mapEntryName;

    protected boolean omitImplAttributes = false;

    protected Element lastExaminedElement = null; // for error reporting

    protected Class defaultListClass = LinkedList.class;
    protected Class defaultSetClass = HashSet.class;
    protected Class defaultMapClass = HashMap.class;

    public XMLTranslator() {
	this(XML.config().defaultClassSyntax());
    }

    public XMLTranslator(ClassSyntax syntax) {
	this.classSyntax = syntax;
	this.implAttributeName =
	    classSyntax.externalNameForField("implClass");
	this.mapEntryName =
	    classSyntax.getClassFinder().externalName("MapEntry");
    }

    public ClassSyntax getClassSyntax() {
	return classSyntax;
    }

    public XMLOutputter getOutputter() {
	return outputter;
    }

    public void setPrefilter(Function1 f) {
	this.prefilter = f;
    }

    public void setPostfilter(Function1 f) {
	this.postfilter = f;
    }

    /**
     * Controlls whether or not <code>implClass</code> attributes
     * are added to List, Set, and Map elements when converting to XML.
     * Note that if it's set to true, the XML will not contain enough
     * information to determine the original implementation classes
     * and the translator will have to do its best to make reasonable
     * choices based on field classes and the defaults.
     */
    public void setOmitImplAttributes(boolean v) {
	omitImplAttributes = v;
    }

    /* *************************** *
     *      The XML Namespace      *
     * *************************** */

    /** Is an I-X namespace required in input XML?  Initially false. */
    protected boolean requireNamespace = false;

    /**
     * The namespace used in elements that represent object classes
     * and fields.  It is defined without a prefix and so will be
     * declared as the default namespace for the elements in which
     * it's used.  (That may sound odd from an XML point of view,
     * but it's how it works in JDOM.)
     */
    protected Namespace homeNamespace = XML.config().getHomeNamespace();
    // /\/: It would be nice if we didn't have to refer to XML.config() here.

    /** The URI for the home namespace. */
    protected String homeNamespaceURI = homeNamespace.getURI();

    /**
     * Returns the home namespace used by this XMLTranslator.
     */
    public Namespace getHomeNamespace() {
	return homeNamespace;
    }

    /** Size 1 cache for fast comparisons. 
     * @see #isHomeNamespace(Namespace) */
    protected Namespace mostRecentHomeNamespace = homeNamespace;

    /**
     * Utility that creates an element in the home namespace.
     */
    protected Element nsElement(String name) {
	return new Element(name, homeNamespace);
    }

    /**
     * Controls whether element names must be in the home namespace
     * for the elements to be translatable to objects.
     *
     * @see #looksLikeAnObjectDocument(Document)
     */
    public void setRequireNamespace(boolean v) {
	requireNamespace = v;
    }

    protected boolean isHomeNamespace(Namespace namespace) {
	// Only the URI ought to matter, but rather than do a
	// string compare each time we take advantage of the JDOM
	// fact that there is only one Namespace object for each
	// prefix,URI pair and that the URI of mostRecentHomeNamespace
	// has already been checked.  /\/: This could be extended
	// to a list of the I-X Namespace objects we've seen.
	if (namespace == mostRecentHomeNamespace)
	    return true;
	else {
	    if (isHomeNamespaceURI(namespace.getURI())) {
		mostRecentHomeNamespace = namespace;
		return true;
	    }
	    else
		return false;
	}
    }

    protected boolean isHomeNamespaceURI(String uri) {
	// /\/: startsWith rather than Equals for now.
	return uri.startsWith(homeNamespaceURI);
    }


    /* ************************ *
     *      Objects to XML      *
     * ************************ */

    public void writeObject(Object obj, File file) throws IOException {
	final Document doc = objectToDocument(obj);
	final OutputStream out = 
	    new BufferedOutputStream(new FileOutputStream(file));
	final XMLOutputter outputter = XML.makePrettyXMLOutputter();
	Util.run(new WithCleanup() {
	    public void body() throws IOException {
		outputter.output(doc, out);
		out.flush();
	    }
	    public void cleanup() throws IOException {
		out.close();
	    }
	});
    }

    /**
     * Converts an object to a string of XML.  The object is converted
     * to a JDOM Document by {@link #objectToDocument(Object)} and then
     * to a string by {@link #documentToXMLString(org.jdom.Document)}.
     */
    public String objectToXMLString(Object obj) {
	return documentToXMLString(objectToDocument(obj));
    }

    /**
     * Converts a JDOM Document to a string of XML.
     */
    public synchronized String documentToXMLString(Document doc) {
	StringWriter writer = new StringWriter();
	try {
	    outputter.output(doc, writer);
	}
	catch (IOException e) {
	    Debug.noteException(e);
	    // Throw because we have nothing meaningful to return
	    // and stray nulls or ""s are a pain.
	    throw new XMLException
		("Can't convert " + doc + " to string", e);
	}
	writer.flush();
	return writer.toString();
    }

    /**
     * Converts an object to a JDOM Document.
     */
    public synchronized Document objectToDocument(Object obj) {
	return new Document(objectToElement(obj));
    }

    /**
     * Converts an object to a JDOM Element.  This is the central
     * recursion point for the conversion process; it determines
     * which of the <code><i>type</i>ToElement(Object)</code> 
     * methods to call.
     *
     * <p>If the prefilter is not null, it is called on the object
     * before translation begins, and the result is used instead
     * the original object.  (Of course, the prefilter could just
     * return the original object, but this enables it to supply
     * a substitute.)
     *
     * @see #setPrefilter(Function1)
     */
    protected Element objectToElement(Object obj) {
	if (prefilter != null) obj = prefilter.funcall(obj);
	if (obj == null)	// irritating special case /\/
	    return nsElement("null");
	else if (obj instanceof String)
	    return stringToElement(obj);
	else {
	    ClassDescr cd = classSyntax.getClassDescr(obj.getClass());
	    if (cd.isPrimitive())
		return primitiveObjectToElement(obj);
	    else if (cd.isList())
		return listToElement(obj);
	    else if (cd.isSet())
		return setToElement(obj);
	    else if (cd.isMap())
		return mapToElement(obj);
	    else if (cd.isStruct())
		return structToElement(obj);
	    else if (cd.isXML())
		return embeddedXMLToElement(obj);
	    else
		throw new ConsistencyException("Unknown type " + cd);
	}
    }

    protected Element primitiveObjectToElement(Object obj) {
	// This is for instances of the Object equivalents of the primitive
	// types -- of Long, for example, rather than Long.TYPE.
	Class c = obj.getClass();
	String cname = classSyntax.externalNameForClass(c);
	Stringer s = classSyntax.getStringer(c);
	String text = s != null ? s.toString(obj) : obj.toString();
	Element elt = nsElement(cname);
	elt.setText(text);
	return elt;
    }

    protected Element stringToElement(Object obj) {
	// JDOM encodes when the string contains XML.
	return primitiveObjectToElement(obj);
    }

    protected Element embeddedXMLToElement(Object obj) {
	LiteralDocument xml = (LiteralDocument)obj;
	String cname = classSyntax.externalNameForClass(LiteralDocument.class);
	Element elt = nsElement(cname);
	// /\/: Have to make a (deep) clone of embedded Element,
	// because putting it in the result Element changes its
	// parent.
	Element contentElt = (Element)xml.getRootElement().clone();
	elt.addContent(contentElt);
	return elt;
    }

    protected Element listToElement(Object obj) {
	return collectionToElement(List.class, (List)obj);
    }

    protected Element setToElement(Object obj) {
	return collectionToElement(Map.class, (Set)obj);
    }

    protected Element collectionToElement(Class iface, Collection obj) {
	// If the elt type is always "List" or "Set" (with the class
	// as an attribute), rather than itself being the class name,
	// it will be clear from the XML that we are looking
	// at a list (without having to look up the class) -
	// and thus that we're breaking the strict alternation
	// between field and class names.
	Class c = obj.getClass();
	String cname = classSyntax.externalNameForClass(c);
	String iname = classSyntax.externalNameForClass(iface);
	Element elt = nsElement(iname);
	if (!omitImplAttributes)
	    setImplAttribute(elt, cname);
	for (Iterator i = obj.iterator(); i.hasNext();) {
	    Object item = i.next();
	    elt.addContent(objectToElement(item));
	}
	return elt;
    }

    protected Element mapToElement(Object obj) {
	Map map = (Map)obj;
	Class c = map.getClass();
	String cname = classSyntax.externalNameForClass(c);
	String iname = classSyntax.externalNameForClass(Map.class);
	Element mapElt = nsElement(iname);
	if (!omitImplAttributes)
	    setImplAttribute(mapElt, cname);
	for (Iterator i = map.entrySet().iterator(); i.hasNext();) {
	    Map.Entry entry = (Map.Entry)i.next();
	    Element entryElt = nsElement(mapEntryName);
	    // /\/: Can't assume "key" and "value" are ok as external names?
	    Element keyElt = nsElement("key");
	    Element valElt = nsElement("value");
	    mapElt.addContent(entryElt);
	    entryElt.addContent(keyElt);
	    entryElt.addContent(valElt);
	    keyElt.addContent(objectToElement(entry.getKey()));
	    valElt.addContent(objectToElement(entry.getValue()));
	    // Maybe just alternate keys and values without
	    // entry elements?  /\/
	}
	return mapElt;
    }

    protected void setImplAttribute(Element elt, String className) {
	elt.setAttribute(implAttributeName, className);
    }

    protected Element structToElement(Object obj) {
	Class c = obj.getClass();
	ClassDescr cd = classSyntax.getClassDescr(c);
	String cname = cd.getExternalName();
	Element elt = nsElement(cname);
	for (Iterator i = cd.getFieldDescrs().iterator(); i.hasNext();) {
	    FieldDescr fd = (FieldDescr)i.next();
	    visitField(obj, fd, elt);
	}
	return elt;
    }

    protected void visitField(Object obj, FieldDescr fd, Element objElt) {
	String fieldName = fd.getExternalName();
	try {
	    Object value = fd.getValue(obj);
	    if (value == null) return;
	    Class valClass = fd.getType();
	    if (isAttributeClass(valClass)) {
		// Add attribute
		// /\/: Need to check that value is a direct
		// instance of the field class?  And make it
		// an element if it isn't?
		Stringer s = classSyntax.getStringer(valClass);
		String text = s != null ? s.toString(value) : value.toString();
		objElt.setAttribute(fieldName, text);
	    }
	    else {
		// Add child element
		Element fieldElt = nsElement(fieldName);
		objElt.addContent(fieldElt);
		fieldElt.addContent(objectToElement(value));
	    }
	}
	catch (Exception e) {
	    Debug.noteException(e);
	    throw new XMLException
		("Cannot get value of field " + fd.getName()
		 + " of class " + obj.getClass().getName(),
		 e);
	}
    }

    public boolean isAttributeClass(Class c) {
	// N.B. Not all classes whose ClassDescr isPrimitive() == true.
	return c.isPrimitive()
	    || SemiPrimitive.class.isAssignableFrom(c)
	    || Number.class.isAssignableFrom(c)
	    || Boolean.class.isAssignableFrom(c);
    }


    /* ************************** *
     *      Objects from XML      *
     * ************************** */

    public Object readObject(URL url) throws IOException {
	return objectFromDocument(XML.parseXML(url));
    }

    /**
     * Converts a string of XML to an object.
     */
    public synchronized Object objectFromXML(String text) {
	Document doc = XML.parseXML(text);
	return objectFromDocument(doc);
    }

    /**
     * Converts a JDOM Document to an object.  This method, or one
     * that calls it, is what should normally be called from outside
     * the translator - rather than calling objectFromElement(elt)
     * directly.
     */
    public synchronized Object objectFromDocument(Document doc) {
	Object result = objectFromElement(doc.getRootElement());
	lastExaminedElement = null;	// no longer needed
	return result;
    }

    public boolean looksLikeAnObjectDocument(Document doc) {
	Element root = doc.getRootElement();
	Namespace rootNS = root.getNamespace();
	if (requireNamespace && !isHomeNamespace(rootNS))
	    return false;
	if (!(isHomeNamespace(rootNS) || rootNS == Namespace.NO_NAMESPACE))
	    return false;
	// See if the element name corresponds to a class
	return classSyntax.classForExternalName(root.getName()) != null;
    }

    /**
     * Returns the JDOM Element most recently examined by the
     * methods that convert XML to objects.  This may be useful
     * when handling or reporting an exception.
     */
    public Element getLastExaminedElement() {
	return lastExaminedElement;
    }

    /**
     * Converts a JDOM Element to an object.  This is the central
     * recursion point of the translation process; it determines
     * which of the <code><i>type</i>FromElement</code> methods
     * to call.
     *
     * <p>If the postfilter is not null, it is called on the object
     * before it is returned, and the result of that call is
     * returned instead.
     *
     * <p>Note that most of the work, and the calling of the postfiler,
     * are actually done in the 2-argument version of this method and
     * that some recursions may go directly to that method.
     *
     * @see #objectFromElement(Class, Element)
     * @see #setPostfilter(Function1)
     */
    public Object objectFromElement(Element elt) {
	lastExaminedElement = elt;
	String className = elt.getName();
	if (className.equals("null"))	// /\/ irritating special case
	    return null;
	Class c = classSyntax.classForExternalName(className);
	if (c == null)
	    throw new XMLException("Cannot find class " + className);
	else
	    return objectFromElement(c, elt);
    }

    protected Object objectFromElement(Class c, Element elt) {
	lastExaminedElement = elt;
	if (requireNamespace && !isHomeNamespace(elt.getNamespace()))
	    throw new XMLException("Not an I-X element: " + elt);
	Object result;
	ClassDescr cd = classSyntax.getClassDescr(c);
	if (c == String.class)	// String is final
	    result = stringFromElement(c, elt);
	else if (cd.isPrimitive())
	    result = primitiveObjectFromElement(c, elt);
	else if (cd.isList())
	    result = listFromElement(c, elt);
	else if (cd.isSet())
	    result = setFromElement(c, elt);
	else if (cd.isMap())
	    result = mapFromElement(c, elt);
	else if (cd.isStruct())
	    result = structFromElement(c, elt);
	else if (cd.isXML())
	    result = embeddedXMLFromElement(c, elt);
	else
	    throw new ConsistencyException("Unknown object type " + cd);
	if (postfilter != null) result = postfilter.funcall(result);
	return result;
    }

    protected Object primitiveObjectFromElement(Class c, Element elt) {
	return primitiveObjectFromString(c, elt.getText());
    }

    protected Object primitiveObjectFromString(Class c, String s) {
	// Special cases
	Stringer stringer = classSyntax.getStringer(c);
	if (stringer != null)
	    return stringer.fromString(s);
	// General case
	try {
	    Method meth = c.getMethod("valueOf", new Class[]{String.class});
	    return meth.invoke(null, new Object[]{s});
	}
	catch (NoSuchMethodException e) {
	    Debug.noteException(e);
	    String cName = lastExaminedElement.getName();
	    Debug.on();
	    Debug.noteln("class name from element:", cName);
	    Debug.noteln("Java name:", 
			 classSyntax.getClassFinder().javaName(cName));
	    Debug.noteln("class:", classSyntax.classForExternalName(cName));
	    Debug.noteln("class syntax:", classSyntax);
	    Debug.noteln("class finder:", classSyntax.getClassFinder());
	    throw new XMLException
		("Can't convert to " + c, e);
	}
	catch (InvocationTargetException e) {
	    Debug.noteException(e);
	    throw new XMLException
		("Can't convert to " + c, e.getTargetException());
	}
	catch (Exception e) {
	    Debug.noteException(e);
	    throw new XMLException
		("Can't convert to " + c, e);
	}
    }

    protected String stringFromElement(Class c, Element elt) {
	return elt.getText();
    }

    protected LiteralDocument embeddedXMLFromElement(Class c, Element elt) {
	LiteralDocument xml = (LiteralDocument)makeInstance(c);
	requireEltSize(1, elt);
	Element contentElt = (Element)elt.getChildren().get(0);
	// /\/: Have to make a (deep) clone of embedded Element,
	// because we don't want to leave it attached to its parent
	// and because we don't dare modify it by detaching it.
	// Modification is out because the XML we're translated might
	// later be used for something else.
	xml.setRootElement((Element)contentElt.clone());
	return xml;
    }

    protected List listFromElement(Class c, Element elt) {
	return (List)collectionFromElement(c, defaultListClass, elt);
    }

    protected Set setFromElement(Class c, Element elt) {
	return (Set)collectionFromElement(c, defaultSetClass, elt);
    }

    protected Collection collectionFromElement(Class c,
					       Class implDefault,
					       Element elt) {
	// If the element class, c, is an interface, get the name of
	// the implementing class from the implClass attribute
	// or else use the default.  (If the interface is a subclass
	// of TypedList, it can give us a better default.)
	// If the element class isn't an interface, use it directly.
	// Note that this allows us to deal with XML in which
	// the element type is already the implementation class.
	Class implClass = c.isInterface()
	    ? getImplClass(elt, c, implDefault)
	    : c;

	// If the desired class is an LList, it's a bit of a special
	// case, because the actual class depends on whether the
	// list is empty or not, so we can't just call the usual
	// 1-arg constructor.  /\/
	if (LList.class.isAssignableFrom(implClass))
	    return makeLListFromElement(elt);

	// /\/: We used to create the result as an empty collection
	// and then add elements to it, but that wouldn't work
	// if the result class was for a fixed-size collection.
	List contents = LinkedList.class.isAssignableFrom(implClass)
	    ? (LinkedList)makeInstance(implClass)
	    : new LinkedList();
	for (Iterator i = elt.getChildren().iterator(); i.hasNext();) {
	    Element childElt = (Element)i.next();
	    Object childObj = objectFromElement(childElt);
	    // col.add(childObj);
	    contents.add(childObj);
	}

	// If we've already built a collection of the desired class,
	// we are done.
	if (contents.getClass() == implClass)
	    return contents;

	// Otherwise, construct an instance of the correct class
	// and add the contents by calling the 1-arg constructor
	// we assume is standard with Collections.
	try {
	    Constructor cons =
		implClass.getConstructor(new Class[] {Collection.class});
	    return (Collection)cons.newInstance(new Object[] {contents});
	}
	catch (Exception e) {
	    Debug.noteException(e);
	    throw new XMLException
		("Could not create " + implClass + " collection", e);
	}
    }

    protected LList makeLListFromElement(Element elt) {
	// While we're here, we make sublists default to LLists.
	LListCollector col = new LListCollector();
	for (Iterator i = elt.getChildren().iterator(); i.hasNext();) {
	    Element childElt = (Element)i.next();
	    String childClassName = childElt.getName();
	    if (childClassName.equals("null")) {
		// Irritating special case /\/
		col.add(null);
		continue;
	    }
	    Class childClass =
		classSyntax.classForExternalName(childClassName);
	    Object childObj = 
		childClass == List.class
		    && getImplAttributeValue(childElt) == null
		? makeLListFromElement(childElt)
		: objectFromElement(childElt);
	    col.add(childObj);
	}
	return col.contents();
    }

    protected String getImplAttributeValue(Element elt) {
	return elt.getAttributeValue(implAttributeName);
    }

    protected Class getImplClass(Element elt, Class c, Class implDefault) {
	Debug.expect(c.isInterface());
	String iname = getImplAttributeValue(elt);
	Class implClass = null;
	// If there's an impl attribute, the class it specifies must exist,
	// and we'll use it as the implementation class.
	if (iname != null) {
	    implClass = classSyntax.classForExternalName(iname);
	    if (implClass == null)
		throw new XMLException
		    ("Cannot find implementation class " + iname);
	    else
		return implClass;
	}
	// If the element class, c, is a TypedList, it can give us
	// a better default.
	if (TypedList.class.isAssignableFrom(c)) {
	    Class preferred =
		ListOf.findImplementationClass(c, LinkedList.class);
	    if (preferred == null)
		throw new XMLException
		    ("Cannot find a linked implementation for " + c);
	    else
		return preferred;
	}

	// Otherwise, return the default
	return implDefault;
    }

    protected Map mapFromElement(Class c, Element elt) {
	Class implClass = c.isInterface()
	    ? getImplClass(elt, c, defaultMapClass)
	    : c;
	Map map = (Map)makeInstance(implClass);
	String entryEltName = mapEntryName;
	for (Iterator i = elt.getChildren().iterator(); i.hasNext();) {
	    Element entryElt = (Element)i.next();
	    requireEltType(entryEltName, entryElt);
	    requireEltSize(2, entryElt);
	    List entryChildren = entryElt.getChildren();
	    Element keyElt = (Element)entryChildren.get(0);
	    Element valElt = (Element)entryChildren.get(1);
	    // /\/: Can't assume "key" and "value" are ok as external names?
	    requireEltType("key", keyElt);
	    requireEltType("value", valElt);
	    requireEltSize(1, keyElt);
	    requireEltSize(1, valElt);
	    Element keyObjElt = (Element)keyElt.getChildren().get(0);
	    Element valObjElt = (Element)valElt.getChildren().get(0);
	    Object key = objectFromElement(keyObjElt);
	    Object val = objectFromElement(valObjElt);
	    map.put(key, val);
	}
	return map;
    }

    protected void requireEltType(String name, Element elt) {
	if (!elt.getName().equals(name))
	    throw new XMLException
		("Expecting " + name + " element but found " + elt.getName());
    }

    protected void requireEltSize(int size, Element elt) {
	int actual = elt.getChildren().size();
	if (actual != size)
	    throw new XMLException
		("Expecting " + size + " children but found " + actual +
		 " in " + elt);
    }

    protected Object structFromElement(Class c, Element elt) {
	// /\/: Should check for fields specified > once.
	// /\/: Should check namespace of field names.
	Object obj = makeInstance(c);
	ClassDescr cd = classSyntax.getClassDescr(c);
	// Process attributes
	for (Iterator i = elt.getAttributes().iterator(); i.hasNext();) {
	    Attribute fieldAttr = (Attribute)i.next();
	    String fieldName = fieldAttr.getName(); // external name
	    String fieldValue = fieldAttr.getValue();
	    FieldDescr fd = cd.fieldForExternalName(fieldName);
	    if (fd == null)
		throw new XMLException
		    ("No field for attribute " + fieldName);
	    Class fieldType = fd.getType();
	    Class fieldClass = fieldType.isPrimitive()
		? wrapperClass(fieldType)
		: fieldType;
	    Object fieldObj = fieldClass == String.class
		? fieldValue
		: primitiveObjectFromString(fieldClass, fieldValue);
	    // Install the field value.
	    try {
		fd.setValue(obj, fieldObj);
	    }
	    catch (Exception e) {
		Debug.noteException(e);
		throw new XMLException
		    ("Cannot set field " + fd.getName(), e);
	    }
	}
	// Process subelements
	for (Iterator i = elt.getChildren().iterator(); i.hasNext();) {
	    Element fieldElt = (Element)i.next();
	    lastExaminedElement = fieldElt;
	    requireEltSize(1, fieldElt);
	    String fieldName = fieldElt.getName(); // external name
	    FieldDescr fd = cd.fieldForExternalName(fieldName);
	    if (fd == null) {
		handleUnknownFieldFromElement(obj, fieldName, fieldElt);
		continue;
	    }
	    Class fieldClass = fd.getType();
	    Element valElt = (Element)fieldElt.getChildren().get(0);
	    lastExaminedElement = valElt;
	    String valClassName = valElt.getName();
	    Class valClass;
	    Object fieldVal;
	    // Turn the subelement into a value for the field.
	    if (valClassName.equals("null")) {
		fieldVal = null; 	// irritating special case /\/
	    }
	    else if ((valClass = classSyntax.classForExternalName
		                                (valClassName))
		     == null) {
		throw new XMLException
		    ("Cannot find field value class " + valClassName);
	    }
	    // We now have a fieldClass from the class definition
	    // and a valClass from the field-value XML element, and
	    // we have to decide which to use.
	    else if 
		(valClass.isInterface()
		  && getImplAttributeValue(valElt) == null
		  && (!fieldClass.isInterface()
		      || TypedList.class.isAssignableFrom(fieldClass))
		  && fieldClass != Object.class // not useful info
		) {
		// Looks like we should try for an instance of
		// the field class.
		fieldVal = objectFromElement(fieldClass, valElt);
	    }
	    else {
		// Forget about the fieldClass and just go by the
		// value element.  But use the valClass we already have
		// rather than look it up again.
		fieldVal = objectFromElement(valClass, valElt);
	    }
	    // Install the field value.
	    try {
		fd.setValue(obj, fieldVal);
	    }
	    catch (Exception e) {
		Debug.noteException(e);
		throw new XMLException
		    ("Cannot set field " + fd.getName(), e);
	    }
	}
	return obj;
    }

    void handleUnknownFieldFromElement(Object obj, String fieldName,
				       Element fieldElt) {
	// We don't think there's any such field, but there might
	// still be a "set" methodl.  Because we don't have a field
	// description, we don't know what value class is desired;
	// but the value element might give us enough information.
	Class c = obj.getClass();
	Debug.noteln("Handling unknown field " + Strings.quote(fieldName) +
		     " in class " + classSyntax.externalNameForClass(c));
	// Work out the method name
	ClassFinder finder = classSyntax.getClassFinder();
	String javaFieldName = finder.javaFieldName(fieldName);
	String methodName = "set" + Strings.capitalize(javaFieldName);
	Debug.noteln("Will try method name", methodName);
	// Get the value class.
	Element valElt = (Element)fieldElt.getChildren().get(0);
	lastExaminedElement = valElt;
	String valClassName = valElt.getName();
	Debug.noteln("Value class name", valClassName);
	if (valClassName.equals("null")) {
	    // Can't get a value class from that
	    throw new XMLException("No field for element " + fieldName);
	}
	Class valClass = classSyntax.classForExternalName(valClassName);
	Debug.noteln("Value class", valClass);
	// See if there's a method methodName(valClass).
	Method m = Fn.getMethod(obj, methodName, new Class[]{valClass});
	// If there's no method, and the val class is an interface,
	// try the implementation class, if we can determine what it is.
	// /\/: We really need proper dynamic method lookup here.
	if (m == null && valClass.isInterface()) {
	    Class implClass = getImplClass(valElt, valClass, valClass);
	    if (implClass != valClass)
		m = Fn.getMethod(obj, methodName, new Class[]{implClass});
	}
	// If we can't find a method, give up.
	if (m == null)
	    throw new XMLException
		("No field or set-method for element " + fieldName);
	// Get the field value.
	Object fieldVal = objectFromElement(valClass, valElt);
	// Try calling the method.
	try {
	    m.invoke(obj, new Object[]{fieldVal});
	}
	catch (Exception e) {
	    Debug.noteException(e);
	    throw new XMLException
		("Cannot set field " + fieldName + " by calling " +
		 methodName, e);
	}
    }

    /**
     * Returns the wrapper class for a primitive class, such as
     * <code>Long.class</code> for <code>long.class</code>.
     */
    public static final Class wrapperClass(Class pc) {
	// /\/: Is there really no better way to do this?
	if (pc == byte.class) return Byte.class;
	else if (pc == char.class) return Character.class;
	else if (pc == short.class) return Short.class;
	else if (pc == int.class) return Integer.class;
	else if (pc == long.class) return Long.class;
	else if (pc == float.class) return Float.class;
	else if (pc == double.class) return Double.class;
	else if (pc == boolean.class) return Boolean.class;
	else if (pc == void.class) return Void.class;
	else
	    throw new IllegalArgumentException
		(pc + " is not a primitive class");
    }

    /**
     * Makes an instance of a class using the 0-argument constructor.
     *
     * @throws XMLException if the attempt fails.
     */
    protected Object makeInstance(Class c) {
	try {
	    return c.newInstance();
	}
	catch (Exception e) {
	    Debug.noteException(e);
	    throw new XMLException
		("Cannot make 0-arg instance of " + c, e);
	}
    }
}

// Issues:
// * For circularity and sharing, give objects that are shared an
//   attribute id="someid" and refer to it using an attribute
//   ref="someref" and then declare these attribute names as
//   type ID and IDREF respectively.  An element is allowed only
//   one attribute of type ID.
