/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Jul 19 15:53:41 2006 by Jeff Dalton
 * Copyright: (c) 2002 - 2004, AIAI, University of Edinburgh
 */

package ix.util.xml;

import javax.swing.*;

import java.awt.Component;

import java.io.*;
import java.net.*;
import java.util.*;

// Imports for using JDOM
import org.jdom.Document;

import ix.iface.domain.DomainParser;

import ix.icore.domain.Domain;
import ix.icore.plan.Plan;

import ix.util.*;

/**
 * A utility for obtaining XML documents, or objects represented
 * as XML, from files or URLs, conducting dialogs as required.
 * Some non-XML syntaxes for objects can also be handled and
 * the objects can be converted to XML documents if desired.
 *
 * @see XMLTreeEditFrame
 */
public class XMLLoader {

    protected File libDir = DomainParser.getLibraryDirectory();
    protected XMLTranslator xmlt = XML.defaultTranslator();
    protected JFileChooser fileChooser = null;
    protected Component frame;		// for dialogs
    protected Class desiredResultClass;

    public XMLLoader(Component frame) {
	this(frame, Object.class);
    }

    public XMLLoader(Component frame, Class desiredResultClass) {
	this.frame = frame;
	this.desiredResultClass = desiredResultClass;
    }

    /**
     * Returns the XMLTranslator that is used by the loadObject method
     * to convert a JDOM document to an object.
     */
    public XMLTranslator getXMLTranslator() {
	return xmlt;
    }

    /**
     * Sets the XMLTranslator that is used by the loadObject method
     * to convert a JDOM document to an object.
     */
    public void setXMLTranslator(XMLTranslator xmlt) {
	this.xmlt = xmlt;
    }

    /**
     * Reads an object from a file selected by the user, conducting
     * all necessary dialogs along the way.  It repeatedly asks the user
     * to select a file until either the Document has been successfully
     * read or the user decides to cancel the operation.  The user is
     * informed of any exceptions thrown while attempting to read, and
     * "success" means that no exceptions were thrown.
     *
     * @return an Object, or else null if the user cancels.
     *
     * @see #readObjectFrom(File)
     */
    public Object loadObject() {

	// Loop until either the data has successfully been read
	// from a file or else the user has decided to "Cancel"
	// rather than try a(nother) file.
	while (true) {

	    // Ask user to select a file.
	    File file = chooseFileToRead();
	    if (file == null)
		return null;		// user cancels

	    try {
		Object obj = readObjectFrom(file);
		return obj;		// success!
	    }
	    catch (Throwable t) {
		Debug.displayException(t);
		// Now 'round the loop again
	    }

	}

    }

    /**
     * Converts the contents of the specified file to an object.
     */
    protected Object readObjectFrom(File file) throws IOException {
	Debug.noteln("XMLLoader reading object from", file);
	return XML.readObject(desiredResultClass, file.toURL());
    }

    /**
     * Reads a JDOM Document from a file selected by the user, conducting
     * all necessary dialogs along the way.  It repeatedly asks the user
     * to select a file until either the Document has been successfully
     * read or the user decides to cancel the operation.  The user is
     * informed of any exceptions thrown while attempting to read, and
     * "success" means that no exceptions were thrown.
     *
     * @return a JDOM Document, or else null if the user cancels.
     *
     * @see #readDocumentFrom(File)
     */
    public Document loadDocument() {

	// Loop until either the data has successfully been read
	// from a file or else the user has decided to "Cancel"
	// rather than try a(nother) file.
	while (true) {

	    // Ask user to select a file.
	    File file = chooseFileToRead();
	    if (file == null)
		return null;		// user cancels

	    try {
		Document doc = readDocumentFrom(file);
		return doc;		// success!
	    }
	    catch (Throwable t) {
		Debug.displayException(t);
		// Now 'round the loop again
	    }

	}

    }

    /**
     * Converts the contents of the specified file to a JDOM Document.
     * In addition to XML files, it is possible to read other syntaxes.
     * However, a non-XML file is first read as an object by
     * {@link #readObjectFrom(File)} and then converted to a Document.
     */
    protected Document readDocumentFrom(File file) throws IOException {
	Debug.noteln("XMLLoader reading document from", file);
	FileSyntaxManager fsm = XML.fileSyntaxManager();
	FileSyntax syntax = fsm.getSyntax(file);
	Debug.noteln("Using syntax", syntax);
	if (syntax == fsm.getSyntaxForType("xml")) {
	    // If it's XML syntax, assume we can parse the file ourself.
	    return XML.parseXML(file);
	}
	else {
	    // Read as object, then convert to Document.
	    Object obj = readObjectFrom(file);
	    return xmlt.objectToDocument(obj);
	}
    }

    /**
     * Asks the user to choose a file.  Returns null if the user cancels.
     */
    protected File chooseFileToRead() {
	if (fileChooser == null)
	    fileChooser = makeFileChooser();
	int option = fileChooser.showOpenDialog(frame);
	if (option == JFileChooser.APPROVE_OPTION)
	    return fileChooser.getSelectedFile();
	else
	    return null;
    }

    /**
     * Constructs a suitable file-chooser for this XMLLoader's
     * desired result class.
     */
    public JFileChooser makeFileChooser() {
	Debug.noteln("Library directory", libDir);
	JFileChooser chooser = new JFileChooser(libDir);
	FileSyntaxManager fsm = XML.fileSyntaxManager();
	Set extensions = new TreeSet();
	for (Iterator i = fsm.getAllSyntaxes().iterator(); i.hasNext();) {
	    FileSyntax syntax = (FileSyntax)i.next();
	    if (fsm.canRead(syntax, desiredResultClass)) {
		extensions.addAll(syntax.getFileTypes());
	    chooser.addChoosableFileFilter(new SyntaxFileFilter(syntax));
	    }
	}
	SyntaxFileFilter classFilter =
	    new SyntaxFileFilter(desiredResultClass, extensions);
	chooser.setFileFilter(classFilter);
	return chooser;
    }

}

// Issues:
// * The FileChooser isn't created until it's needed so that XMLLoaders
//   can be used in otherwise GUI-less applications without initializing
//   any Swing/AWT things.  Without this, the event thread seems to get
//   going, and then the app won't exit when it runs off the end of its
//   main(String[]) method.

