/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat Dec  3 06:02:16 2005 by Jeff Dalton
 * Copyright: (c) 2004 - 2005, AIAI, University of Edinburgh
 */

package ix.util.xml;

import javax.swing.JMenu;
import java.util.*;

import org.jdom.Namespace;
import org.jdom.output.XMLOutputter;
import org.jdom.output.Format;

import ix.util.*;
import ix.util.reflect.*;
import ix.util.lisp.*;

/**
 * A class that can be used to configure the XML tools for a particular
 * application or set of applications.  It provides factory methods and
 * other information that affects the behaviour of the XML tools.
 */
public class XMLConfig {

    protected ClassFinder defaultClassFinder;
    protected ClassSyntax defaultClassSyntax;
    protected XMLTranslator defaultXMLTranslator;
    protected FileSyntaxManager defaultFileSyntaxManager;

    protected String homeNameSpaceURI = "http://not_a_real_namespace/";

    protected String SAXDriverClass =
	"org.apache.xerces.parsers.SAXParser";    // For Xerces 1 or 2:
	// "gnu.xml.aelfred2.SAXDriver";
        // "org.apache.crimson.parser.XMLReaderImpl";

    /* * * Constructors * * */

    public XMLConfig() { }

    /* * * Factories * * */

    public ClassFinder makeClassFinder() {
	return new DashSyntaxClassFinder();
    }

    public ClassSyntax makeClassSyntax(ClassFinder finder) {
	return new ClassSyntax(finder);
    }

    public XMLTranslator makeXMLTranslator(ClassSyntax syntax) {
	XMLTranslator xmlt = new XMLTranslator(syntax);
	xmlt.setOmitImplAttributes(true);
	return xmlt;
    }

    public XMLTranslator makeXMLTranslator() {
	return makeXMLTranslator(defaultClassSyntax());
    }

    public FileSyntaxManager makeFileSyntaxManager() {
	return new FileSyntaxManager();
    }

    /* * * Default instances * * */

    // N.B. We can't make the default instances when this XMLConfig
    // is first created, because they may refer to XML.config()
    // in some constructors.  The XMLTranslator definitely does this.

    protected void makeDefaultInstances() {
	defaultClassFinder = makeClassFinder();
	defaultClassSyntax = makeClassSyntax(defaultClassFinder);
	defaultXMLTranslator = makeXMLTranslator(defaultClassSyntax);
	defaultFileSyntaxManager = makeFileSyntaxManager();
    }

    public ClassFinder defaultClassFinder() {
	if (defaultClassFinder == null) makeDefaultInstances();
	return defaultClassFinder;
    }

    public ClassSyntax defaultClassSyntax() {
	if (defaultClassSyntax == null) makeDefaultInstances();
	return defaultClassSyntax;
    }

    public XMLTranslator defaultXMLTranslator() {
	if (defaultXMLTranslator == null) makeDefaultInstances();
	return defaultXMLTranslator;
    }

    public FileSyntaxManager defaultFileSyntaxManager() {
	if (defaultFileSyntaxManager == null) makeDefaultInstances();
	return defaultFileSyntaxManager;
    }

    /* * * ClassFinder imports * * */

    /**
     * Adds an import in the default ClassFinder.  It will therefore
     * affect all XML translators that use that ClassFinder.
     */
    public void addImport(String name) {
	// /\/: Because this affects only the default finder directly,
	// ClassSyntaxes need to at least use a finder that shares the
	// same list of imports.  See the ClassFinder(ClassFinder base)
	// constructor.
	defaultClassFinder().addImport(name);
    }

    /* * * Namespaces * * */

    public Namespace getHomeNamespace() {
	return Namespace.getNamespace(homeNameSpaceURI);
    }

    public void setHomeNamespaceURI(String uri) {
	homeNameSpaceURI = uri;
    }

    /* * * JDOM * * */

    /**
     * Returns the class name of the SAX parser that should be
     * used by the <code>XML.parseXML</code> methods.
     */
    public String SAXDriverClass() {
	return SAXDriverClass;
    }

    public void setSAXDriverClass(String name) {
	SAXDriverClass = name;
    }

    /**
     * Constructs a JDOM XMLOutputter that outputs a JDOM Document
     * in a nicely indented fashion.  For example:
     * <pre>
     *    XMLConfig conf = ...;
     *    XMLOutputter outputter = conf.makePrettyXMLOutputter()
     *    outputter.output(doc, System.out);
     *    System.out.flush();
     * </pre>
     * The outputter has 1 space indent and newlines=true
     */
    public XMLOutputter makePrettyXMLOutputter() {
//  	XMLOutputter outputter = new XMLOutputter(" ", true);
//  	outputter.setLineSeparator(System.getProperty("line.separator"));
//  	// outputter.setTrimText(true); // deprecated in beta-7 & wrong anyway
//  	outputter.setTextNormalize(false);
//  	return outputter;
	Format f = Format.getPrettyFormat();
	f.setIndent(" ");
	f.setLineSeparator(System.getProperty("line.separator")); // /\/ ??
	return new XMLOutputter(f);
    }

    /* * * Tree editor * * */

    public Class[] treeEditorTemplateClassRoots() {
	return new Class[]{};
    }

    // /\/: Might be better to let the whole frame be
    // specified, not just a menu.

    public JMenu makeTreeEditorLookAtMenu(XMLTreeEditFrame frame) {
	return null;
    }

    /* * * XMLSyntax * * */

    public List xmlSyntaxClasses(ClassSyntax classSyntax, Class rootClass) {
	// /\/: Here's where we try to make up for the limitations
	// of the methods used in the ClassSyntax class.
	return Lisp.NIL;
    }

}
