/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Feb 16 00:14:41 2006 by Jeff Dalton
 * Copyright: (c) 2001 - 2006, AIAI, University of Edinburgh
 */

package ix.util.xml;

import java.util.*;

import java.io.IOException;
import java.io.StringReader;
import java.io.File;

import java.net.URL;

// Imports for using JDOM
// import java.io.IOException;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.Attribute;
import org.jdom.Namespace;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.XMLOutputter;

import org.xml.sax.*;

import ix.util.reflect.*;
import ix.util.lisp.*;
import ix.util.*;

/**
 * A class containing static XML utilities.
 *
 * @see XMLConfig
 */

public class XML {

    private XML() { }		// don't allow instantiation

    private static XMLConfig config;
    private static XMLTranslator defaultTranslator;
    static {
	setConfig(new ix.ip2.Ip2XMLConfig());
    }

    /**
     * Returns the configuration currently being used by the XML tools.
     */
    public static XMLConfig config() {
	return config;
    }

    /**
     * Sets the configuration that will be used by the XML tools.
     */
    public static void setConfig(XMLConfig c) {
	config = c;
	defaultTranslator = c.defaultXMLTranslator();
    }

    /**
     * The XMLTranslator used by static method in this class.
     */
    public static XMLTranslator defaultTranslator() {
	return defaultTranslator;
    }

    /**
     * Allows temporary replacement of the default translator.
     */
    public static void withDefaultTranslator(XMLTranslator xmlt,
					     Runnable thunk) {
	XMLTranslator saved = defaultTranslator;
	try {
	    defaultTranslator = xmlt;
	    thunk.run();
	}
	finally {
	    defaultTranslator = saved;
	}
    }

    /**
     * Returns the JDOM Namespace used by the default translator. 
     */
    public static Namespace getHomeNamespace() {
	return defaultTranslator.getHomeNamespace();
    }

    /**
     * Returns the class-name that would be used by the default
     * ClassFinder.
     */
    public static String nameForClass(Class c) {
	return config().defaultClassFinder().nameForClass(c);
    }

    /**
     * Adds an import in the default ClassFinder.  It will therefore
     * affect all XML translators that use that ClassFinder.
     */
    public static void addImport(String name) {
	config().addImport(name);
    }

    /**
     * Returns the FileSyntaxManager used by static methods in this class.
     */
    public static FileSyntaxManager fileSyntaxManager() {
	return config().defaultFileSyntaxManager();
    }


    /*
     *     XML to Objects
     */

    /**
     * Converts a string of XML to an Object using the default translator.
     */
    public static Object objectFromXML(String xml) {
	return defaultTranslator.objectFromXML(xml);
    }

    // /\/: Maybe names should be objectFromXMLString, ...FromXMLFile, etc.
    // Or perhaps just have objectFrom and let the argument class select
    // from there.

    /**
     * Converts a File of XML to an Object using the default translator.
     *
     * @deprecated  As of I-X 3.3, use {@link #readObject(String)}.
     */
    public static Object objectFromFile(File file) {
	return objectFromDocument(parseXML(file));
    }

    /**
     * Reads an object from the specified resource.  /\/ explain ...
     */
    public static Object readObject(String resourceName) {
	return readObject(Object.class, resourceName);
    }

    /**
     * Reads an object from the specified resource.  /\/ explain...
     */
    public static Object readObject(Class desiredClass, String resourceName) {
	return fileSyntaxManager().readObject(desiredClass, resourceName);
    }

    /**
     * Reads an object from the specified resource.  /\/ explain...
     */
    public static Object readObject(Class desiredClass, URL url) {
	return fileSyntaxManager().readObject(desiredClass, url);
    }

    /**
     * Reads a JDOM Document from the specified resource.  /\/ explain ... 
     */
    public static Document readDocument(String resourceName) {
	FileSyntaxManager fsm = fileSyntaxManager();
	FileSyntax syntax = fsm.getSyntax(resourceName);
	Debug.noteln("Using syntax", syntax);
	if (syntax == fsm.getSyntaxForType("xml")) {
	    // If it's XML syntax, assume we can read as a Document.
	    URL url = fsm.toURL(resourceName);
	    if (url == null)
		throw new IllegalArgumentException
		    ("Can't find a resource named " +
		     Strings.quote(resourceName));
	    return parseXML(url);
	}
	else {
	    // Read as object, then convert to Document.
	    Object obj = readObject(resourceName);
	    return defaultTranslator.objectToDocument(obj);
	}
    }

    /**
     * Converts a JDOM Document to an object using the default translator.
     */
    public static Object objectFromDocument(Document doc) {
	return defaultTranslator.objectFromDocument(doc);
    }

    /**
     * Asks the default translator whether the document looks
     * like it can be read as an Object.
     *
     * @see XMLTranslator#looksLikeAnObjectDocument(Document)
     */
    public static boolean looksLikeAnObjectDocument(Document doc) {
	return defaultTranslator.looksLikeAnObjectDocument(doc);
    }

    /**
     * Asks the default translator whether the document looks
     * like it can be read as an Object.
     *
     * @deprecated  As of I-X 3.3, use
     *                {@link #looksLikeAnObjectDocument(Document)}.
     */
    public static boolean looksLikeAnIXDocument(Document doc) {
	return defaultTranslator.looksLikeAnObjectDocument(doc);
    }

    /**
     * Converts a JDOM Element to an object using the default translator.
     */
    public static Object objectFromElement(Element elt) {
	return defaultTranslator.objectFromElement(elt);
    }

    /**
     * The class of the SAX parser used by the <code>parseXML</code>
     * methods.
     */
    private static String SAXDriverClass() {
	return config().SAXDriverClass();
    }

    /**
     * A minor variation on the JDOM SAXBuilder class.
     */
    public static class XSAXBuilder extends SAXBuilder {
//  	String driverClass = null; // superclass saxDriverClass is private
	public XSAXBuilder() {
	    super();
	}
	public XSAXBuilder(boolean validate) {
	    super(validate);
	}
	public XSAXBuilder(String saxDriverClass) {
	    super(saxDriverClass);
//  	    this.driverClass = saxDriverClass;
	} 
	public XSAXBuilder(String saxDriverClass, boolean validate) {
	    super(saxDriverClass, validate);
//  	    this.driverClass = saxDriverClass;
	}
	protected XMLReader createParser() throws JDOMException {
	    String driverClass = getDriverClass();
	    // if (driverClass != null)
	    //     Debug.noteln("Trying to use XML parser class", driverClass);
	    XMLReader parser = super.createParser();
	    // Debug.noteln("Using XML parser", parser);
	    return parser;
	}
    }

    /**
     * Converts a String of XML to a JDOM Document.
     */
    public static Document parseXML(String text) {
	StringReader reader = new StringReader(text);
	// Debug.noteln("Parsing XML string using", SAXDriverClass());
	try {
	    SAXBuilder builder = new XSAXBuilder(SAXDriverClass());
	    Document doc = builder.build(reader);
	    return doc;
	}
	catch (Exception e) {
	    Debug.noteException(e);
	    Debug.noteln("Problematic text:", text);
	    throw new XMLException("Can't parse string because "
				   + Debug.describeException(e));
	}
    }

    /**
     * Converts a File of XML to a JDOM Document.
     */
    public static Document parseXML(File file) {
	Debug.noteln("Parsing XML file " + file + " using",
		     SAXDriverClass());
	try {
	    SAXBuilder builder = new XSAXBuilder(SAXDriverClass());
	    Document doc = builder.build(file);
	    return doc;
	}
	catch (JDOMException e) {
	    // Should already mention the File.
	    Debug.noteException(e);
	    throw new XMLException(e);
	}
	catch (Exception e) {
	    Debug.noteException(e);
	    throw new XMLException("Can't parse " + file + " because "
				   + Debug.describeException(e));
	}
    }

    /**
     * Converts XML from a URL to a JDOM Document.
     */
    public static Document parseXML(URL url) {
	Debug.noteln("Parsing XML URL " + url + " using",
		     SAXDriverClass());
	try {
	    SAXBuilder builder = new XSAXBuilder(SAXDriverClass());
	    Document doc = builder.build(url);
	    return doc;
	}
	catch (JDOMException e) {
	    // Should already mention the URL?  /\/
	    Debug.noteException(e);
	    throw new XMLException(e);
	}
	catch (Exception e) {
	    Debug.noteException(e);
	    throw new XMLException("Can't parse " + url + " because "
				   + Debug.describeException(e));
	}
    }

    /**
     * Returns a URL for a specified resource.
     *
     * @see FileSyntaxManager#toURL(String)
     */
    public static URL toURL(String resourceName) {
	return fileSyntaxManager().toURL(resourceName);
    }


    /*
     *     Objects to XML
     */

    /**
     * Converts an object to a string of XML using the default translator.
     */
    public static String objectToXMLString(Object obj) {
	return defaultTranslator.objectToXMLString(obj);
    }

    /**
     * Converts a JDOM Document to a string of XML using the
     * default translator.
     */
    public static String documentToXMLString(Document doc) {
	return defaultTranslator.documentToXMLString(doc);
    }

    /**
     * Converts an object to a JDOM Document using the default translator.
     */
    public static Document objectToDocument(Object obj) {
	return defaultTranslator.objectToDocument(obj);
    }

    /**
     * Writes an object as XML to the specified file.  /\/ explain ...
     */
    public static void writeObject(Object obj, String filename) {
	fileSyntaxManager().writeObject(obj, filename);
    }

    /**
     * An XML outputter created by makePrettyXMLOutputter().
     */
    public static XMLOutputter prettyXMLOutputter = makePrettyXMLOutputter();

    /**
     * Constructs a JDOM XMLOutputter that outputs a JDOM Document
     * in a nicely indented fashion.
     *
     * @see XMLConfig#makePrettyXMLOutputter()
     */
    public static XMLOutputter makePrettyXMLOutputter() {
	return config().makePrettyXMLOutputter();
    }


    /**
     * Prints a document as text to System.out with blank lines
     * between top-level parts of the root element.
     * Calls {@link #printXMLWithWhitespace(String xml, int indentation)}.
     */
    public static void printXMLWithWhitespace(Document doc, int indentation) {
	String xml = defaultTranslator.documentToXMLString(doc);
	printXMLWithWhitespace(xml, indentation);
    }

    /**
     * Prints a multi-line string of XML with blank lines between
     * top-level parts of the root element.
     */
    public static void printXMLWithWhitespace(String xml, int indentation) {
	// /\/: Assumes that the xml string has 1 space per level indentation
	// and then adds any additional indentation requested.
	if (indentation < 1)
	    throw new IllegalArgumentException("Can't indent " + indentation);
	List lines = Strings.breakIntoLines(xml);
	int oldIndent = 0;
	for (Iterator li = lines.iterator(); li.hasNext();) {
	    String line = (String)li.next();
	    int indent = line.indexOf("<");
	    if (indent < 0) indent = 2;
	    // Blank line if indent goes from 0 or 1 to 1
	    // or (which happens only at the end) from 1 to 0.
	    if (oldIndent <= 1 && indent == 1 || oldIndent == 1 && indent < 1)
		System.out.println("");
	    // Determine extra indentation required.
	    int level = indent;
	    int required = level * indentation;
	    String padding = Strings.repeat(required - indent, " ");
	    System.out.println(padding + line);
	    oldIndent = indent;
	}
    }


    /*
     * JDOM utilities
     */

    /**
     * Sets attributes of a JDOM Element as specified by a String[][]
     * array.  Each row of the array is a {name, value} pair.
     */
    public static void setAttributes(Element elt, String[][] attributes) {
	for (int i = 0; i < attributes.length; i++) {
	    String key = attributes[i][0];
	    String value = attributes[i][1];
	    elt.setAttribute(key, value);
	}
    }


    /*
     * Encode and decode utilities
     */

    /**
     * Encodes ampersands and angle brackets so that they can appear
     * in XML without being mistaken for markup.
     */
    public static String encodeText(String text) {
	return encodeForXML(text, false);
    }

    /**
     * Decodes text containing encoded ampersands and angle brackets.
     * This is the inverse of {@link #encodeText(String)}.
     */
    public static String decodeText(String text) {
	return decodeForXML(text);
    }

    /**
     * Encodes double quotes as well as ampersands and angle brackets
     * to make text safe for use as XML attribute values.
     */
    public static String encodeAttribute(String text) {
	return encodeForXML(text, true);
    }

    /**
     * Decodes text containing encoded double quotes, ampersands,
     * and angle brackets.
     * This is the inverse of {@link #encodeAttribute(String)}.
     */
    public static String decodeAttribute(String text) {
	return decodeForXML(text);
    }

    private static String encodeForXML(String text, boolean isForAttribute) {
	final String special = isForAttribute ? "&<>\"" : "&<>";
	int where = Strings.indexOfAny(special, text);
	if (where < 0)
	    return text;
	int increase = 0;	// how many more chars needed in result
	while (where >= 0) {
	    increase += charEncoding(text.charAt(where)).length() - 1;
	    where = Strings.indexOfAny(special, where + 1, text);
	}
	// Now that we know how many characters are needed, we construct
	// the result in a SimpleStringBuffer.  /\/: Use char[] instead?
	int textLen = text.length();
	SimpleStringBuffer buf = new SimpleStringBuffer(textLen + increase);
	for (int i = 0; i < textLen; i++) {
	    char c = text.charAt(i);
	    if (special.indexOf(c) >= 0)
		buf.append(charEncoding(c));
	    else
		buf.append(c);
	}
	Debug.expect(buf.length() == textLen + increase);
	return buf.toString();
    }

    private static String charEncoding(char c) {
	// Could use an array, I suppose.  /\/
	switch ("&<>\"".indexOf(c)) {
	case 0:		// &
	    return "&amp;";
	case 1:		// <
	    return "&lt;";
	case 2:		// >
	    return "&gt;";
	case 3:		// double quote
	    return "&quot;";
	}
	throw new RuntimeException("No encoding for " + String.valueOf(c));
    }

    private static String decodeForXML(String text) {
	if (text.indexOf('&') < 0)
	    return text;
	int i = 0, textLen = text.length();
	SimpleStringBuffer buf = new SimpleStringBuffer(textLen); //char[]?/\/
	while (i < textLen) {
	    char c = text.charAt(i);
	    if (c != '&') { buf.append(c); i++; }
	    else {
		char clue = text.charAt(i + 1);
		char decode;
		String code;
		if      (clue == 'a') { decode = '&'; code = "&amp;"  ; }
		else if (clue == 'l') { decode = '<'; code = "&lt;"   ; }
		else if (clue == 'g') { decode = '>'; code = "&gt;"   ; }
		else if (clue == 'q') { decode = '"'; code = "&quot;" ; }
		else
		    throw new RuntimeException("Cannot decode " + text);
		Debug.expect(text.charAt(i + code.length() - 1) == ';');
		buf.append(decode);
		i += code.length();
	    }
	}
	return buf.toString();
    }

    /**
     * Test loop that asks the user for a line of text and
     * shows the result of calling {@link #encodeAttribute(String)}
     * on the input then calling {@link #decodeAttribute(String)}
     * on the encoded result.
     */
    public static void main(String[] argv) {
	for (;;) {
	    String text = Util.askLine("Text:");
	    if (text.equals("bye"))
		return;
	    String encoded = encodeAttribute(text);
	    Debug.noteln("Encode:", encoded);
	    Debug.noteln("Decode:", decodeAttribute(encoded));
	}
    }

}
