/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat Jan 17 03:02:17 2004 by Jeff Dalton
 */

package ix.util.xml;

import java.io.*;
import java.util.*;

// JDOM imports
import org.jdom.*;

import ix.util.*;
import ix.util.reflect.*;
import ix.util.lisp.*;

/**
 * Describes the (XML) syntax of I-X data objects as a Relax NG schema.
 */
public class RelaxNGSyntax extends XMLSyntax {

    // /\/: Dubious state variable
    Schema theSchema = null;

    public static final Namespace relaxNGNamespace
	= Namespace.getNamespace("", "http://relaxng.org/ns/structure/1.0");

    public RelaxNGSyntax() {
	super();
    }

    public RelaxNGSyntax(XMLTranslator xmlt) {
	super(xmlt);
    }

    public Schema makeSchema(Class rootClass) {
	return makeSchema
	    (rootClass,
	     XML.config().xmlSyntaxClasses(classSyntax, rootClass));
    }

    Schema makeSchema(Class rootClass, List relevantClasses) {
	Schema schema = new Schema();

	theSchema = schema;
	inheritance = new InheritanceTree(relevantClasses);

	// Add start element
	schema.addContent(makeRngElement("start")
			  .addContent(makeRef(rootClass)));

	// Add a pattern for arbitrary objects allowed in plans.
	schema.addContent(objectPatternDefinition(relevantClasses));

	// Lists of arbitrary objects
	schema.addContent(listPatternDefinition());

	// Maps
	schema.addContent(mapPatternDefinition());

	// Add information from each class.
	for (Iterator i = relevantClasses.iterator(); i.hasNext();) {
	    Class c = (Class)i.next();
	    addClassSyntax(schema, c);
	}

	return schema;
    }

    /**
     * An Relax NG schema.
     */
    class Schema extends Element {

	/** Create a schema. */
	public Schema() {
	    super("grammar", relaxNGNamespace);
	    setAttribute("ns", xmlt.getHomeNamespace().getURI());
	    setAttribute("datatypeLibrary",
			 "http://www.w3.org/2001/XMLSchema-datatypes");
	}

	void addComment(String text) {
	    addContent(new Comment(text));
	}

    }

    Element objectPatternDefinition(List relevantClasses) {
	Debug.expect(!relevantClasses.contains(Object.class));
	List objectClasses = new LinkedList(relevantClasses);
	objectClasses.add(0, List.class);
	objectClasses.add(1, Map.class);
	return makeDef(Object.class)
	    .addContent(makeRngElement("choice")
			.setContent(classRefs(objectClasses)));
    }

    Element listPatternDefinition() {
	return makeDef(List.class)
	    .addContent(listofPattern(Object.class));
    }

    Element mapPatternDefinition() {
	return makeDef(Map.class)
	    .addContent(makeElementElement("map")
			.addContent(makeRngElement("zeroOrMore")
				    .addContent(mapEntryPattern())));
    }

    Element mapEntryPattern() {
	return makeElementElement("map-entry")
	    .addContent(makeElementElement("key")
			.addContent(makeRef(Object.class)))
	    .addContent(makeElementElement("value")
			.addContent(makeRef(Object.class)));
    }

    void addClassSyntax(Schema schema, Class c) {
	ClassDescr cd = getClassDescr(c);
	schema.addComment(cd.getExternalName());
	if (cd.isStruct())
	    addStructSyntax(schema, c);
	else if (cd.isEnumeration())
	    addEnumerationSyntax(schema, c);
	else if (cd.isPrimitive())
	    addPrimitiveSyntax(schema, c);
	else if (cd.isXML())
	    addLiteralDocumentSyntax(schema, c);
	else
	    schema.addContent(new Comment("No syntax for " + c));
    }

    String valueName(Class c) {
	return getElementName(c) + "-value";
    }

    String patternName(Class c) {
	// return getElementName(c) + "-pattern";
	return getElementName(c);
    }

    /*
     * Structs
     */

    void addStructSyntax(Schema schema, Class c) {
	ClassDescr cd = getClassDescr(c);
	if (cd.isAbstract()) {
	    schema.addContent(inheritancePatternDefinition(c));
	}
	else {
	    schema.addContent(structPatternDefinition(c));
	}
    }

    Element inheritancePatternDefinition(Class c) {
	List subclasses = inheritance.getSubclasses(c);
	if (subclasses == null || subclasses.isEmpty())
	    throw new IllegalArgumentException
		("Abstract class without subclasses: " + c.getName());
	return makeDef(c)
	    .addContent(makeRngElement("choice")
			.setContent(classRefs(subclasses)));
    }

    List classRefs(List classes) {
	List refs = new LinkedList();
	for (Iterator i = classes.iterator(); i.hasNext();) {
	    Class c = (Class)i.next();
	    refs.add(makeRef(c));
	}
	return refs;
    }

    Element structPatternDefinition(Class c) {
	return makeDef(c)
	    .addContent(makeElementElement(getElementName(c))
			.setContent(structContent(c)));
    }

    List structContent(Class c) {
	ClassDescr cd = getClassDescr(c);
	List fields = cd.getFieldDescrs();
	List attrFields = attributeFields(fields);
	List eltFields = elementFields(fields);
	Debug.expect(fields.size() == attrFields.size() + eltFields.size());

	List attrPatterns = new LinkedList();
	for (Iterator i = attrFields.iterator(); i.hasNext();) {
	    FieldDescr fd = (FieldDescr)i.next();
	    attrPatterns.add(optionalPattern(attributeFieldPattern(fd)));
	}

	List eltPatterns = new LinkedList();
	for (Iterator i = eltFields.iterator(); i.hasNext();) {
	    FieldDescr fd = (FieldDescr)i.next();
	    eltPatterns.add(optionalPattern(elementFieldPattern(fd)));
	}

        List content = new LinkedList();
	if (!attrPatterns.isEmpty())
	    content.addAll(attrPatterns);
	if (!eltPatterns.isEmpty())
	    content.add(makeRngElement("interleave")
			.setContent(eltPatterns));
	return content;
    }

    Element optionalPattern(Element pattern) {
	return makeRngElement("optional").addContent(pattern);
    }

    Element attributeFieldPattern(FieldDescr fd) {
	return makeAttributeElement(getElementName(fd))
	    .addContent(makeRef(valueName(fd.getType())));
    }

    Element elementFieldPattern(FieldDescr fd) {
	return makeElementElement(getElementName(fd))
	    .addContent(fieldValuePattern(fd));
    }

    Element fieldValuePattern(FieldDescr fd) {
	Class valueClass = fd.getType();
	ClassDescr valueDescr = fd.getTypeDescr();
	if (valueDescr.isList()) {
	    ClassDescr eltType = valueDescr.getEltType();
	    if (eltType == null)
		return makeRef(List.class);
	    else
		return listofPattern(eltType.theClass);
	}
	else if (valueDescr.isSet()) {
	    Debug.expect(false, "Can't handle Set values");
	    return null;
	}
	else if (valueDescr.isMap()) {
	    return makeRef(Map.class);
	}
	else {
	    return makeRef(valueClass);
	}
    }

    Element listofPattern(Class eltClass) {
	return makeElementElement("list")
	    .addContent(makeRngElement("zeroOrMore")
			.addContent(makeRef(eltClass)));
    }

    /*
     * Enumerations
     */

    void addEnumerationSyntax(Schema schema, Class c) {
	schema.addContent(elementRefDefinition(c, valueName(c)));
	schema.addContent(enumValueDefinition(c));
    }

    Element elementRefDefinition(Class c, String refName) {
	return makeDef(c)
	    .addContent(makeElementElement(getElementName(c))
			.addContent(makeRef(refName)));
    }

    Element enumValueDefinition(Class c) {
	return makeDef(valueName(c))
	    .addContent(makeRngElement("choice")
			.setContent(enumValueElements(c)));
    }

    List enumValueElements(Class c) {
	List elts = new LinkedList();
	for (Iterator i = getEnumerationValues(c).iterator(); i.hasNext();) {
	    elts.add(makeRngElement("value")
		     .setText(i.next().toString()));
	}
	return elts;
    }

    /*
     * Primitive types
     */

    void addPrimitiveSyntax(Schema schema, Class c) {
	schema.addContent(elementRefDefinition(c, valueName(c)));
	if (c == String.class)
	    schema.addContent(textValueDefinition(c));
	else
	    schema.addContent(dataValueDefinition(c));
    }

    Element textValueDefinition(Class c) {
	return makeDef(valueName(c))
	    .addContent(makeRngElement("text"));
    }

    Element dataValueDefinition(Class c) {
	String type = XMLSchemaSyntax.getSimpleType(c);
	return makeDef(valueName(c))
	    .addContent(makeRngElement("data")
			.setAttribute("type", type));
    }

    /*
     * Literal Documents (embedded XML)
     */

    void addLiteralDocumentSyntax(Schema schema, Class c) {
	schema.addContent(literalDocumentPatternDefinition());
	schema.addContent(anyXMLPatternDefinition());
    }

    Element literalDocumentPatternDefinition() {
	String ldocEltName = getElementName(LiteralDocument.class);
	return makeDef(LiteralDocument.class)
	    .addContent(makeElementElement(ldocEltName)
			.addContent(makeRef("anyXML")));
    }

    Element anyXMLPatternDefinition() {
	return makeDef("anyXML").addContent(anyXMLPattern());
    }

    Element anyXMLPattern() {
	// Follows an example from the Relax NG Tutorial.
	return makeRngElement("element")
	    .addContent(makeRngElement("anyName"))
	    .addContent
	        (makeRngElement("zeroOrMore")
		 .addContent
		     (makeRngElement("choice")
		      .addContent(makeRngElement("attribute")
				  .addContent(makeRngElement("anyName")))
		      .addContent(makeRngElement("text"))
		      .addContent(makeRef("anyXML"))));
    }

    /*
     * Element utilities.
     */

    Element makeElement(String name, Namespace namespace,
			       String[][] attributes) {
	Element elt = new Element(name, namespace);
	XML.setAttributes(elt, attributes);
	return elt;
    }

    Element makeRngElement(String name, String[][] attributes) {
	return makeElement(name, relaxNGNamespace, attributes);
    }

    Element makeRngElement(String name) {
	return new Element(name, relaxNGNamespace);
    }

    Element makeRef(String toName) {
	return makeRngElement("ref").setAttribute("name", toName);
    }

    Element makeRef(Class c) {
	return makeRef(patternName(c));
    }

    Element makeDef(String name) {
	return makeRngElement("define").setAttribute("name", name);
    }

    Element makeDef(Class c) {
	return makeDef(patternName(c));
    }

    Element makeElementElement(String name) {
	return makeRngElement("element").setAttribute("name", name);
    }

    Element makeAttributeElement(String name) {
	return makeRngElement("attribute").setAttribute("name", name);
    }

    /**
     * Outputs a schema for I-X plans or for the class specified
     * by the "root" parameter.  
     *
     * @see ix.util.Parameters#getParameter(String)
     */
    public static void main(String[] argv) {
	Debug.off();
	Parameters.processCommandLineArguments(argv);

	RelaxNGSyntax syntax = new RelaxNGSyntax();
	Class root = syntax.classSyntax.
	    classForExternalName(Parameters.getParameter("root", "plan"));
 	Schema schema = syntax.makeSchema(root);
 	Document doc = new Document(schema);
 	XML.printXMLWithWhitespace(doc, 2);
    }

}

// Issues:
// * Deliberately different from XMLSchemaSyntax in certain ways,
//   for experience with both styles:
//   - noun-phrase method names such as optionalPattern instead of
//     verb-phrase names such as makeOptionalPattern.
//   - listof(Class) in-line rather than by ref to separate def.
// * Is it possible to use attribute-sensitive content models
//   to handle the known constraint types?
