/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sun Sep  3 23:49:02 2006 by Jeff Dalton
 * Copyright: (c) 2003, 2006, AIAI, University of Edinburgh
 */

package ix.util.reflect;

import java.util.*;

import ix.util.*;

/**
 * A translator between names and classes that uses lower-case
 * names and separates words within names with dashes.
 */
public class DashSyntaxClassFinder extends ClassFinder {

    public DashSyntaxClassFinder() {
	super();
    }

    public DashSyntaxClassFinder(ClassFinder base) {
	super(base);
    }

    /**
     * Returns the Java name that corresponds to an dash-syntax name.
     */
    public String javaName(String externalName) {
        externalName = Strings.replace("__", "$", externalName);
	return Strings.dashNameToFullJavaName(externalName);
    }

    /**
     * Returns the Java name that corresponds to an external field name.
     */
    public String javaFieldName(String externalName) {
	return Strings.uncapitalize(javaName(externalName));
    }

    /**
     * Returns the dash-syntax name that corresponds to a Java name.
     */
    public String externalName(String javaName) {
	return Strings.replace("$", "__",
                               Strings.fullJavaNameToDashName(javaName));
    }

    protected void cacheBothWays(String name, Class c) {
	// /\/: We want the name "double" to map to Double.class
	// which means we have to avoid caching "double" --> Double.TYPE
	// if nameForClass happens to be called on the primitive
	// class double.
	if (c.isPrimitive())
	    classToNameCache.put(c, name); // but not nameToClassCache
	else
	    super.cacheBothWays(name, c);
    }

    /**
     * Simple main program for testing.  This method calls
     * {@link #do_main(ClassFinder)} with an instance of
     * this class as the finder.
     */
    public static void main(String[] argv) {
	do_main(new DashSyntaxClassFinder());
    }

}
