/* File: MatchCase.java
 * Contains: A class for pattern-matching
 * Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Created: January 1998
 * Updated: Mon Feb 17 05:19:37 2003 by Jeff Dalton
 * Copyright: (c) 1998, AIAI, University of Edinburgh
 */

package ix.util.match;

import java.util.*;

/**
 * The root class for entries in MatchTables. <p>
 *
 * If you think of a MatchTable as analogous to a case statement, 
 * a MatchCase is analogous to a single case clause, with the
 * MatchCase's ifSelected method containing the code that is executed
 * in the body of that clause.<p>
 *
 * A MatchCase normally contains a pattern that can be examined by
 * the MatchTable that contains the MatchCase.  The MatchCase's
 * tryMatch method normally implements a match against that pattern.<p>
 * 
 * If the MatchCase is used as a MatchTable default, the pattern
 * and the tryMatch method are ignored.
 *
 * @see MatchTable
 * @see MatchEnv
 */

public abstract class MatchCase {

    /**
     * An object to match against.
     */
    protected Object pattern;

    /**
     * getPattern returns this MatchCase's pattern.
     */
    public Object getPattern() {
	return pattern;
    }

    /**
     * tryMatch attempts to match against the pattern, returning
     * null if the match fails and some non-null object (such as a 
     * MatchEnv) if the match succeeds.
     */
    public abstract Object tryMatch(Object data);

    /** 
     * ifSelected is called by the MatchTable when tryMatch returns
     * a non-null result.  The object that was matched, and the non-null
     * match result, are passed as arguments to ifSelected.  The method
     * provided here simply returns the same match result.
     */
    public Object ifSelected(Object data, Object matchResult) {
	return matchResult;
    }

}
