/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue May 30 00:37:05 2006 by Jeff Dalton
 * Copyright: (c) 2005, AIAI, University of Edinburgh
 */

package ix.util.lisp;

import java.net.URL;
import java.io.*;
import java.util.*;

import ix.util.*;
import ix.util.xml.XML;

/**
 * An interpreter for a simple dialect of Lisp.
 */
public class LispInterpreter extends Interpreter {

    public static final Symbol
	QUOTE = Symbol.intern("quote");

    public static final Symbol
	LAMBDA = Symbol.intern("lambda"),
	PROGN = Symbol.intern("progn"),
	COND = Symbol.intern("cond"),
	ELLIPSIS = Symbol.intern("...");

    protected Map parserTable = new HashMap();

    protected LispReader reader = new LispProgramReader(System.in);

    public LispInterpreter() {
	DEFAULT = Lisp.NIL;
	installDefinitions();
    }

    public static void main(String[] argv) {

	Parameters.processCommandLineArguments(argv);

	LispInterpreter lin = new LispInterpreter();
	Symbol BYE = Symbol.intern("bye");

	lin.loadLisp(Parameters.getList("load-lisp"));

	for (;;) {
	    System.out.print("Lisp: ");
	    try {
		Object in = lin.reader.readObject();
		if (in == BYE)
		    return;
		System.out.println(lin.topLevelEval(in));
	    }
	    catch (Throwable t) {
		Debug.noteException(t);
	    }
	}
    }

    public void loadLisp(List resourcenNames) {
	for (Iterator i = resourcenNames.iterator(); i.hasNext();) {
	    loadLisp((String)i.next());
	}
    }

    public void loadLisp(String resourceName) {
	URL url = XML.toURL(resourceName);
	if (url == null)
	    throw new IllegalArgumentException
		("Can't find a resource named " + Strings.quote(resourceName));
	try {
	    BufferedReader in = new BufferedReader(Util.openURLReader(url));
	    LispReader r = new LispProgramReader(in);
	    Object form;
	    while ((form = r.readObject()) != Lisp.EOF) {
		topLevelEval(form);
	    }
	}
	catch (Exception e) {
	    throw new RethrownException
		(e, "Problem with loading " + Strings.quote(resourceName) +
		    ": " + Debug.describeException(e));
	}
    }

    public boolean isTrue(Object obj) {
	return !(obj == Lisp.NIL || obj.equals(Boolean.FALSE)); // ??? /\/
    }

    public Object topLevelEval(Object form) {
	return super.topLevelEval(parseForm(form));
    }

    public Expr parseForm(Object form) {
	if (form instanceof Keyword)
	    return new Literal(form);
	else if (form instanceof Symbol)
	    return new VarRef((Symbol)form);
	else if (form instanceof LList && form != Lisp.NIL) {
	    LList lis = (LList)form;
	    Object operator = lis.get(0);
	    Parser p = (Parser)parserTable.get(operator);
	    Expr result = p != null
		? p.parse(lis)
		: new Call(parseForm(operator), parseList(lis.cdr()));
	    result.setDescription(form);
	    return result;
	}
	else
	    return new Literal(form);
    }

    public List parseList(List forms) {
	List result = new LinkedList();
	for (Iterator i = forms.iterator(); i.hasNext();) {
	    result.add(parseForm(i.next()));
	}
	return result;
    }

    public static interface Parser {
	public Expr parse(LList form);
    }

    public abstract class Syntax implements Parser {

	protected Symbol name;

	public Syntax(String name) {
	    this.name = Symbol.intern(name);
	}

	protected void checkLength(int required, LList form) {
	    int len = form.length();
	    if (len > required)
		throw new SyntaxError("Too many parameters in", form);
	    else if (len < required)
		throw new SyntaxError("Too few parameters in", form);
	}

	protected Object mustBe(Class c, Object obj) {
	    if (c.isInstance(obj))
		return obj;
	    else
		throw new SyntaxError
		    (obj + " is not " + Strings.indefinite(nameForClass(c)));
	}

	protected LList mustBeLListOf(Class c, Object obj) {
	    LList ll = (LList)mustBe(LList.class, obj);
	    for (Iterator i = ll.iterator(); i.hasNext();) {
		if (!c.isInstance(i.next()))
		    throw new SyntaxError
			(obj + " is not a list of " + nameForClass(c) + "s.");
	    }
	    return ll;
	}

	protected Expr parseBody(LList body) {
	    if (body.cdr().isEmpty())
		return parseForm(body.get(0));
	    else
		return describe(new Sequence(parseList(body)),
				new Cons(PROGN, body));
	}

	protected Expr describe(Expr e, Object description) {
	    e.setDescription(description);
	    return e;
	}

    }

    public static class SyntaxError extends RuntimeException {
	public SyntaxError(String message) {
	    super(message);
	}
	public SyntaxError(String message, Object culprit) {
	    super(message + " " + culprit);
	}
    }

   /* * * Definitions * * */

    protected void installDefinitions() {
	defineSyntax();
	defineBuiltins();
	defineInitialValues();
    }

    public void define(Syntax syntax) {
	parserTable.put(syntax.name, syntax);
    }

    public void define(JFunction builtin) {
	globalEnv.assign(builtin.name, builtin);
    }

    public void define(String varName, Object val) {
	globalEnv.assign(Symbol.intern(varName), val);
    }

    /* * * Syntax * * */

    protected void defineSyntax() {

	define(new Syntax("quote") {
	    public Expr parse(LList form) {
		checkLength(2, form);
		return new Literal(form.get(1));
	    }
	});

	define(new Syntax("if") {
	    public Expr parse(LList form) {
		checkLength(4, form);
		return new If(parseForm(form.get(1)),
			      parseForm(form.get(2)),
			      parseForm(form.get(3)));
	    }
	});

	define(new Syntax("cond") {
	    public Expr parse(LList form) {
		return parseClauses(form.cdr());
	    }
	    Expr parseClauses(LList clauses) {
		if (clauses == Lisp.NIL)
		    return new Literal(DEFAULT);
		else if (clauses.car() instanceof LList) {
		    LList clause1 = (LList)clauses.car();
		    Expr test = parseForm(clause1.car());
		    Expr rest = parseClauses(clauses.cdr());
		    LList descr = new Cons(COND, new Cons(ELLIPSIS, clauses));
		    if (clause1.cdr() == Lisp.NIL) {
			// (cond ... (test) ...)
			return describe(new Or(Lisp.list(test, rest)), descr);
		    }
		    else {
			// (cond ... (test form...) ...)
			Expr body = parseBody(clause1.cdr());
			return describe(new If(test, body, rest), descr);
		    }
		}
		else {
		    throw new SyntaxError("Invalid cond clause",
					  clauses.car());
		}
	    }
	});

	define(new Syntax("and") {
	    public Expr parse(LList form) {
		return new And(parseList(form.cdr()));
	    }
	});

	define(new Syntax("or") {
	    public Expr parse(LList form) {
		return new Or(parseList(form.cdr()));
	    }
	});

	define(new Syntax("setq") {
	    public Expr parse(LList form) {
		checkLength(3, form);
		return
		    new Assignment((Symbol)mustBe(Symbol.class, form.get(1)),
				   parseForm(form.get(2)));
	    }
	});

	define(new Syntax("define") { // same as setq
	    public Expr parse(LList form) {
		checkLength(3, form);
		return
		    new Assignment((Symbol)mustBe(Symbol.class, form.get(1)),
				   parseForm(form.get(2)));
	    }
	});

	define(new Syntax("defun") {
	    public Expr parse(LList form) {
		Symbol name = (Symbol)mustBe(Symbol.class, form.get(1));
		LList vars = (LList)mustBeLListOf(Symbol.class, form.get(2));
		LList body = form.drop(3);
		Expr lambda = describe(new Lambda(vars, parseBody(body)),
				       new Cons(LAMBDA, form.cdr()));
		return new Sequence
		         (Lisp.list(new Assignment(name, lambda),
				    new Literal(name)));
	    }
	});

	define(new Syntax("progn") {
	    public Expr parse(LList form) {
		return new Sequence(parseList(form.cdr()));
	    }
	});

	define(new Syntax("let") {
	    public Expr parse(LList form) {
		LList binds = (LList)mustBe(LList.class, form.get(1));
		LList body = form.drop(2);
		List vars = new LinkedList();
		List valForms = new LinkedList();
		for (Iterator i = binds.iterator(); i.hasNext();) {
		    LList pair = (LList)mustBe(LList.class, i.next());
		    Symbol var = (Symbol)mustBe(Symbol.class, pair.get(0));
		    Object valForm = pair.get(1);
		    vars.add(var);
		    valForms.add(valForm);
		}
		return new Let(vars, parseList(valForms), parseBody(body));
	    }
	});

	define(new Syntax("let*") {
	    public Expr parse(LList form) {
		LList binds = (LList)mustBe(LList.class, form.get(1));
		LList body = form.drop(2);
		return parseLetStar(binds, body);
	    }
	    Expr parseLetStar(LList binds, LList body) {
		if (binds.isEmpty())
		    return parseBody(body);
		else {
		    LList pair = (LList)mustBe(LList.class, binds.car());
		    Symbol var = (Symbol)mustBe(Symbol.class, pair.get(0));
		    Object valExpr = parseForm(pair.get(1));
		    return new Let(Lisp.list(var),
				   Lisp.list(valExpr),
				   parseLetStar(binds.cdr(), body));
		}
	    }
	});

	define(new Syntax("lambda") {
	    public Expr parse(LList form) {
		LList vars = (LList)mustBeLListOf(Symbol.class, form.get(1));
		LList body = form.drop(2);
		return new Lambda(vars, parseBody(body));
	    }
	});

	define(new Syntax("while") {
	    public Expr parse(LList form) {
		return new While(parseForm(form.get(1)),
				 parseBody(form.drop(2)));
	    }
	});

    }

    /* * * Built-in functions * * */

    protected void defineBuiltins () {

	define(new JFunction("identity", 1) {
	    public Object applyTo(Object[] args) {
		return args[0];
	    }
	});

	/* Apply */

	define(new JFunction("apply", Function.ANY_ARITY) {
	    public Object applyTo(Object[] args) {
		// We need at least two arguments.
		int len = args.length;
		if (len < 2)
		    throw new Error
			("Apply wasn't given at least two arguments.");
		// If the 0th argument is a Symbol, treat it as
		// a global function name.
		if (args[0] instanceof Symbol) {
		    Object fval = globalEnv.lookup((Symbol)args[0]);
		    if (fval instanceof Function)
			args[0] = fval;
		    else
			throw new ClassCastException
			    (args[0] + " is not defined as a function.");
		}
		// Get the function to apply.
		Function f = (Function)mustBe(Function.class, args[0]);
		// Construct the array of arguments to give to the function.
		// The last argument to apply is a list that is broken
		// into separate arguments to the function being called.
		List list_arg = (List)mustBe(List.class, args[len-1]);
		int n_args = len-2 + list_arg.size();
		Object[] f_args = new Object[n_args];
		System.arraycopy(args, 1, f_args, 0, len-2);
		int i = len-2;
		for (Iterator li = list_arg.iterator(); li.hasNext();) {
		    f_args[i++] = li.next();
		}
		Debug.expect(i == n_args);
		// Apply the function to its arguments.
		// Debug.noteln("Applying", Arrays.asList(f_args));
		f.checkArity(f_args.length);
		return f.applyTo(f_args);
	    }
	});

	/* Error signalling */

	define(new JFunction("error", Function.ANY_ARITY) {
	    public Object applyTo(Object[] args) {
		if (args.length < 1)
		    throw new Error("Error for unknown reason.");
		else {
		    StringBuffer buf = new StringBuffer(args.length * 70);
		    for (int i = 0; i < args.length; i++) {
			buf.append(args[i].toString());
		    }
		    throw new Error(buf.toString());
		}
	    }
	});

	/* Trace */

//  	define(new JFunction("trace", 1) {
//  	    public Object applyTo(Object[] traceArgs) {
//  		final Symbol fn = (Symbol)mustBe(Symbol.class, traceArgs[0]);
		
//  		return new JFunction("tracer", ANY_ARITY) {
//  		    public Object applyTo(Object[] args) {
//  			Debug.note("Calling " + fn + ":");
//  			for (int i = 0; i < args.length; i++) {
//  			    Debug.note(" " + args[i]);
//  			}
//  			Debug.noteln("");
//  			Object result = topLevelApply(fn, args);
//  			Debug.noteln(fn + " returned " + result);
//  			return result;
//  		    }
//  		};
//  	    }
//  	});

	/* Iteration */

	define(new JFunction("for-each", 2) {
	    public Object applyTo(Object[] args) {
		Function f = (Function)mustBe(Function.class, args[0]);
		Collection c = (Collection)mustBe(Collection.class, args[1]);
		f.checkArity(1);
		Object[] f_args = new Object[1];
		for (Iterator i = c.iterator(); i.hasNext();) {
		    f_args[0] = i.next();
		    f.applyTo(f_args);
		}
		return c;
	    }
	});

	define(new JFunction("map", 2) {
	    public Object applyTo(Object[] args) {
		Function f = (Function)mustBe(Function.class, args[0]);
		Collection c = (Collection)mustBe(Collection.class, args[1]);
		f.checkArity(1);
		Object[] f_args = new Object[1];
		LListCollector result = new LListCollector();
		for (Iterator i = c.iterator(); i.hasNext();) {
		    f_args[0] = i.next();
		    result.add(f.applyTo(f_args));
		}
		return result.contents();
	    }
	});

	/* Predicates */

	define(new JFunction("symbolp", 1) {
	    public Object applyTo(Object[] args) {
		return Boolean.valueOf(args[0] instanceof Symbol);
	    }
	});

	define(new JFunction("numberp", 1) {
	    public Object applyTo(Object[] args) {
		return Boolean.valueOf(args[0] instanceof Number);
	    }
	});

	define(new JFunction("stringp", 1) {
	    public Object applyTo(Object[] args) {
		return Boolean.valueOf(args[0] instanceof String);
	    }
	});

	define(new JFunction("listp", 1) {
	    public Object applyTo(Object[] args) {
		return Boolean.valueOf(args[0] instanceof LList);
	    }
	});

	define(new JFunction("consp", 1) {
	    public Object applyTo(Object[] args) {
		return Boolean.valueOf(args[0] instanceof Cons);
	    }
	});

	define(new JFunction("not", 1) {
	    public Object applyTo(Object[] args) {
		return Boolean.valueOf(!isTrue(args[0]));
	    }
	});

	define(new JFunction("eq", 2) {
	    public Object applyTo(Object[] args) {
		return Boolean.valueOf(args[0] == args[1]);
	    }
	});

	define(new JFunction("equal", 2) {
	    public Object applyTo(Object[] args) {
		return Boolean.valueOf(Lisp.equal(args[0], args[1]));
	    }
	});

	/* List functions */

	define(new JFunction("null", 1) {
	    public Object applyTo(Object[] args) {
		return Boolean.valueOf(args[0] == Lisp.NIL);
	    }
	});

	define(new JFunction("cons", 2) {
	    public Object applyTo(Object[] args) {
		return new Cons(args[0],
				(LList)mustBe(LList.class, args[1]));
	    }
	});

	define(new JFunction("car", 1) {
	    public Object applyTo(Object[] args) {
		LList arg0 = (LList)mustBe(LList.class, args[0]);
		return arg0.car();
	    }
	});

	define(new JFunction("cdr", 1) {
	    public Object applyTo(Object[] args) {
		LList arg0 = (LList)mustBe(LList.class, args[0]);
		return arg0.cdr();
	    }
	});

	define("first", globalEnv.lookup(Symbol.intern("car")));
	define("rest", globalEnv.lookup(Symbol.intern("cdr")));

	define(new JFunction("list", Function.ANY_ARITY) {
	    public Object applyTo(Object[] args) {
		LList result = Lisp.NIL;
		for (int i = args.length - 1; i >= 0; i--) {
		    result = new Cons(args[i], result);
		}
		return result;
	    }
	});

	define(new JFunction("make-list", 1) {
	    public Object applyTo(Object[] args) {
		Collection c = (Collection)mustBe(Collection.class, args[0]);
		return LList.newLList(c);
	    }
	});

	topLevelEval
	  (Lisp.readFromString
	     ("(defun mapcar (f l) " +
              "  (if (null l) nil (cons (f (car l)) (mapcar f (cdr l)))))"));

	/* Sequence functions */

	define(new JFunction("empty", 1) {
	    public Object applyTo(Object[] args) {
		Object a = args[0];
		if (a instanceof LList)
		    return Boolean.valueOf(a == Lisp.NIL);
		else if (a instanceof String)
		    return Boolean.valueOf(((String)a).length() == 0);
		else if (a instanceof Collection)
		    return Boolean.valueOf(((Collection)a).isEmpty());
		else if (a instanceof Object[])
		    return Boolean.valueOf(((Object[])a).length == 0);
		else
		    throw new IllegalArgumentException
			("Can't apply empty to " + a);
	    }
	});

	define(new JFunction("length", 1) {
	    public Object applyTo(Object[] args) {
		Object a = args[0];
		if (a instanceof LList)
		    return new Long(((LList)a).length());
		else if (a instanceof String)
		    return new Long(((String)a).length());
		else if (a instanceof Collection)
		    return new Long(((Collection)a).size());
		else if (a instanceof Object[])
		    return new Long(((Object[])a).length);
		else
		    throw new IllegalArgumentException
			("Can't take the length of " + a);
	    }
	});

	define(new JFunction("elt", 2) {
	    public Object applyTo(Object[] args) {
		Object a = args[0];
		int i = ((Number)mustBe(Number.class, args[1])).intValue();
		if (a instanceof LList)
		    return ((LList)a).get(i);
		else if (a instanceof String)
		    return ((String)a).substring(i, i+1);
		else if (a instanceof List)
		    return ((List)a).get(i);
		else if (a instanceof Object[])
		    return ((Object[])a)[i];
		else
		    throw new IllegalArgumentException
			("Can't apply elt to " + a);
	    }
	});

	/* Collections */

	define(new JFunction("make-collection", 2) {
	    public Object applyTo(Object[] args) {
		Symbol type = (Symbol)mustBe(Symbol.class, args[0]);
		Collection c = (Collection)mustBe(Collection.class, args[1]);
		Collection result = makeInstance(type);
		result.addAll(c);
		return result;
	    }
	    Collection makeInstance(Symbol type) {
		if (type == LIST) return new LinkedList();
		else if (type == SET) return new LinkedHashSet(); // /\/
		else if (type == SORTED_SET) return new TreeSet();
		else
		    throw new IllegalArgumentException
			(type + " is not a valid type for make-collection.");
	    }
	    private final Symbol
		LIST = Symbol.intern("list"),
		SET = Symbol.intern("set"),
		SORTED_SET = Symbol.intern("sorted-set");
	});

	define(new JFunction("contains", 2) {
	    public Object applyTo(Object[] args) {
		Collection c = (Collection)mustBe(Collection.class, args[0]);
		Object a = args[1];
		return Boolean.valueOf(c.contains(a));
	    }
	});

	define(new JFunction("add", 2) {
	    public Object applyTo(Object[] args) {
		Collection c = (Collection)mustBe(Collection.class, args[0]);
		Object a = args[1];
		return Boolean.valueOf(c.add(a));
	    }
	});

	define(new JFunction("remove", 2) {
	    public Object applyTo(Object[] args) {
		Collection c = (Collection)mustBe(Collection.class, args[0]);
		Object a = args[1];
		return Boolean.valueOf(c.remove(a));
	    }
	});

	/* Hash tables */

	define(new JFunction("make-hash-map", 0) {
	    public Object applyTo(Object[] args) {
		return new StableHashMap();
	    }
	});

	define(new JFunction("make-context-hash-map", 0) {
	    public Object applyTo(Object[] args) {
		return new ix.util.context.ContextHashMap();
	    }
	});

	define(new JFunction("get", 3) { // args: map, key, default-value
	    public Object applyTo(Object[] args) {
		Map map = (Map)mustBe(Map.class, args[0]);
		Object key = args[1];
		Object defaultValue = args[2];
		Object value = map.get(key);
		return value == null ? defaultValue : value;
	    }
	});

	define(new JFunction("put", 3) { // args: map, key, value
	    public Object applyTo(Object[] args) {
		Map map = (Map)mustBe(Map.class, args[0]);
		Object key = args[1];
		Object value = args[2];
		map.put(key, value);
		return value;
	    }
	});

	define(new JFunction("for-each-entry", 2) {
	    public Object applyTo(Object[] args) {
		Function f = (Function)mustBe(Function.class, args[0]);
		Map m = (Map)mustBe(Map.class, args[1]);
		f.checkArity(2);
		Object[] f_args = new Object[2];
		for (Iterator i = m.entrySet().iterator(); i.hasNext();) {
		    Map.Entry e = (Map.Entry)i.next();
		    f_args[0] = e.getKey();
		    f_args[1] = e.getValue();
		    f.applyTo(f_args);
		}
		return m;
	    }
	});

	/* Numeric functions */

	define(new JFunction("=", 2) {
	    public Object applyTo(Object[] args) {
		Number n1 = (Number)mustBe(Number.class, args[0]);
		Number n2 = (Number)mustBe(Number.class, args[1]);
		if (n1 instanceof Long && n2 instanceof Long)
		    return Boolean.valueOf
			     (n1.longValue() == n2.longValue());
		else
		    return Boolean.valueOf
			     (n1.doubleValue() == n2.doubleValue());
	    }
	});

	define(new JFunction("/=", 2) {
	    public Object applyTo(Object[] args) {
		Number n1 = (Number)mustBe(Number.class, args[0]);
		Number n2 = (Number)mustBe(Number.class, args[1]);
		if (n1 instanceof Long && n2 instanceof Long)
		    return Boolean.valueOf
			     (n1.longValue() != n2.longValue());
		else
		    return Boolean.valueOf
			     (n1.doubleValue() != n2.doubleValue());
	    }
	});

	define(new JFunction("<", 2) {
	    public Object applyTo(Object[] args) {
		Number n1 = (Number)mustBe(Number.class, args[0]);
		Number n2 = (Number)mustBe(Number.class, args[1]);
		if (n1 instanceof Long && n2 instanceof Long)
		    return Boolean.valueOf
			     (n1.longValue() < n2.longValue());
		else
		    return Boolean.valueOf
			     (n1.doubleValue() < n2.doubleValue());
	    }
	});

	define(new JFunction(">", 2) {
	    public Object applyTo(Object[] args) {
		Number n1 = (Number)mustBe(Number.class, args[0]);
		Number n2 = (Number)mustBe(Number.class, args[1]);
		if (n1 instanceof Long && n2 instanceof Long)
		    return Boolean.valueOf
			     (n1.longValue() > n2.longValue());
		else
		    return Boolean.valueOf
			     (n1.doubleValue() > n2.doubleValue());
	    }
	});

	define(new JFunction("<=", 2) {
	    public Object applyTo(Object[] args) {
		Number n1 = (Number)mustBe(Number.class, args[0]);
		Number n2 = (Number)mustBe(Number.class, args[1]);
		if (n1 instanceof Long && n2 instanceof Long)
		    return Boolean.valueOf
			     (n1.longValue() <= n2.longValue());
		else
		    return Boolean.valueOf
			     (n1.doubleValue() <= n2.doubleValue());
	    }
	});

	define(new JFunction(">=", 2) {
	    public Object applyTo(Object[] args) {
		Number n1 = (Number)mustBe(Number.class, args[0]);
		Number n2 = (Number)mustBe(Number.class, args[1]);
		if (n1 instanceof Long && n2 instanceof Long)
		    return Boolean.valueOf
			     (n1.longValue() >= n2.longValue());
		else
		    return Boolean.valueOf
			     (n1.doubleValue() >= n2.doubleValue());
	    }
	});

	define(new JFunction("+", Function.ANY_ARITY) {
	    public Object applyTo(Object[] args) {
		double double_result = 0;
		long long_result = 0;
		boolean isFloat = false;
		for (int i = 0; i < args.length; i++) {
		    Number n = (Number)mustBe(Number.class, args[i]);
		    if (isFloat)
			double_result += n.doubleValue();
		    else if (n instanceof Long)
			long_result += n.longValue();
		    else if (n instanceof Double) {
			double_result = (double)long_result;
			double_result += n.doubleValue();
			isFloat = true;
		    }
		    else
			throw new ClassCastException
			    ("I don't know how to add " + n);
		}
		if (isFloat)
		    return new Double(double_result);
		else
		    return new Long(long_result);
	    }
	});

	define(new JFunction("-", 2) {
	    public Object applyTo(Object[] args) {
		Number n1 = (Number)mustBe(Number.class, args[0]);
		Number n2 = (Number)mustBe(Number.class, args[1]);
		if (n1 instanceof Long && n2 instanceof Long)
		    return new Long(n1.longValue() - n2.longValue());
		else
		    return new Double(n1.doubleValue() - n2.doubleValue());
	    }
	});

	define(new JFunction("*", Function.ANY_ARITY) {
	    public Object applyTo(Object[] args) {
		double double_result = 1;
		long long_result = 1;
		boolean isFloat = false;
		for (int i = 0; i < args.length; i++) {
		    Number n = (Number)mustBe(Number.class, args[i]);
		    if (isFloat)
			double_result *= n.doubleValue();
		    else if (n instanceof Long)
			long_result *= n.longValue();
		    else if (n instanceof Double) {
			double_result = (double)long_result;
			double_result *= n.doubleValue();
			isFloat = true;
		    }
		    else
			throw new ClassCastException
			    ("I don't know how to add " + n);
		}
		if (isFloat)
		    return new Double(double_result);
		else
		    return new Long(long_result);
	    }
	});

	define(new JFunction("/", 2) {
	    public Object applyTo(Object[] args) {
		Number n1 = (Number)mustBe(Number.class, args[0]);
		Number n2 = (Number)mustBe(Number.class, args[1]);
		if (n1 instanceof Long && n2 instanceof Long)
		    return new Long(n1.longValue() / n2.longValue());
		else
		    return new Double(n1.doubleValue() / n2.doubleValue());
	    }
	});

	define(new JFunction("sin", 1) {
	    public Object applyTo(Object[] args) {
		Number n = (Number)mustBe(Number.class, args[0]);
		double d = n.doubleValue();
		return new Double(Math.sin(d));
	    }
	});

	define(new JFunction("cos", 1) {
	    public Object applyTo(Object[] args) {
		Number n = (Number)mustBe(Number.class, args[0]);
		double d = n.doubleValue();
		return new Double(Math.cos(d));
	    }
	});

	define(new JFunction("tan", 1) {
	    public Object applyTo(Object[] args) {
		Number n = (Number)mustBe(Number.class, args[0]);
		double d = n.doubleValue();
		return new Double(Math.tan(d));
	    }
	});

	define(new JFunction("asin", 1) {
	    public Object applyTo(Object[] args) {
		Number n = (Number)mustBe(Number.class, args[0]);
		double d = n.doubleValue();
		return new Double(Math.asin(d));
	    }
	});

	define(new JFunction("acos", 1) {
	    public Object applyTo(Object[] args) {
		Number n = (Number)mustBe(Number.class, args[0]);
		double d = n.doubleValue();
		return new Double(Math.acos(d));
	    }
	});

	define(new JFunction("atan", 1) {
	    public Object applyTo(Object[] args) {
		Number n = (Number)mustBe(Number.class, args[0]);
		double d = n.doubleValue();
		return new Double(Math.atan(d));
	    }
	});

	define(new JFunction("atan2", 2) {
	    public Object applyTo(Object[] args) {
		Number n1 = (Number)mustBe(Number.class, args[0]);
		Number n2 = (Number)mustBe(Number.class, args[1]);
		return new Double(Math.atan2(n1.doubleValue(),
					     n2.doubleValue()));
	    }
	});

	define(new JFunction("to-radians", 1) {
	    public Object applyTo(Object[] args) {
		Number n = (Number)mustBe(Number.class, args[0]);
		double d = n.doubleValue();
		return new Double(Math.toRadians(d));
	    }
	});

	define(new JFunction("to-degrees", 1) {
	    public Object applyTo(Object[] args) {
		Number n = (Number)mustBe(Number.class, args[0]);
		double d = n.doubleValue();
		return new Double(Math.toDegrees(d));
	    }
	});

	define(new JFunction("sqrt", 1) {
	    public Object applyTo(Object[] args) {
		Number n = (Number)mustBe(Number.class, args[0]);
		double d = n.doubleValue();
		return new Double(Math.sqrt(d));
	    }
	});

	/* Simple output, chiefly for debugging */

	define(new JFunction("print", Function.ANY_ARITY) {
	    public Object applyTo(Object[] args) {
		for (int i = 0; i < args.length; i++) {
		    System.out.print(args[i].toString());
		}
		return Lisp.NIL;
	    }
	});

	define(new JFunction("println", Function.ANY_ARITY) {
	    public Object applyTo(Object[] args) {
		for (int i = 0; i < args.length; i++) {
		    System.out.print(args[i].toString());
		}
		System.out.println("");
		return Lisp.NIL;
	    }
	});

	/* Input */

	define(new JFunction("read", 0) {
	    public Object applyTo(Object[] args) {
		return reader.readObject();
	    }
	});

	define(new JFunction("read-line", 0) {
	    public Object applyTo(Object[] args) {
		return Util.readLine(System.in);
	    }
	});

	/* Java-related utilities */

    }

    /* * * Initial values * * */

    protected void defineInitialValues () {

	define("t", TRUE);

	define("TRUE", TRUE);

	define("FALSE", FALSE);

	define("PI", new Double(Math.PI));

    }

}
