/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sun Feb 19 00:01:32 2006 by Jeff Dalton
 */

package ix.util.lisp;

import java.io.*;
import java.net.URL;
import java.util.*;

import ix.util.*;
import ix.util.xml.XML;


/** The class for static Lisp utilities. */

public class Lisp {

    private Lisp() { }		// can't instantiate


    public static final Null NIL = new Null();

    public static final Object EOF = new UniqueObject("EOF");


    /* cons */

    public static final Cons cons(Object car, LList cdr) {
	// /\/: maybe Object cdr and cast to LList.
	return new Cons(car, cdr);
    }


    /* list */

    public static final Null list() {
	return NIL;
    }

    public static final Cons list(Object a) {
	return new Cons(a, NIL);
    }

    public static final Cons list(Object a, Object b) {
	return new Cons(a, new Cons(b, NIL));
    }

    public static final Cons list(Object a, Object b, Object c) {
	return new Cons(a, new Cons(b, new Cons(c, NIL)));
    }

    public static final Cons list(Object a, Object b, Object c, Object d) {
	return new Cons(a, new Cons(b, new Cons(c, new Cons(d, NIL))));
    }

    public static final Cons list(Object a, Object b, Object c,
				  Object d, Object e) {
	return new Cons(a, new Cons(b, new Cons(c, 
                 new Cons(d, new Cons(e, NIL)))));
    }

    public static final Cons list(Object a, Object b, Object c,
				  Object d, Object e, Object f) {
	return new Cons(a, new Cons(b, new Cons(c, 
                 new Cons(d, new Cons(e, new Cons(f, NIL))))));
    }


    /* equal */

    public static final boolean equal(Object a, Object b) {
	if (a == b)
	    return true;
	else if (a instanceof Cons)
	    return b instanceof Cons
		   ? equal(((Cons)a).car, ((Cons)b).car) &&
		     equal(((Cons)a).cdr, ((Cons)b).cdr)
		   : false;
	else if (a instanceof String)
	    return b instanceof String ? a.equals(b) : false;
	else if (a instanceof Number)
	    return b instanceof Number ? a.equals(b) : false;
	else
	    return false;
    }

    public static LispReader openForInput(URL url) throws IOException {
	Reader reader = Util.openURLReader(url);
	return new LispReader(new BufferedReader(reader));
    }

    public static LispReader openForInput(String resourceName)
	   throws IOException {
	URL url = XML.toURL(resourceName);
	if (url == null)
	    throw new IllegalArgumentException("Can't find " + resourceName);
	else
	    return openForInput(url);
    }

    /* readFromString */

    public static Object readFromString(String s) {
	LispReader lr = new LispReader(s);
	try {
	    Object result = lr.readObject();
	    if (lr.readObject() != Lisp.EOF)
		throw new LispReadException(
		    s.trim().startsWith("(")
		    ? "Extra close paren"
		    : "More than one object");
	    return result;
	}
	catch (LispReadException e) {
	    throw new LispReadException(
	      e.getMessage() + " in string " + Strings.quote(s));
	}
    }

    public static LList elementsFromString(String s) {
	// return (LList)Lisp.readFromString("(" + s + ")");
	LListCollector result = new LListCollector();
	LispReader lr = new LispReader(s);
	try {
	    while (true) {
		Object elt = lr.readObject();
		if (elt == Lisp.EOF)
		    return result.contents();
		else
		    result.add(elt);
	    }
	}
	catch (LispReadException e) {
	    throw new LispReadException(
	      e.getMessage() + " in string " + Strings.quote(s));
	}
    }


    /* printToString */

    public static String printToString(Object a) {
	if (a == null)
	    return "#<null>";
	else if (a instanceof DelimitedSymbol)
	    return quotedAndEscaped('|', a.toString());
	else if (a instanceof Symbol)
	    return a.toString();
	else if (a instanceof Number)
	    return a.toString();
	else if (a instanceof String)
	    return quotedAndEscaped('"', (String)a);
	else if (a instanceof Null)
	    // return "()";
	    return "nil";
	else if (a instanceof Cons) {
	    LList at = (Cons)a;
	    String s = "(" + printToString(at.car());
	    for (; at.cdr() instanceof Cons; at = at.cdr()) {
		s += " " + printToString(at.cdr().car());
	    }
	    return s + ")";
	}
	else {
	    // Debug.warn("printToString got a " + a.getClass() + ":" + a);
	    // return "#<" + a.getClass() + ">";
	    // return "#<" /* + a.getClass() +": " */ + a.toString() + ">";
	    // For a long time, it was:
	    // return a.toString();
	    if (a instanceof ix.icore.Variable)
		return a.toString(); 		// /\/ hack
	    else
		return Lisp.hashName(a);
	}
    }

    public static boolean isFullyPrintable(Object a) {
	return a instanceof Symbol
	    || a instanceof String
	    ||(a instanceof Cons && isFullyPrintableCons((Cons)a))
	    || a instanceof Null
	    || a instanceof Long
	    || a instanceof Double
	    || a instanceof Integer
	    || a instanceof Float;
    }
    private static boolean isFullyPrintableCons(Cons c) {
	for (LList tail = c; tail != Lisp.NIL; tail = tail.cdr()) {
	    if (!isFullyPrintable(tail.car()))
		return false;
	}
	return true;
    }

    public static String elementsToString(LList elts) {
	if (elts.isEmpty())
	    return "";
	else {
	    String text = printToString(elts);
	    // Remove outermost parens.
	    return text.substring(1, text.length() - 1);
	}
    }

    public static String quotedAndEscaped(char quote, String s) {
	int len = s.length();
	// See how many escape chars we'll need to add
	int increase = 0;
	for (int i = 0; i < len;) {
	    char c = s.charAt(i);
	    if (c == quote || c == '\\' || c == '\n')
		increase++;
	    else if (c == '\r') {
		increase++;
		// deal with possible CRLF
		int j = i + 1;
		if (j < len) {
		    char c2 = s.charAt(j);
		    if (c2 == '\n')
			i = j;
		}
	    }
	    i++;
	}
	// Make a buffer large enough to hold the text, the escape chars,
	// and a quote at each end.
	SimpleStringBuffer buf = new SimpleStringBuffer(len + increase + 2);
	buf.append(quote);
	for (int i = 0; i < len;) {
	    char c = s.charAt(i);
	    if (c == quote || c == '\\' || c == '\n')
		buf.append('\\');
	    else if (c == '\r') {
		buf.append('\\');
		// deal with possible CRLF
		int j = i + 1;
		if (j < len) {
		    char c2 = s.charAt(j);
		    if (c2 == '\n') {
			buf.append(c);
			i = j;
			c = c2;
		    }
		}
	    }
	    buf.append(c);
	    i++;
	}
	buf.append(quote);
	return buf.toString();
    }

    /* object hash and unhash */

    private static ObjectHash defaultHash = new ObjectHash();

    public static int hash(Object obj) {
	return defaultHash.hash(obj);
    }

    public static Object unhash(int h) {
	return defaultHash.unhash(h);
    }

    public static String hashName(Object obj) {
	return defaultHash.hashName(obj);
    }

    public static Object parseHashName(String name) {
	return defaultHash.parseHashName(name);
    }

}
