/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sun May 14 18:52:55 2006 by Jeff Dalton
 */

package ix.util.lisp;

import java.util.*;
import java.lang.reflect.*;

import ix.util.*;


/** Non-empty lists. */

public class Cons extends LList implements Cloneable {

    Object car;
    LList cdr;			// only proper lists

    public Cons(Object car, LList cdr) {
	this.car = car;
	this.cdr = cdr;
    }

    // Code for building from other collections ...

    public Cons(Collection c) {
	this(null, Lisp.NIL);
	Iterator i = c.iterator();
	if (!i.hasNext())
	    throw new IllegalArgumentException
		("Trying make a Cons from an empty collection");
	writeIn(this, c.iterator());
    }

    /**
     * Creates a new instance of this class.  This method must be
     * redefined any any Cons subclass.
     */
    protected Cons newInstance(Object car, LList cdr) {
	return new Cons(car, cdr);
    }

    private static final Class[] consSig =
	new Class[] {Object.class, LList.class};

    public static Cons typedCons(Class consClass, Object car, LList cdr) {
	// Does not Assume subclasses redefine newInstance.
	// The newInstance called here is the consClass's constructor's
	// newInstance(Object[]) method.
	Debug.expect(Cons.class.isAssignableFrom(consClass));
	try {
	    Constructor cons = consClass.getConstructor(consSig);
	    return (Cons)cons.newInstance(new Object[] {car, cdr});
	}
	catch (Exception e) {
	    throw new RethrownException
		(e, "Can't do typedCons for class " + consClass +
		    " because " + Debug.describeException(e));
	}
    }

    public boolean addAll(Collection c) {
	Iterator i = c.iterator();
	if (!i.hasNext())
	    return false;
	Cons tail = (Cons)this.lastCons();
	Cons new_tail = newInstance(null, Lisp.NIL);
	tail.setCdr(new_tail);
	writeIn(new_tail, i);
	return true;
    }

    private void writeIn(Cons tail, Iterator i) {
	Debug.expect(tail.car() == null);
	tail.setCar(i.next());
	while (i.hasNext()) {
	    tail.setCdr(newInstance(i.next(), Lisp.NIL));
	    tail = (Cons)tail.cdr();
	}
    }

    // ... end of code for building from other collections

    public boolean isNull() { return false; }

    public Object car() { return car; }
    public LList cdr() { return cdr; }

    public void setCar(Object c) {
	car = c;
    }

    public void setCdr(Object c) {
	cdr = (LList)c;
    }

    public int length() {
	int i = 1;
	for (Cons at = this; at.cdr != Lisp.NIL; at = (Cons)at.cdr, ++i)
	    {}
	return i;
    }

    public Object elementAt(int i) {
	int j = 0;
	for (LList at = this; at != Lisp.NIL; at = ((Cons)at).cdr) {
	    if (j++ == i)
		return ((Cons)at).car;
	}
	return Lisp.NIL;
    }

    public Enumeration elements() {
	return new ConsEnumeration(this);
    }

    public boolean equal(LList list) {
	return Lisp.equal(this, list);
    }

    public boolean find(Object a) {
	for (LList at = this; at != Lisp.NIL; at = ((Cons)at).cdr) {
	    if (((Cons)at).car == a)
		return true;
	}
	return false;
    }

    public LList append(LList tail) {
	return new Cons(this.car, this.cdr.append(tail));
    }

    /*
     * Java Object stuff
     */

    public Object clone() {
	return append(Lisp.NIL);	// yes, that old trick
    }

//      public boolean equals(Object a) {
//  	    return Lisp.equal(this, a);
//      }

    // public int hashCode() {
    //     return car.hashCode();	// yes, I know that's totally losing
    // }

    public int compareTo(Object o) {
	if (o instanceof Cons) {
	    Cons that = (Cons)o;
	    Iterator thisIter = this.iterator();
	    Iterator thatIter = that.iterator();
	    while (thisIter.hasNext() && thatIter.hasNext()) {
		int c = ((Comparable)thisIter.next())
		            .compareTo(thatIter.next());
		if (c != 0)
		    return c;
	    }
            if (!thisIter.hasNext()) {
		if (!thatIter.hasNext())
		    return 0;
		else
		    return -1;
            }
	    else {
		Debug.expect(!thatIter.hasNext());
		return +1;
	    }
	}
	else
	    throw new ClassCastException
		("Cannot compare an LList to " + o +
		 ".  The LList was " + this);
    }

    public String toString() {
	return Lisp.printToString(this);
    }

}


/** Cons enumerations. */

class ConsEnumeration extends LListEnumeration {

    protected LList tail;

    ConsEnumeration(Cons c) {
	tail = c;
    }

    public boolean hasMoreElements() {
	return tail instanceof Cons;
    }

    public Object nextElement() {
	Object elt = ((Cons)tail).car;
	tail = ((Cons)tail).cdr;
	return elt;
    }

}
