/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Sep 11 06:29:08 2006 by Jeff Dalton
 * Copyright: (c) 2001 - 2006, AIAI, University of Edinburgh
 */

package ix.util.ipc;

import java.util.*;
import java.io.*;
import java.net.*;

import javax.swing.*;

import ix.icore.Sendable;

import ix.iface.util.IFUtil;

import ix.util.*;
import ix.util.xml.*;
import ix.util.lisp.*;

/**
 * A communication strategy in which a destination is mapped to a
 * host and port number, and objects are sent by writing their
 * serialization to a socket. <p>
 *
 * Command-line arguments / parameters:
 *
 * <pre>
 *    -name-server=<i>host</i>:<i>port</i>
 *    -run-as-name-server=<i>boolean</i>
 *    -host=<i>host</i>
 * </pre>
 *
 * <tt>-host</tt> is used when it is necessary to specify the host name
 * this agent should use when registering with the name-server.
 *
 * @see ix.util.Parameters
 */
public class SerializedCommunicationStrategy 
             implements IPC.CommunicationStrategy {

    static final String DEFAULT_NAME_SERVER_ADDRESS = "localhost:5555";

    ServiceAddress nameServerAddress = null;

    HashMap agentTable = new HashMap();

    String thisAgentsName = "not an agent name";

    public SerializedCommunicationStrategy() {
    }

    public Object preEncode(Object contents) {
	return contents;
    }

    public Object postDecode(Object contents) {
	return contents;
    }

    public void sendObject(Object destination, Object contents) {
	getAgentRecord((String)destination).sendObject(contents);
    }

    public Object sendRequest(Object destination, Object contents) {
	return getAgentRecord((String)destination).sendRequest(contents);
    }

    public void setDestinationAddress(String destination,
				      ServiceAddress addr) {
	getAgentRecord(destination).setAddress(addr);
    }

    AgentRecord getAgentRecord(String name) {
	synchronized (agentTable) {
	    AgentRecord rec = (AgentRecord)agentTable.get(name);
	    if (rec == null) {
		rec = new AgentRecord(name);
		agentTable.put(name, rec);
	    }
	    return rec;
	}
    }

    class AgentRecord {

	String name = null;
	ServiceAddress addr = null;
	IPC.Connection connection = null;

	AgentRecord(String name) {
	    this.name = name;
	}

	synchronized void setAddress(ServiceAddress address) {
	    this.addr = address;
	}

	synchronized void sendObject(Object contents) {
	    Debug.noteln("Sending to " + name + " at " + addr);
	    boolean isOldConnection = connection != null;
	    boolean isOldAddr = addr != null;
	    // Connect if not already connected.
	    IPC.Connection c = null;
	    try { 
		c = ensureConnection();
	    }
	    catch (IPC.IPCException e) {
		Debug.noteln("Cannot connect because", e);
		discardAnyConnection();
		// Discard any address.  This also ensures that
		// isOldAddr will be false in a recursive call.
		addr = null;
		if (isOldAddr) {
		    // It's possible the agent exited and restarted
		    // at a different address.
		    Debug.noteln("Will try again with", name);
		    // Tell the user what's happening.
		    if (Parameters.isInteractive()) {
			Util.displayAndWait(null, new Object[] {
			    "Cannot connect to " + name,
			    "Will try resending once to its current address."
			});
		    }
		    // Try sending again.
		    sendObject(contents);
		    return;
		}
		else
		    throw e;
	    }
	    // Send
	    try {
		c.send(preEncode(contents));
	    }
	    catch (IPC.IPCException e) {
		Debug.noteln("Can't send because", e);
		// Assume the connection is broken and so should be
		// discarded.  This also ensures that isOldConnection
		// will be false in a recursive call.
		discardAnyConnection();
		if (isOldConnection) {
		    // If the failure's with a old connection, it's possible
		    // that a new connection might work, for instance if
		    // the other agent has exited and been restarted.
		    Debug.noteln("Will attempt one resend to", name);
		    // Tell the user what's up.
		    if (Parameters.isInteractive()) {
			Util.displayAndWait(null, new Object[] {
			    "Broken connection to " + name,
			    "Will try resending once with a new connection."
			});
		    }
		    // Try sending again.
		    sendObject(contents);
		    return;
		}
		else {
		    throw e;
		}
	    }
	}

	synchronized Object sendRequest(Object contents) {
	    try { 
		Debug.noteln("Sending request to " + name, contents);
		IPC.Connection c = ensureConnection();
		c.send(preEncode(contents));
		Object reply = postDecode(c.receive());
		Debug.noteln("Received reply from " + name, reply);
		return reply;
	    }
	    catch (IPC.IPCException e) {
		discardAnyConnection();
		throw e;
	    }
	}

	private IPC.Connection ensureConnection() {
	    if (connection == null) {
		ensureAddress();
		connection = connect();
	    }
	    return connection;
	}

	private void ensureAddress() {
	    if (addr == null) {
		Debug.expect(!name.equals("name-server"),
			     "Asking name-server for its own address.");
		addr = askNameServer(name);
	    }
	}

	private void discardAnyConnection() {
	    if (connection != null) {
		Debug.noteln("Discarding connection to", name);
		connection.close();
		connection = null;
	    }
	    else
		Debug.noteln("No connection to " + name + " to discard.");
	}

	private IPC.Connection connect() {
	    Debug.noteln("Connecting to " + name
			 + " on " + addr.getHost()
			 + " port " + addr.getPort());
	    Debug.expect(connection == null, "connecting twice");
	    try {
		Socket s = new Socket(addr.getHost(), addr.getPort());
		IPC.Connection c = new ObjectStreamConnection(name, s);
		return c;
	    }
	    catch (IOException e) {
		Debug.noteException(e, false);
		// Assume addr may no longer work
		ServiceAddress oldAddr = addr;
		addr = null;
		throw new IPC.IPCException
		    ("Cannot connect to " + name + " at " + oldAddr, e);
	    }
	}

    }

    /**
     * Requests a ServiceAddress from the name-server.
     *
     * @throws AssertionFailure if the name-server address isn't known
     *    or if the name-server returns an address that isn't a
     *    ServiceAddress.
     *
     * @throws IPC.IPCException if the name-server does not know a
     *    address for the specified destination.
     */
    ServiceAddress askNameServer(Object destination) {
	Object addr = sendRequest("name-server", destination);
	if (addr.equals("unknown")) {
	    throw new IPC.IPCException
		("Name server doesn't know about " + destination);
	}
	else {
	    Debug.expect(addr instanceof ServiceAddress,
			 "Bogus addr for " + destination,
			 addr);
	    return (ServiceAddress)addr;
	}
    }

    public void setupServer(Object destination, IPC.MessageListener listener) {
	// In this method, the destination is the agent who is setting
	// up the server, not a remote agent.
	Debug.noteln("Setting up a SerializedCommunicationStrategy");
	thisAgentsName = (String)destination;
	setupNameServerAddress();
	getAgentRecord("name-server").addr = nameServerAddress;
	if (Parameters.haveParameter("run-name-server"))
	    setupNameServer();
	ObjectStreamServer server = new ObjectStreamServer
	                             (this, destination, listener);
	server.setup();
	server.start();
    }

    protected void setupNameServerAddress() {
	// If the user's said "-no name-server", or in some other way
	// made the value of the "name-server" parameter be "false",
	// don't set up the address.
	if (Parameters.getParameter("name-server", "unspecified")
	    .equals("false"))
	    return;

	// If we already have a name-server addr and no name-server
	// parameter has been specified, we don't have to do anything.
	if (nameServerAddress != null
	        && !Parameters.haveParameter("name-server"))
	    return;

	// Syntax is host:port, both required.
	String addr = Parameters.getParameter("name-server",
					      DEFAULT_NAME_SERVER_ADDRESS);
	Debug.noteln("Setting name-server addr", addr);
	nameServerAddress = new ServiceAddress(addr);
    }

    protected void setupNameServer() {
	ServiceAddress a = nameServerAddress;
	Debug.expect(a != null, "Can't run name server, address unknown");
	new ObjectStreamNameServer(this, a)
	    .start();
    }


    /**
     * A Thread that accepts connections to a ServerSocket
     * and creates an object-reading thread for each connection.
     * Each of the object-reading threads notifies the specified
     * listener when an object is received.  The listener's
     * <tt>messageReceived</tt> method should presumably therefore
     * be synchronized.
     */
    public static class ObjectStreamServer extends CatchingThread {

	SerializedCommunicationStrategy strategy;
	Object destination;
	IPC.MessageListener listener;

	ServiceAddress addr;
	ServerSocket servSock;

	public ObjectStreamServer(SerializedCommunicationStrategy strategy,
				  Object destination,
				  IPC.MessageListener listener) {
	    this.strategy = strategy;
	    this.destination = destination;
	    this.listener = listener;
	}

	public void setup() {
	    try {
		// If we haven't been told a port number, ask the
		// operating system for a free one by specifying
		// port 0
		int port = Parameters.getInt("port", 0);
		servSock = new ServerSocket(port);
		// It's not clear how we ought to get the host name.
		// Note that 
		//   InetAddress in_addr = servSock.getInetAddress();
		//   String host = in_addr.getHostName();
		// just seems to get "0.0.0.0".  /\/
		String host = Util.getHostName();
		addr = new ServiceAddress
		    (host,
		     servSock.getLocalPort());
		// By this point, we should have a table entry for the
		// name-server if there's supposed to be one.
		// N.B. If SimpleNameServer.isStandAlone(), then
		// we are in the mini-agent running the name-server
		// and should not register.
		if (strategy.nameServerAddress != null
		      && !SimpleNameServer.isStandAlone())
		    registerWithNameServer();
	    }
	    catch (IOException e) {
		Debug.noteException(e);
		throw new IPC.IPCException
		    ("Problem setting up server for " + destination, e);
	    }
	}

	public void innerRun() {
	    Debug.expect(servSock != null, "Server started before set up");
	    Debug.noteln("Server running for", destination);
	    try {
		// ... our main loop ...
		while (true) {
		    Socket s = servSock.accept();
		    Debug.noteln("Client connection", s);
		    receiveFrom(new ObjectStreamConnection(s));
		}
	    }
	    catch (IOException e) {
		Debug.noteException(e);
		throw new IPC.IPCException
		    ("Problem in server for " + destination, e);
	    }
	}

	protected void registerWithNameServer() {
	    Debug.noteln("Registering " + destination + " as " +
			 addr.host + ":" + addr.port);
	    while (true) {
		try {
		    Object reply = strategy.sendRequest
			("name-server",
			 Lisp.list("register", destination, addr));
		    if (reply.equals("ok"))
			return;
		}
		catch (IPC.IPCException e) {
		    Debug.noteException(e, false);
		}
		if (!shouldWaitForNameServer())
		    return;
		// /\/: Restore name-server addr after failed
		// attempt to connect.
		strategy.getAgentRecord("name-server").addr =
		    strategy.nameServerAddress;
	    }
	}

	protected boolean shouldWaitForNameServer() {
	    Object server_addr = strategy.nameServerAddress;
	    Object[] message = {
		"Could not register " + destination
		    + " with name-server at " + server_addr,
		"Do you want to try again?"
	    };
	    return Util.dialogConfirms(null, message);
	}

	protected void receiveFrom(final ObjectStreamConnection connection) {
	    // An exception thrown by connection.receive() will propagate
	    // beyond the run method and hence cause the thread to die,
	    // which is more or less what we want.
	    new CatchingThread() {
		public void innerRun() {
		    Debug.noteln(destination + " receiver running ...");
		    receiveLoop(connection);
		}
	    }.start();
	}

	protected void receiveLoop(ObjectStreamConnection connection) {
	    try {
		while (true) {
		    Debug.noteln(destination + " waiting to receive ...");
		    Object contents =
			strategy.postDecode(connection.receive());
		    Debug.noteln(destination + " received", contents);
		    if (contents instanceof Sendable) {
			Object from = ((Sendable)contents).getSenderId();
			if (from != null) {
			    Debug.noteln("Sender was", from);
			    connection.setDestination(from);
			}
		    }
		    listener.messageReceived
			(new IPC.BasicInputMessage(contents));
		}
	    }
	    catch (IPC.BrokenConnectionException e) {
		// Happens when the other agent exits, so more or less
		// "normal" termination.
		connection.close();
		Debug.noteln(destination + " lost connection to " +
			     connection.getDestination());
	    }
	}

    }

    /**
     * A Thread that acts as a name-server on a specified port.
     */
    public static class ObjectStreamNameServer extends CatchingThread {

	SerializedCommunicationStrategy strategy;
	ServiceAddress addr;

	IPC.DestinationTable nameTable = new IPC.BasicDestinationTable();

	TextAreaFrame textFrame;

	ServerSocket servSock;

	boolean isStandAlone;

	public ObjectStreamNameServer
	        (SerializedCommunicationStrategy strategy,
		 ServiceAddress addr) {
	    this.strategy = strategy;
	    this.addr = addr;
	    this.isStandAlone = SimpleNameServer.isStandAlone();

	    // Set up the text frame and the server socket here,
	    // rather than in the new thread, so that it's clear
	    // when they've happened.

	    textFrame = new TranscriptFrame(addr, isStandAlone);

	    try {
		servSock = new ServerSocket(addr.port);
	    }
	    catch (IOException e) {
		Debug.noteException(e);
		// e.fillInStackTrace();
		throw new IPC.IPCException
		    ("Problem setting up name-server ServerSocket at " +
		     addr, e);
	    }

	}

	public void innerRun() {
	    Debug.noteln("I-X Name-server running at", addr);
	    try {
		while (true) {
		    Socket s = servSock.accept();
		    Debug.noteln("Client connection", s);
		    serveClientOn(new ObjectStreamConnection(s));
		}
	    }
	    catch (IOException e) {
		Debug.noteException(e);
		// e.fillInStackTrace();
		throw new IPC.IPCException
		    ("Problem in name-server at " + addr, e);
	    }
	}

	protected void serveClientOn(final ObjectStreamConnection connection) {
	    // An exception thrown by connection.receive() will propagate
	    // beyond the innerRun method and hence cause the thread to die,
	    // which is more or less what we want.  /\/: Not quite like
	    // that any more.
	    new CatchingThread() {
		public void innerRun() {
		    clientServiceLoop(connection);
		}
	    }.start();
	}

	protected void clientServiceLoop(ObjectStreamConnection connection) {
	    try {
		while (true) {
		    Object contents =
			strategy.postDecode(connection.receive());
		    ObjectStreamNameServer.this
			.handleMessage(connection, contents);
		}
	    }
	    catch (IPC.BrokenConnectionException e) {
		// Happens when the other agent exits, so more or less
		// "normal" termination.  /\/: Remove registration?
		connection.close();
		Debug.noteln("Name-server lost connection to",
			     connection.getDestination());
		transcript("Lost connection to " +
			   connection.getDestination());
	    }
	}

	protected synchronized void handleMessage
	    (ObjectStreamConnection connection,
	     Object contents) {

	    Debug.noteln("Name-server received", contents);

	    if (contents instanceof String) {
		ServiceAddress addr = (ServiceAddress)nameTable.get(contents);
		if (addr != null)
		    sendReply(connection, addr);
		else
		    sendReply(connection, "unknown");
	    }
	    else if (contents instanceof List) {
		List req = (List)contents;
		if (req.get(0).equals("register")
		      && req.get(1) instanceof String
		      && req.get(2) instanceof ServiceAddress) {
		    recordRegistration(connection,
				       (String)req.get(1),
				       (ServiceAddress)req.get(2));
		    sendReply(connection, "ok");
		}
		else {
		    Debug.noteln("Ilegal name-server request", contents);
		}
	    }
	    else {
		Debug.noteln("Ilegal name-server request", contents);
	    }

	}

	protected void sendReply(ObjectStreamConnection connection,
				 Object reply) {
	    connection.send(strategy.preEncode(reply));
	}

	protected void recordRegistration(ObjectStreamConnection connection,
					  String name,
					  ServiceAddress addr) {
	    connection.setDestination(name);
	    if (Parameters.getBoolean("ns-use-ip-addrs", false)) {
		Socket s = connection.getSocket();
		InetAddress remoteIPAddr = s.getInetAddress();
		ServiceAddress ipAddr =
		    new ServiceAddress(remoteIPAddr.getHostAddress(),
				       addr.getPort());
		nameTable.put(name, ipAddr);
		transcript("Registering " + name + " at " + ipAddr +
			   " instead of " + addr);
	    }
	    else {
		nameTable.put(name, addr);
		transcript("Registering " + name + " at " + addr);
	    }
	}

	protected void transcript(final String line) {
	    Debug.noteln("Name-server:", line);
	    javax.swing.SwingUtilities.invokeLater(new Runnable() {
		public void run() {
		    textFrame.appendLine(line);
		    if (!textFrame.getFrame().isShowing())
			textFrame.setVisible(true);
		}
	    });
	}

	class TranscriptFrame extends TextAreaFrame {
	    TranscriptFrame(ServiceAddress addr, boolean isStandAlone) {
		super("I-X Name-Server at " + addr.host + ":" + addr.port);
		setEditable(false);
		if (isStandAlone) {
		    JMenu file = IFUtil.ensureMenuBarMenu(frame, "File");
		    file.removeAll();
		    file.add(frame.makeMenuItem("Exit"));
		}
	    }
	    public void whenClosed() {
		// Don't close, because there's no way to get
		// the frame back up (before the name-server
		// wants to write something in it). /\/
	    }
	    public void fireButtonPressed(String command) {
		String msg = "Are you sure you want to exit the name-server?";
		if (command.equals("Exit")) {
		    if (Util.dialogConfirms(this, msg))
			System.exit(0);
		}
	    }
	}

    }

}
