/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Mar 31 19:06:06 2005 by Jeff Dalton
 * Copyright: (c) 2004, 2005, AIAI, University of Edinburgh
 */

package ix.util.context;

import java.util.*;

import ix.util.*;
import ix.util.lisp.*;

/**
 * A context-layered version of HashMap.  Properties:
 * <ul>
 *
 * <li>Entries cannot be removed.  The remove() and clear() methods,
 *   and the entry set's remove(), removeAll(), retainAll(), and clear()
 *   methods, are not supported.  However, there is a method that
 *   will remove all entries in all contexts: {@link #clearCompletely()}.
 *
 * <li>Iterators over the entry-set, key-set, or values return items
 *   in the order in which the keys were first added to the map.
 *
 * <li>Like the java.util classes WeakHashMap and IdentityHashMap,
 *   it is not a subclass of HashMap.
 *
 * </ul>
 *
 * <p>As of I-X 3.3, entires can be removed by calling the remove(Object)
 * method, but still not in any other way.  If a key is removed, then
 * used again with a put, it regains its original position in the
 * iteration order.  However, the order is forgotten after a
 * clearCompletely().</p>
 *
 * <p>As of I-X 4.1, clear() is also supported.  It removes all entries
 * in the current context.</p>
 */
public class ContextHashMap extends AbstractMap implements ContextMap {

    // Used for removals.
    protected static final Object UNDEF = new Object(); // a unique object

    protected ContextHolder holder;
    protected HashMap map;

    protected Entry keyChain = new Entry(null, null);   // header
    protected Entry keyChainLast = keyChain;

    /* * * Constructors * * */

    public ContextHashMap() {
	this(Context.getContextHolder());
    }

    public ContextHashMap(int initialCapacity) {
	// This constructor exists only for compatibility with HashMap.
	this.holder = Context.getContextHolder();
	this.map = new HashMap(initialCapacity);
    }

    public ContextHashMap(int initialCapacity, float loadFactor) {
	// This constructor exists only for compatibility with HashMap.
	this.holder = Context.getContextHolder();
	this.map = new HashMap(initialCapacity, loadFactor);
    }

    public ContextHashMap(Map m) {
	this(Context.getContextHolder(), m);
    }

    public ContextHashMap(ContextHolder holder) {
	this.holder = holder;
	this.map = new HashMap();
    }

    public ContextHashMap(ContextHolder holder, final Map m) {
	// /\/: Perhaps we should put m's entries in the root context.
	// That would be consistent with ContextValue, ContextInt, etc.
	// Ok - done.
	this.holder = holder;
	this.map = new HashMap();
	Context.inContext(holder, Context.rootContext, new Runnable() {
	    public void run() {
		putAll(m);
	    }
	});
    }

    /* * * Instance methods * * */

    // AbstractMap requires only entrySet() and, for a modifiable Map,
    // put(Object key, Object value) and an entrySet().iterator() that
    // supports remove().

    public Object put(Object key, Object value) {
	// Irritatingly complicated by the need to return the
	// previous value.
	Entry e = (Entry)map.get(key);
	Context c = holder.getContext();
	if (e == null) {
	    // New key, not seen before.  Create a new entry for it
	    // and put it at the end of the key-chain.
	    Entry ent = new Entry(key, new ContextLink(c, value));
	    ent.linkAfter(keyChainLast);
	    keyChainLast = ent;
	    map.put(key, ent);
	    return null;
	}
	else {
	    ContextLink link = e.valLink;
	    Object oldValue = Context.getInContext(link, c);
	    Context.setInContext(link, c, value);
	    return oldValue == UNDEF ? null : oldValue;
	}
    }

    public Object get(Object key) {
	Entry e = (Entry)map.get(key);
	if (e == null)
	    return null;
	ContextLink link = e.valLink;
	Object value = Context.getInContext(link, holder.getContext());
	return value == UNDEF ? null : value;
    }

    public Set entrySet() {
	// Because all the values in the underlying map are
	// already Map.Entries, we don't have to create any
	// entry obects.  But we can't quite use the underlying
	// map's entry set as-is.
	return new EntrySet(keyChain);
    }

    public boolean containsKey(Object key) {
	// The method from AbstractMap iterates over the entry-set.
	Entry e = (Entry)map.get(key);
	if (e == null)
	    return false;
	else {
	    ContextLink valLink = 
		Context.getValueLinkInContext
		    (e.valLink, holder.getContext());
	    return valLink != null && valLink.value != UNDEF;
	}
    }

    /** Remove all entries in all contexts. */
    public void clearCompletely() {
	map.clear();
	keyChain = new Entry(null, null);
	keyChainLast = keyChain;
    }

    public Object remove(Object key) {
	return put(key, UNDEF);
    }

    public void clear() {
	for (Iterator i = entrySet().iterator(); i.hasNext();) {
	    ((Map.Entry)i.next()).setValue(UNDEF);
	}
    }

    /**
     * The Map.Entry implementation for a ContextHashMap.
     */
    protected class Entry implements Map.Entry {

	Object key;
	ContextLink valLink;
	Entry nextEntry = null;

	Entry(Object key, ContextLink valLink) {
	    this.key = key;
	    this.valLink = valLink;
	}

	public Object getKey() {
	    return key;
	}
         
	public Object getValue() {
	    return Context.getInContext(valLink, holder.getContext());
	}

	public Object setValue(Object value) {
	    return put(key, value);
	}

	Entry getNext() {
	    return nextEntry;
	}

	void linkAfter (Entry previous) {
	    Debug.expect(previous.nextEntry == null, "pre already linked");
	    Debug.expect(this.nextEntry == null, "this already linked");
	    previous.nextEntry = this;
	}

	public boolean equals(Object o) {
	    if (!(o instanceof Map.Entry))
		return false;
	    Map.Entry e = (Map.Entry)o;
	    Object e_key = e.getKey();
	    Object e_value = e.getValue();
	    Object k = key;
	    Object v = getValue();
	    return (k == null ? e_key == null : k.equals(e_key))
		&& (v == null ? e_value == null : v.equals(e_value));
	}
         
	public int hashCode() {
	    Object k = key;
	    Object v = getValue();
	    return (k == null ? 0 : k.hashCode()) ^
		   (v == null ? 0 : v.hashCode());
 	}

	public String toString() {
	    return "CtxtMapEntry[" + getKey() + "-->" + getValue() +"]";
	}
	
    }

    /**
     * The Set implementation for a ContextHashMap entrySet.
     */
    protected class EntrySet extends AbstractSet {

	Context creationContext;

	Entry chain;
	int size = -1;

	EntrySet(Entry chain) {
	    this.chain = chain;
	    this.creationContext = holder.getContext();
	}

	public int size() {
	    if (size < 0)
		size = calcSize();
	    return size;
	}

	public Iterator iterator() {
	    return new EntrySetIterator(chain, creationContext);
	}

	protected int calcSize() {
	    int count = 0;
	    for (Iterator i = iterator(); i.hasNext();) {
		count++;
		i.next();
	    }
	    return count;
	}

    }

    /**
     * The Iterator implementation for a ContextHashMap entrySet.
     */
    protected class EntrySetIterator implements Iterator {

	Context setCreationContext;
	Entry chain;
	Entry next = null;

	EntrySetIterator(Entry chain, Context setCreationContext) {
	    Debug.expect(chain != null);
	    this.chain = chain;
	    this.setCreationContext = setCreationContext;
	    findNext();
	}

	private void findNext() {
	    Context c = holder.getContext();
	    if (c != setCreationContext)
		throw new IteratorContextException(setCreationContext, c);
	    while ((next = chain.getNext()) != null) {
		chain = next;
		ContextLink valLink =
		    Context.getValueLinkInContext(next.valLink, c);
		if (valLink != null && valLink.value != UNDEF)
		    return;
	    }
	}

	public boolean hasNext() {
	    return next != null;
	}

	public Object next() {
	    if (next == null) throw new ConsistencyException("next is null");
	    Object result = next;
	    findNext();
	    return result;
	}

	public void remove() {
	    throw new UnsupportedOperationException
		(this + " does not support remove()");
	}

    }

    protected class IteratorContextException extends ContextChangeException {

	IteratorContextException(Context expected, Context current) {
	    super("The iterator for a ContextHashMap entry-set" +
		  " created in " + expected +
		  " was used in " + current);
	}

    }

}
